package ru.yandex.autotests.directapi.reports.multigoals;

import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import com.yandex.direct.api.v5.general.AttributionModelEnum;
import com.yandex.direct.api.v5.reports.FieldEnum;
import org.junit.ClassRule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.darkside.Logins;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.model.api5.reports.ReportDefinitionMap;
import ru.yandex.autotests.directapi.model.api5.reports.ReportsData;
import ru.yandex.autotests.directapi.reports.ReportsFeatures;
import ru.yandex.autotests.directapi.reports.ReportsLogins;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;

/**
 * Created by lightelfik on 30.03.18.
 * https://st.yandex-team.ru/DIRECT-59898
 */
@Aqua.Test
@Description("Проверка корректной работы репортов для нескольких целей")
@Issue("https://st.yandex-team.ru/DIRECT-59898")
@Features(ReportsFeatures.OFFLINE_CUSTOM_REPORT)
public class MultiGoalsTest {

    private final static Long CAMPAIGN_ID = 6922642L;
    private final static List<Long> GOAL_IDS = Arrays.asList(2391553L, 2391559L);
    private final static String DATE_FROM = "2020-10-01";
    private final static String DATE_TO = "2020-11-19";

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(Logins.LOGIN_SUPER).clientLogin(ReportsLogins.CLIENT_RUB_WITH_STAT2);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Test
    public void checkReportWithOneGoal() {
        Long goalId = GOAL_IDS.get(0);
        ReportDefinitionMap reportDefinitionMap = new ReportDefinitionMap()
                .withDefaultCustomReportWithUniqueReportName()
                .withCustomDatesAndCampaignIds(DATE_FROM, DATE_TO, CAMPAIGN_ID)
                .withGoals(goalId)
                .withAttributionModels(AttributionModelEnum.LSC)
                .withFieldNames(FieldEnum.CAMPAIGN_ID, FieldEnum.CONVERSIONS);

        String conversionsField = String.format("%s_%d_%s", FieldEnum.CONVERSIONS.value(), goalId, AttributionModelEnum.LSC);
        List<String> expectedFields = Arrays.asList(FieldEnum.CAMPAIGN_ID.value(), conversionsField);

        ReportsData reportsData = api.userSteps.reportsSteps().callReportsXml(reportDefinitionMap);
        assertThat("вернулись правильные названия полей", reportsData.getColumnsHeadersList(), equalTo(expectedFields));
    }

    @Test
    public void checkReportWithTwoGoals() {
        ReportDefinitionMap reportDefinitionMap = new ReportDefinitionMap()
                .withDefaultCustomReportWithUniqueReportName()
                .withCustomDatesAndCampaignIds(DATE_FROM, DATE_TO, CAMPAIGN_ID)
                .withGoals(GOAL_IDS.toArray(new Long[0]))
                .withAttributionModels(AttributionModelEnum.LSC)
                .withFieldNames(FieldEnum.CONVERSIONS);

        List<String> conversionsFields = GOAL_IDS.stream().map(
                g -> String.format("%s_%d_%s", FieldEnum.CONVERSIONS.value(), g, AttributionModelEnum.LSC)
        ).collect(Collectors.toList());

        ReportsData reportsData = api.userSteps.reportsSteps().callReportsXml(reportDefinitionMap);
        assertThat("вернулись правильные названия полей", reportsData.getColumnsHeadersList(), equalTo(conversionsFields));
    }

    @Test
    public void checkReportWithTwoGoalsAndTwoModels() {
        List<AttributionModelEnum> MODELS = Arrays.asList(AttributionModelEnum.LSC, AttributionModelEnum.FC);

        ReportDefinitionMap reportDefinitionMap = new ReportDefinitionMap()
                .withDefaultCustomReportWithUniqueReportName()
                .withCustomDatesAndCampaignIds(DATE_FROM, DATE_TO, CAMPAIGN_ID)
                .withGoals(GOAL_IDS.toArray(new Long[0]))
                .withAttributionModels(MODELS.toArray(new AttributionModelEnum[0]))
                .withFieldNames(FieldEnum.CONVERSIONS);

        List<String> conversionsFields = MODELS.stream().map(
                m -> String.format("%s_%d_%s", FieldEnum.CONVERSIONS.value(), GOAL_IDS.get(0), m)
        ).collect(Collectors.toList());
        conversionsFields.addAll(MODELS.stream().map(
                m -> String.format("%s_%d_%s", FieldEnum.CONVERSIONS.value(), GOAL_IDS.get(1), m)
        ).collect(Collectors.toList()));

        ReportsData reportsData = api.userSteps.reportsSteps().callReportsXml(reportDefinitionMap);
        assertThat("вернулись правильные названия полей", reportsData.getColumnsHeadersList(), equalTo(conversionsFields));
    }

    @Test
    public void checkReportOnlyFilterByGoal() {
        ReportDefinitionMap reportDefinitionMap = new ReportDefinitionMap()
                .withDefaultCustomReportWithUniqueReportName()
                .withCustomDates(DATE_FROM, DATE_TO)
                .withGoals(GOAL_IDS.toArray(new Long[0]))
                .withAttributionModels(AttributionModelEnum.LC)
                .withFieldNames(FieldEnum.CONVERSIONS);

        List<String> conversionsFields = GOAL_IDS.stream().map(
                g -> String.format("%s_%d_%s", FieldEnum.CONVERSIONS.value(), g, AttributionModelEnum.LC)
        ).collect(Collectors.toList());

        ReportsData reportsData = api.userSteps.reportsSteps().callReportsXml(reportDefinitionMap);
        assertThat("вернулись правильные названия полей", reportsData.getColumnsHeadersList(), equalTo(conversionsFields));
    }

    @Test
    public void checkReportWithGoalAndWithoutFieldsDependentOnGoal() {
        ReportDefinitionMap reportDefinitionMap = new ReportDefinitionMap()
                .withDefaultCustomReportWithUniqueReportName()
                .withCustomDates(DATE_FROM, DATE_TO)
                .withGoals(GOAL_IDS.toArray(new Long[0]))
                .withAttributionModels(AttributionModelEnum.LC)
                .withFieldNames(FieldEnum.IMPRESSIONS);

        ReportsData reportsData = api.userSteps.reportsSteps().callReportsXml(reportDefinitionMap);
        assertThat("вернулись правильные названия полей",
                reportsData.getColumnsHeadersList().get(0), equalTo(FieldEnum.IMPRESSIONS.value())
        );
    }


}
