package ru.yandex.autotests.directapi.reports.orderby;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

import com.yandex.direct.api.v5.general.YesNoEnum;
import com.yandex.direct.api.v5.reports.DateRangeTypeEnum;
import com.yandex.direct.api.v5.reports.FieldEnum;
import com.yandex.direct.api.v5.reports.FormatEnum;
import com.yandex.direct.api.v5.reports.ReportTypeEnum;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.darkside.Logins;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.model.api5.reports.OrderByMap;
import ru.yandex.autotests.directapi.model.api5.reports.PageMap;
import ru.yandex.autotests.directapi.model.api5.reports.ReportDefinitionMap;
import ru.yandex.autotests.directapi.model.api5.reports.ReportsData;
import ru.yandex.autotests.directapi.model.api5.reports.ReportsLine;
import ru.yandex.autotests.directapi.model.api5.reports.SelectionCriteriaMap;
import ru.yandex.autotests.directapi.reports.ReportsFeatures;
import ru.yandex.autotests.directapi.reports.ReportsLogins;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.irt.testutils.matchers.OrderMatcher;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.not;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by onotole on 18.07.16.
 * https://st.yandex-team.ru/TESTIRT-9739
 */
@Aqua.Test
@Issue("https://st.yandex-team.ru/TESTIRT-9244")
@Description("проверка работы OrderBy, поля представленные числовыми значениями")
@Features(ReportsFeatures.ONLINE_CUSTOM_REPORT)
@RunWith(Parameterized.class)
public class OrderByDecimalFieldsTest {
    public static final String CLIENT = ReportsLogins.CLIENT_RUB_WITH_STAT4;
    private static final String DATE_FROM = "2019-12-23";
    private static final String DATE_TO = "2019-12-29";

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(Logins.LOGIN_SUPER).clientLogin(CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Parameterized.Parameter(0)
    public FieldEnum fieldEnum;

    @Parameterized.Parameters(name = "{0}")
    public static Collection<Object[]> data () {
        return Arrays.asList((Object[][]) new Object[][]{
                {FieldEnum.AD_GROUP_ID},
                {FieldEnum.AD_ID},
                {FieldEnum.AVG_CPC},
                {FieldEnum.AVG_CLICK_POSITION},
                {FieldEnum.COST_PER_CONVERSION },
                {FieldEnum.AVG_IMPRESSION_POSITION},
                {FieldEnum.AVG_PAGEVIEWS },
                {FieldEnum.BOUNCE_RATE},
                {FieldEnum.BOUNCES},
                {FieldEnum.CAMPAIGN_ID},
                {FieldEnum.CLICKS},
                {FieldEnum.CONVERSIONS},
                // TODO DIRECT-79630 раскомментировать после DIRECT-79630
                // {FieldEnum.CONVERSION_RATE},
                {FieldEnum.COST},
                {FieldEnum.CTR},
//                {FieldEnum.GOALS_ROI}, FIXME Есть неопределенные значения, которые нужно обрабатывать отдельно
                {FieldEnum.IMPRESSIONS},
                {FieldEnum.LOCATION_OF_PRESENCE_ID},
                {FieldEnum.PROFIT},
                {FieldEnum.REVENUE},
                {FieldEnum.TARGETING_LOCATION_ID}
        });
    }

    @Test
    public void checkOrderByField() {
        ReportDefinitionMap reportDefinitionMap = new ReportDefinitionMap()
                .withPage(new PageMap().withLimit(1000L))
                .withFieldNames(fieldEnum, FieldEnum.AGE)
                .withOrderBy(new OrderByMap()
                        .withField(fieldEnum))
                .withUniqueReportName()
                .withReportType(ReportTypeEnum.CUSTOM_REPORT)
                .withDateRangeType(DateRangeTypeEnum.CUSTOM_DATE)
                .withFormat(FormatEnum.TSV)
                .withIncludeVAT(YesNoEnum.YES)
                .withIncludeDiscount(YesNoEnum.NO)
                .withSelectionCriteria(new SelectionCriteriaMap()
                    .withDateTo(DATE_TO)
                    .withDateFrom(DATE_FROM)
                );

        ReportsData reportsData = api.userSteps.reportsSteps().callReportsXml(reportDefinitionMap);
        assumeThat("вернулся непустой отчет", reportsData.getReportsLines(), not(empty()));

        List<BigDecimal> valueList = reportsData.getReportsLines().stream().map((s) -> getValue(s, fieldEnum))
                .collect(Collectors.toList());
        assertThat("сортировка по полю " + fieldEnum.value() + " работает", valueList, OrderMatcher.isAscendingOrdered());
    }


    private BigDecimal getValue(ReportsLine line, FieldEnum fieldEnum) {
        BigDecimal result = null;
        switch(fieldEnum) {
            case AD_GROUP_ID:
                result = BigDecimal.valueOf(line.getAdGroupId());
                break;
            case AD_ID:
                result = BigDecimal.valueOf(line.getAdId());
                break;
            case AVG_CPC:
                result = line.getAvgCpc();
                break;
            case AVG_CLICK_POSITION:
                result = line.getAvgClickPosition();
                break;
            case COST_PER_CONVERSION :
                result = line.getCostPerConversion();
                break;
            case AVG_IMPRESSION_POSITION:
                result = line.getAvgImpressionPosition();
                break;
            case AVG_PAGEVIEWS :
                result = line.getAvgPageviews();
                break;
            case BOUNCE_RATE:
                result = line.getBounceRate();
                break;
            case BOUNCES:
                result = BigDecimal.valueOf(line.getBounces());
                break;
            case CAMPAIGN_ID:
                result = BigDecimal.valueOf(line.getCampaignId());
                break;
            case CLICKS:
                result = BigDecimal.valueOf(line.getClicks());
                break;
            case CONVERSIONS:
                result = BigDecimal.valueOf(line.getConversions());
                break;
            case CONVERSION_RATE:
                result = line.getConversionRate();
                break;
            case COST:
                result = line.getCost();
                break;
            case CTR:
                result = line.getCtr();
                break;
            case GOALS_ROI:
                result = line.getGoalsRoi();
                break;
            case IMPRESSIONS:
                result = BigDecimal.valueOf(line.getImpressions());
                break;
            case LOCATION_OF_PRESENCE_ID:
                result = BigDecimal.valueOf(line.getLocationOfPresenceId());
                break;
            case PROFIT:
                result = line.getProfit();
                break;
            case REVENUE:
                result = line.getRevenue();
                break;
            case TARGETING_LOCATION_ID:
                result = BigDecimal.valueOf(line.getTargetingLocationId());
                break;
        }
        return result;
    }

}
