package ru.yandex.autotests.directapi.reports.orderby;

import java.util.Arrays;
import java.util.Collection;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.base.MoreObjects;
import com.yandex.direct.api.v5.general.AgeRangeEnum;
import com.yandex.direct.api.v5.general.SortOrderEnum;
import com.yandex.direct.api.v5.general.YesNoEnum;
import com.yandex.direct.api.v5.reports.DateRangeTypeEnum;
import com.yandex.direct.api.v5.reports.FieldEnum;
import com.yandex.direct.api.v5.reports.FormatEnum;
import com.yandex.direct.api.v5.reports.ReportTypeEnum;
import org.apache.commons.lang.ArrayUtils;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.model.api5.reports.OrderByMap;
import ru.yandex.autotests.directapi.model.api5.reports.ReportDefinitionMap;
import ru.yandex.autotests.directapi.model.api5.reports.ReportsData;
import ru.yandex.autotests.directapi.model.api5.reports.ReportsLine;
import ru.yandex.autotests.directapi.model.api5.reports.SelectionCriteriaMap;
import ru.yandex.autotests.directapi.reports.ReportsFeatures;
import ru.yandex.autotests.directapi.reports.ReportsLogins;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.contains;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by onotole on 18.07.16.
 * https://st.yandex-team.ru/TESTIRT-9739
 */
@Aqua.Test
@Issue("https://st.yandex-team.ru/TESTIRT-9244")
@Description("проверка работы OrderBy, поля представленные фиксированным набором значений (enum)")
@Features(ReportsFeatures.ONLINE_CUSTOM_REPORT)
@RunWith(Parameterized.class)
public class OrderByEnumFieldsTest {
    private static final String CLIENT = ReportsLogins.CLIENT_WITH_STAT_WITH_ALL_TYPE_ADS;
    private static final String CLIENT_FOR_AD_FORMAT = ReportsLogins.CLIENT_WITH_STAT_WITH_ALL_AD_FORMAT;

    private static final String DEFAULT_DATE_FROM = "2022-04-09";
    private static final String DEFAULT_DATE_TO = "2022-04-11";

    private ReportDefinitionMap reportDefinitionMap;

    @Rule
    public ApiSteps api = new ApiSteps();

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Parameterized.Parameter(0)
    public FieldEnum fieldEnum;

    @Parameterized.Parameter(1)
    public Function<ReportsLine, String> fieldValueGetter;

    @Parameterized.Parameter(2)
    public String[] expectedOrder;

    @Parameterized.Parameter(3)
    public String client;

    @Parameterized.Parameter(4)
    public String dateFrom;

    @Parameterized.Parameter(5)
    public String dateTo;

    @Parameterized.Parameters(name = "{0}")
    public static Collection<Object[]> testData() {
        return Arrays.asList(new Object[][]{
                {FieldEnum.AGE, (Function<ReportsLine, String>) ReportsLine::getAge,
                        new String[]{AgeRangeEnum.AGE_0_17.value(), AgeRangeEnum.AGE_18_24.value(),
                                AgeRangeEnum.AGE_25_34.value(), AgeRangeEnum.AGE_35_44.value(),
                                AgeRangeEnum.AGE_45_54.value(), AgeRangeEnum.AGE_55.value(),
                                "UNKNOWN"}, CLIENT,
                        null, null},
                {FieldEnum.AD_NETWORK_TYPE, (Function<ReportsLine, String>) ReportsLine::getAdNetworkType,
                        new String[]{"AD_NETWORK", "SEARCH"}, CLIENT,
                        null, null},
                {FieldEnum.CARRIER_TYPE, (Function<ReportsLine, String>) ReportsLine::getCarrierType,
                        new String[]{"CELLULAR", "STATIONARY"}, CLIENT,
                        null, null},
                {FieldEnum.CLICK_TYPE, (Function<ReportsLine, String>) ReportsLine::getClickType,
                        new String[]{"SITELINK1", "SITELINK2", "SITELINK3", "SITELINK4",
                                "TITLE", "UNKNOWN"}, CLIENT,
                        null, null},
                {FieldEnum.CRITERIA_TYPE, (Function<ReportsLine, String>) ReportsLine::getCriteriaType,
                        new String[]{"AUTOTARGETING", "DYNAMIC_TEXT_AD_TARGET", "KEYWORD"}, CLIENT,
                        null, null},
                {FieldEnum.CRITERION_TYPE, (Function<ReportsLine, String>) ReportsLine::getCriterionType,
                        new String[]{"FEED_FILTER", "KEYWORD", "RETARGETING"}, CLIENT_FOR_AD_FORMAT,
                        null, null},
                {FieldEnum.DEVICE, (Function<ReportsLine, String>) ReportsLine::getDevice,
                        new String[]{"DESKTOP", "MOBILE", "TABLET"}, CLIENT,
                        null, null},
                {FieldEnum.GENDER, (Function<ReportsLine, String>) ReportsLine::getGender,
                        new String[]{"GENDER_FEMALE", "GENDER_MALE", "UNKNOWN"}, CLIENT,
                        null, null},
                {FieldEnum.AD_FORMAT, (Function<ReportsLine, String>) ReportsLine::getAdFormat,
                        new String[]{"IMAGE", "SMART_MULTIPLE", "SMART_SINGLE", "SMART_TILE", "TEXT", "VIDEO"}, CLIENT_FOR_AD_FORMAT,
                        null, null},
                {FieldEnum.MATCH_TYPE, (Function<ReportsLine, String>) ReportsLine::getMatchType,
                        new String[]{"KEYWORD","NONE", "SYNONYM"}, CLIENT_FOR_AD_FORMAT,
                        null, null},
                {FieldEnum.MOBILE_PLATFORM, (Function<ReportsLine, String>) ReportsLine::getMobilePlatform,
                        new String[]{"ANDROID", "IOS", "OTHER"}, CLIENT,
                        null, null},
                {FieldEnum.SLOT, (Function<ReportsLine, String>) ReportsLine::getSlot,
                        new String[]{"ALONE", "OTHER", "PREMIUMBLOCK", "PRODUCT_GALLERY", "SUGGEST"}, CLIENT,
                        null, null}
        });
    }

    @Before
    public void prepareReportRequest() {
        api.as(ReportsLogins.SUPER_LOGIN).clientLogin(client);
        reportDefinitionMap = new ReportDefinitionMap()
                .withDateRangeType(DateRangeTypeEnum.CUSTOM_DATE)
                .withSelectionCriteria(new SelectionCriteriaMap()
                        .withDateFrom(MoreObjects.firstNonNull(dateFrom, DEFAULT_DATE_FROM))
                        .withDateTo(MoreObjects.firstNonNull(dateTo, DEFAULT_DATE_TO)))
                .withFieldNames(fieldEnum)
                .withOrderBy(new OrderByMap()
                        .withField(fieldEnum))
                .withUniqueReportName()
                .withReportType(ReportTypeEnum.CUSTOM_REPORT)
                .withFormat(FormatEnum.TSV)
                .withIncludeVAT(YesNoEnum.YES)
                .withIncludeDiscount(YesNoEnum.NO);
    }

    @Test
    public void checkOrderByField() {
        reportDefinitionMap.withOrderBy(
                new OrderByMap()
                        .withField(fieldEnum)
                        .withSortOrder(SortOrderEnum.ASCENDING)
        );

        ReportsData reportsData = api.userSteps.reportsSteps().callReportsXml(reportDefinitionMap);

        assertThat("сортировка по полю " + fieldEnum.value() + " работает",
                reportsData.getReportsLines().stream()
                        .map(fieldValueGetter)
                        .collect(Collectors.toList()),
                contains(expectedOrder));
    }

    @Test
    public void checkOrderByFieldReverse() {
        reportDefinitionMap.withOrderBy(
                new OrderByMap()
                        .withField(fieldEnum)
                        .withSortOrder(SortOrderEnum.DESCENDING)
        );

        ReportsData reportsData = api.userSteps.reportsSteps().callReportsXml(reportDefinitionMap);

        String[] reverseOrder = expectedOrder.clone();
        ArrayUtils.reverse(reverseOrder);

        assertThat("сортировка по полю " + fieldEnum.value() + " в порядке убывания работает",
                reportsData.getReportsLines().stream()
                        .map(fieldValueGetter)
                        .collect(Collectors.toList()),
                contains(reverseOrder));
    }

}
