package ru.yandex.autotests.directapi.reports.orderby;

import com.yandex.direct.api.v5.general.SortOrderEnum;
import com.yandex.direct.api.v5.general.YesNoEnum;
import com.yandex.direct.api.v5.reports.FieldEnum;
import com.yandex.direct.api.v5.reports.FormatEnum;
import com.yandex.direct.api.v5.reports.ReportTypeEnum;
import org.junit.ClassRule;
import org.junit.Test;
import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.darkside.Logins;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.model.api5.reports.*;
import ru.yandex.autotests.directapi.reports.ReportsFeatures;
import ru.yandex.autotests.directapi.reports.ReportsLogins;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.*;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by onotole on 19.07.16.
 * https://st.yandex-team.ru/TESTIRT-9739
 */
@Aqua.Test
@Issue("https://st.yandex-team.ru/TESTIRT-9244")
@Description("проверка работы OrderBy, сортировка по нескольки полям")
@Features(ReportsFeatures.ONLINE_CUSTOM_REPORT)
public class OrderByManyFieldsTest {
    public static final String CLIENT = ReportsLogins.STAT_CLIENT_FOR_SHARDING;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(Logins.LOGIN_SUPER).clientLogin(CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Test
    public void twoFieldOrderByTest() {
        FieldEnum fieldEnum1 = FieldEnum.TARGETING_LOCATION_ID;
        FieldEnum fieldEnum2 = FieldEnum.CLICKS;
        ReportsLine previousLine;
        ReportsLine currentLine;
        Boolean isOrderCorrect = true;

        ReportDefinitionMap reportDefinitionMap = new ReportDefinitionMap()
                .withDefaultLimitedPeriod()
                .withFieldNames(fieldEnum1, fieldEnum2, FieldEnum.CAMPAIGN_ID)
                .withOrderBy(new OrderByMap()
                        .withSortOrder(SortOrderEnum.DESCENDING)
                        .withField(fieldEnum1),
                            new OrderByMap()
                        .withField(fieldEnum2))
                .withUniqueReportName()
                .withReportType(ReportTypeEnum.CUSTOM_REPORT)
                .withFormat(FormatEnum.TSV)
                .withIncludeVAT(YesNoEnum.YES)
                .withIncludeDiscount(YesNoEnum.NO);

        ReportsData reportsData = api.userSteps.reportsSteps().callReportsXml(reportDefinitionMap);
        assumeThat("вернулся не пустой репорт", reportsData.getReportsLines().size(), greaterThan(0));

        previousLine = reportsData.getReportsLines().get(0);
        for (int i = 1; i < reportsData.getReportsLines().size(); i++) {
            currentLine = reportsData.getReportsLines().get(i);
            if (currentLine.getTargetingLocationId() > previousLine.getTargetingLocationId()) {
                isOrderCorrect = false;
                break;
            }
            if ((currentLine.getClicks() < previousLine.getClicks()) && (currentLine.getTargetingLocationId() == previousLine.getTargetingLocationId())) {
                isOrderCorrect = false;
                break;
            }
            previousLine = currentLine;
        }
        assertThat("сортировка по двум полям в разных направления работает", isOrderCorrect, equalTo(true));
    }

    @Test
    public void someDifferentTypeFieldOrderByTest() {
        FieldEnum fieldEnumEnum = FieldEnum.AGE;
        FieldEnum fieldEnumInteger = FieldEnum.CLICKS;
        FieldEnum fieldEnumString = FieldEnum.CAMPAIGN_NAME;
        ReportsLine previousLine;
        ReportsLine currentLine;
        Boolean isOrderCorrect = true;

        ReportDefinitionMap reportDefinitionMap = new ReportDefinitionMap()
                .withDefaultLimitedPeriod()
                .withFieldNames(fieldEnumEnum, fieldEnumInteger, fieldEnumString)
                .withOrderBy(new OrderByMap()
                                .withSortOrder(SortOrderEnum.DESCENDING)
                                .withField(fieldEnumEnum),
                        new OrderByMap()
                                .withField(fieldEnumInteger),
                        new OrderByMap()
                                .withField(fieldEnumString))
                .withUniqueReportName()
                .withReportType(ReportTypeEnum.CUSTOM_REPORT)
                .withFormat(FormatEnum.TSV)
                .withIncludeVAT(YesNoEnum.YES)
                .withIncludeDiscount(YesNoEnum.NO);

        ReportsData reportsData = api.userSteps.reportsSteps().callReportsXml(reportDefinitionMap);
        assumeThat("вернулся не пустой репорт", reportsData.getReportsLines().size(), greaterThan(0));

        previousLine = reportsData.getReportsLines().get(0);
        for (int i = 1; i < reportsData.getReportsLines().size(); i++) {
            currentLine = reportsData.getReportsLines().get(i);
            // check first field order, descending order
            if (currentLine.getAge().compareTo(previousLine.getAge()) > 0) {
                isOrderCorrect = false;
                break;
            } else if (currentLine.getAge().equals(previousLine.getAge())){
                // check second field order, ascending order
                if (currentLine.getClicks() < previousLine.getClicks() ) {
                    isOrderCorrect = false;
                    break;
                } else if (currentLine.getClicks().equals(previousLine.getClicks())) {
                    // check first field order, ascending order
                    if ( currentLine.getCampaignName().compareToIgnoreCase(previousLine.getCampaignName()) < 0) {
                        isOrderCorrect = false;
                        break;
                    }
                }
            }
            previousLine = currentLine;
        }
        assertThat("сортировка по нескольки полям разного типа работает", isOrderCorrect, equalTo(true));
    }

    @Test
    public void manyFieldsOrderByTest() {
        // fields with problem: "ru.yandex.autotests.directapi.exceptions.DirectAPIException: В строке отчета вернулось количество полей, несовпадающее с размером FieldNames из запроса. Строка номер 28"
        // FieldEnum.CRITERIA_ID

        // These fields couldn't be in one request with DATE
        // FieldEnum.MONTH, FieldEnum.YEAR, FieldEnum.WEEK, FieldEnum.QUARTER,

        FieldEnum[] fieldEnums = {
                FieldEnum.AD_GROUP_ID, FieldEnum.AD_GROUP_NAME, FieldEnum.AD_ID, FieldEnum.AD_NETWORK_TYPE,
                FieldEnum.AGE, FieldEnum.AVG_CLICK_POSITION, FieldEnum.AVG_CPC, FieldEnum.COST_PER_CONVERSION ,
                FieldEnum.AVG_IMPRESSION_POSITION, FieldEnum.AVG_PAGEVIEWS , FieldEnum.CAMPAIGN_ID, FieldEnum.CAMPAIGN_TYPE,
                FieldEnum.CAMPAIGN_NAME, FieldEnum.CARRIER_TYPE, FieldEnum.CLICKS,
                FieldEnum.CONVERSIONS, FieldEnum.CONVERSION_RATE, FieldEnum.COST, FieldEnum.CRITERIA_TYPE,
                FieldEnum.CTR, FieldEnum.DATE,
                FieldEnum.DEVICE, FieldEnum.EXTERNAL_NETWORK_NAME, FieldEnum.GENDER, FieldEnum.GOALS_ROI,
                FieldEnum.AD_FORMAT, FieldEnum.IMPRESSIONS, FieldEnum.LOCATION_OF_PRESENCE_NAME, FieldEnum.MOBILE_PLATFORM,
                FieldEnum.REVENUE, FieldEnum.SLOT, FieldEnum.TARGETING_LOCATION_ID, FieldEnum.TARGETING_LOCATION_NAME
        };
        OrderByMap[] ordersByMap = new OrderByMap[fieldEnums.length];

        for (int i = 0; i < fieldEnums.length; i++) {
            ordersByMap[i] = new OrderByMap().withField(fieldEnums[i]);
        }

        ReportDefinitionMap reportDefinitionMap = new ReportDefinitionMap()
                .withDefaultLimitedPeriod()
                .withFieldNames(fieldEnums)
                .withOrderBy(ordersByMap)
                .withUniqueReportName()
                .withReportType(ReportTypeEnum.CUSTOM_REPORT)
                .withPage(new PageMap().withLimit(1000L))
                .withFormat(FormatEnum.TSV)
                .withIncludeVAT(YesNoEnum.YES)
                .withIncludeDiscount(YesNoEnum.NO);

        ReportsData reportsData = api.userSteps.reportsSteps().callReportsXml(reportDefinitionMap);

        assertThat("сортировка по многим полям не упала", reportsData.getReportsLines(), not(empty()));

    }

    @Test
    public void orderByOneGoodFieldOneBadNegativeTest() {
        FieldEnum fieldEnumNegative = FieldEnum.CRITERIA; // по этому полю сортировать нельзя
        FieldEnum fieldEnumPositive = FieldEnum.CAMPAIGN_ID;

        ReportDefinitionMap reportDefinitionMap = new ReportDefinitionMap()
                .withDefaultLimitedPeriod()
                .withFieldNames(fieldEnumNegative, fieldEnumPositive)
                .withOrderBy(new OrderByMap()
                        .withField(fieldEnumNegative)
                        ,new OrderByMap()
                        .withField(fieldEnumPositive)
                )
                .withUniqueReportName()
                .withReportType(ReportTypeEnum.CUSTOM_REPORT)
                .withFormat(FormatEnum.TSV)
                .withIncludeVAT(YesNoEnum.YES)
                .withIncludeDiscount(YesNoEnum.NO);

        api.userSteps.reportsSteps().expectXmlErrorOnReports(reportDefinitionMap,
                new Api5Error(4000, Api5ErrorDetails.REPORTS_UNSUPPORTED_FIELD, "OrderBy.Field", fieldEnumNegative.value()));
    }





}
