package ru.yandex.autotests.directapi.reports.orderby;

import com.yandex.direct.api.v5.general.YesNoEnum;
import com.yandex.direct.api.v5.reports.*;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.darkside.Logins;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.model.api5.reports.*;
import ru.yandex.autotests.directapi.reports.ReportsFeatures;
import ru.yandex.autotests.directapi.reports.ReportsLogins;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;

import static org.hamcrest.Matchers.contains;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by onotole on 18.07.16.
 * https://st.yandex-team.ru/TESTIRT-9739
 */
@Aqua.Test
@Issue("https://st.yandex-team.ru/TESTIRT-9244")
@Description("проверка работы OrderBy, поля представленные строковыми значениями")
@Features(ReportsFeatures.ONLINE_CUSTOM_REPORT)
@RunWith(Parameterized.class)
public class OrderByStringFieldsTest {
    public static final String CLIENT = ReportsLogins.STAT_CLIENT_FOR_SHARDING;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(Logins.LOGIN_SUPER).clientLogin(CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Parameterized.Parameter(0)
    public FieldEnum fieldEnum;

    @Parameterized.Parameters(name = "{0}")
    public static Collection<Object[]> data () {
        return Arrays.asList(new Object[][]{
                {FieldEnum.AD_GROUP_NAME},
                {FieldEnum.CAMPAIGN_NAME},
                {FieldEnum.CAMPAIGN_TYPE},
                {FieldEnum.DATE},
                {FieldEnum.EXTERNAL_NETWORK_NAME},
//                {FieldEnum.LOCATION_OF_PRESENCE_NAME}, // DIRECT-60066
                {FieldEnum.MONTH},
//                {FieldEnum.PLACEMENT}, // DIRECT-60065
                {FieldEnum.QUARTER},
//                {FieldEnum.TARGETING_LOCATION_NAME}, // DIRECT-60066
                {FieldEnum.WEEK},
                {FieldEnum.YEAR}
        });
    }

    @Test
    public void checkOrderByField() {
        ReportDefinitionMap reportDefinitionMap = new ReportDefinitionMap()
                .withDefaultLimitedPeriod()
                .withFieldNames(fieldEnum)
                .withOrderBy(new OrderByMap()
                        .withField(fieldEnum))
                .withUniqueReportName()
                .withReportType(ReportTypeEnum.CUSTOM_REPORT)
                .withFormat(FormatEnum.TSV)
                .withIncludeVAT(YesNoEnum.YES)
                .withIncludeDiscount(YesNoEnum.NO);

        ReportsData reportsData = api.userSteps.reportsSteps().callReportsXml(reportDefinitionMap);

        Comparator<String> comparator;
        if (fieldEnum == FieldEnum.PLACEMENT) {
            comparator = (v1, v2) -> {
                if (v1.equals(v2)) {
                    return 0;
                }

                // Яндекс идёт первым
                if (v1.equals("Яндекс")) {
                    return -1;
                }

                if (v2.equals("Яндекс")) {
                    return 1;
                }

                return v1.compareTo(v2);
            };
        } else if (fieldEnum == FieldEnum.AD_GROUP_NAME) {
            // потому что на ТС может быть старая база, в которых каких-то
            // названий групп нет; такие группы возвращаются с названиями "--",
            // которые превращаются в null и идут в конце
            comparator = (v1, v2) -> {
                if (v1 == null) {
                    return 1;
                }

                if (v2 == null) {
                    return -1;
                }

                return v1.compareTo(v2);
            };
        } else {
            comparator = String.CASE_INSENSITIVE_ORDER;
        }

        List<String> values = reportsData.getReportsLines().stream()
                .map(line -> getValue(line, fieldEnum))
                .collect(Collectors.toList());

        List<String> sortedValues = values.stream()
                .sorted(comparator)
                .collect(Collectors.toList());

        assertThat("сортировка по полю " + fieldEnum.value() + " работает",
                values,
                contains(sortedValues.toArray(new String[0])));
    }

    private String getValue(ReportsLine line, FieldEnum fieldEnum) {
        String result = null;
        switch(fieldEnum) {
            case AD_GROUP_NAME:
                result = line.getAdGroupName();
                break;
            case CAMPAIGN_NAME:
                result = line.getCampaignName();
                break;
            case CAMPAIGN_TYPE:
                result = line.getCampaignType();
                break;
            case DATE:
                result = line.getDate();
                break;
            case EXTERNAL_NETWORK_NAME:
                result = line.getExternalNetworkName();
                break;
            case LOCATION_OF_PRESENCE_NAME:
                result = line.getLocationOfPresenceName();
                break;
            case MONTH:
                result = line.getMonth();
                break;
            case PLACEMENT:
                result = line.getPlacement();
                break;
            case QUARTER:
                result = line.getQuarter();
                break;
            case TARGETING_LOCATION_NAME:
                result = line.getTargetingLocationName();
                break;
            case WEEK:
                result = line.getWeek();
                break;
            case YEAR:
                result = line.getYear();
                break;
        }
        return result;
    }

}
