package ru.yandex.autotests.directapi.reports.restrictions;

import com.yandex.direct.api.v5.reports.*;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;
import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.ApiFeatures;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.darkside.Logins;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.model.api5.reports.*;
import ru.yandex.autotests.directapi.reports.ReportsFeatures;
import ru.yandex.autotests.directapi.reports.ReportsLogins;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import java.time.LocalDateTime;

/**
 * Created by onotole on 25.11.16.
 */
@Aqua.Test
@Description("Проверка работы ограничений клиентских запросов на создание отчетов")
@Issue("https://st.yandex-team.ru/TESTIRT-10462")
@Features({ApiFeatures.NOT_REGRESSION_YET, ApiFeatures.ONLY_FOR_TC, ReportsFeatures.OFFLINE_CUSTOM_REPORT})
public class RestrictionsOnReportsTest
{
    private static final String CLIENT_FOR_SIMULTANEOUS_REQS = ReportsLogins.CLIENT_EUR_WITH_STAT;
    private static final String CLIENT_FOR_REPORTS_QUEUE = ReportsLogins.CLIENT_FOR_RESTRICTIONS2;
    private static final String CLIENT_FOR_LIMIT_IN_PERIOD = ReportsLogins.CLIENT_FOR_RESTRICTIONS;

    private static final int OFFLINE_REPORTS_QUEUE_SIZE = 5;
    private static final int REQUESTS_LIMIT_IN_PERIOD_OF_TIME = 20;
    private static final int PERIOD_OF_TIME = 10;
    private static final int SIMULTANEOUS_REQUESTS_LIMIT = 10;
    private final int INCORRECT_CAMPAIGN_ID = 12345;


    @ClassRule
    public static ApiSteps api = new ApiSteps().as(Logins.LOGIN_SUPER, CLIENT_FOR_SIMULTANEOUS_REQS);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @BeforeClass
    public static void initTest() {
        api.userSteps.clientFakeSteps().setApiReportsLimit(CLIENT_FOR_REPORTS_QUEUE, OFFLINE_REPORTS_QUEUE_SIZE);
    }

    @Test
    public void simultaneousOfflineMultithreadReports() {
        api.as(ReportsLogins.SUPER_LOGIN, CLIENT_FOR_REPORTS_QUEUE);
        for (int i = 0; i < OFFLINE_REPORTS_QUEUE_SIZE; i++) {
            ReportDefinitionMap reportDefinitionMap = getLongTimeReportMap();
            new Thread(() -> api.userSteps.reportsSteps().callOfflineReportsXmlDoNotWaitForReady(reportDefinitionMap)).start();
        }

        try {
            Thread.sleep(100);
        } catch (InterruptedException e) {
            e.printStackTrace();
        }

        api.userSteps.reportsSteps().expectXmlErrorOnOfflineReports(getShortTimeReportMap(),
                new Api5Error(9000, Api5ErrorDetails.REPORTS_OFFLINE_QUEUE_EXCEED, OFFLINE_REPORTS_QUEUE_SIZE));
    }

    @Test
    public void manyRequestsInPeriodOfTime() {
        api.as(ReportsLogins.SUPER_LOGIN, CLIENT_FOR_LIMIT_IN_PERIOD);
        LocalDateTime currentDateTime = LocalDateTime.now();
        int currentSec = currentDateTime.getSecond() % 10;
        try {
            Thread.sleep((10 - currentSec) * 1000);
        } catch (InterruptedException e) {
            e.printStackTrace();
        }

        for (int i = 0; i < REQUESTS_LIMIT_IN_PERIOD_OF_TIME; i++) {
            Thread t = new Thread(() -> api.userSteps.reportsSteps().callReportsXml(getShortTimeReportMap()));
            t.start();
            try {
                Thread.sleep(200);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }

        api.userSteps.reportsSteps().expectXmlErrorOnReports(getShortTimeReportMap(),
                new Api5Error(56, Api5ErrorDetails.LIMIT_OF_METHOD_REQUESTS, REQUESTS_LIMIT_IN_PERIOD_OF_TIME,
                        PERIOD_OF_TIME));
    }

    @Test
    public void simultaneousRequestsLimit() {
        // лимит на количество одновременно обрабатываемых запросов можно задать поклиентно в базе
        // Ожидается, что для используемого в тесте клиента задано ограничение в 10 запросов
        // direct-sql ts:ppc:5 " update api_special_user_options set value = 10 where keyname = 'concurrent_calls' and clientid = 4861875";
        api.as(ReportsLogins.SUPER_LOGIN, CLIENT_FOR_SIMULTANEOUS_REQS);
        Thread[] threads = new Thread[SIMULTANEOUS_REQUESTS_LIMIT];

        for (int i = 0; i < SIMULTANEOUS_REQUESTS_LIMIT; i++) {
            threads[i] = new Thread(() -> api.userSteps.reportsSteps().callReportsXml(getLongTimeReportMap()));
        }

        for (int i = 0; i < SIMULTANEOUS_REQUESTS_LIMIT; i++) {
            threads[i].start();
        }
        try {
            Thread.sleep(1000);
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
        api.userSteps.reportsSteps() .expectXmlErrorOnReports(getShortTimeReportMap(),
                new Api5Error(506, Api5ErrorDetails.EXCEED_LIMIT_CONNECTIONS));
    }

    private ReportDefinitionMap getLongTimeReportMap() {
        return new ReportDefinitionMap()
                .withDefaultCustomReportWithUniqueReportName()
                .withFieldNames(FieldEnum.AGE, FieldEnum.AD_NETWORK_TYPE, FieldEnum.MOBILE_PLATFORM,
                        FieldEnum.GENDER,
                        FieldEnum.WEEK, FieldEnum.CLICK_TYPE, FieldEnum.CLICKS)
                .withPage(new PageMap().withLimit(1L))
                .withOrderBy(new OrderByMap()
                        .withField(FieldEnum.CLICKS))
                .withDateRangeType(DateRangeTypeEnum.ALL_TIME)
                .withSelectionCriteria(new SelectionCriteriaMap());
    }

    private ReportDefinitionMap getShortTimeReportMap() {
        return new ReportDefinitionMap()
                .withDefaultCustomReportWithUniqueReportName()
                .withSelectionCriteria(new SelectionCriteriaMap()
                        .withFilter(new FilterItemMap()
                                .withField(FieldEnum.CAMPAIGN_ID)
                                .withOperator(FilterOperatorEnum.EQUALS)
                                .withValues(INCORRECT_CAMPAIGN_ID)));
    }
}
