package ru.yandex.autotests.directapi.reports.searchquery;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.yandex.direct.api.v5.general.SortOrderEnum;
import com.yandex.direct.api.v5.reports.DateRangeTypeEnum;
import com.yandex.direct.api.v5.reports.FieldEnum;
import com.yandex.direct.api.v5.reports.FilterOperatorEnum;
import org.hamcrest.Matcher;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.darkside.Logins;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.model.api5.reports.FilterItemMap;
import ru.yandex.autotests.directapi.model.api5.reports.OrderByMap;
import ru.yandex.autotests.directapi.model.api5.reports.PageMap;
import ru.yandex.autotests.directapi.model.api5.reports.ReportDefinitionMap;
import ru.yandex.autotests.directapi.model.api5.reports.ReportsData;
import ru.yandex.autotests.directapi.model.api5.reports.ReportsLine;
import ru.yandex.autotests.directapi.model.api5.reports.SelectionCriteriaMap;
import ru.yandex.autotests.directapi.reports.ReportsFeatures;
import ru.yandex.autotests.directapi.reports.ReportsLogins;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.irt.testutils.matchers.OrderMatcher;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.not;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by onotole on 08.11.16.
 * https://st.yandex-team.ru/TESTIRT-10416
 */
@Aqua.Test
@Description("тесты на сортировку по числовым полям")
@Issue("https://st.yandex-team.ru/TESTIRT-9244")
@Features(ReportsFeatures.SEARCH_QUERY_PERFORMANCE_REPORT)
@RunWith(Parameterized.class)
public class OrderByNumericTest {
    /**
     * Если дата уехала за границы возможности отчёта нужно поменять дату на как можно более свежую
     * Если за нужные даты нет статистики нужно искать нового клиента.
     * 1. Для поиска можно воспользоваться таким YQL:
     * https://yql.yandex-team.ru/Operations/YEIsgwuEIxfY-KkPhO-LuH665V45N8hzX1Zr6r8FVLo=
     * --------------
     * use hahn;
     *
     * -- Кампании, которые мы ищем должны быть на ТС, devtest и dev7
     * -- смотрим чтобы время создания было примерно год назад,
     * -- ожидаем, что базы переналивают хотя бы раз в год
     * $create_time = '2020-03-05 03:00:00';
     *
     * SELECT u.login, c.cid, osd.Clicks, osd.UpdateTime
     * FROM `//home/yabs/stat/OrderStatDay` as osd
     * JOIN `home/direct/db/campaigns` as c on c.OrderID = osd.OrderID
     * JOIN `home/direct/db/users` as u on c.uid = u.uid
     * WHERE osd.Clicks > 1000
     * AND c.create_time < $create_time
     * ORDER BY UpdateTime DESC
     * LIMIT 10
     * --------------
     * 2. Проверяем, что найденная кампания есть во всех тестовых базах
     * direct-sql {dt,dev7,ts}:ppc:all 'select cid from campaigns where cid = <cid>'
     */
    private static final String CLIENT = ReportsLogins.CLIENT_FOR_ORDER_BY_NUMERIC;
    private static final String DATE_FROM = "2022-07-29";
    private static final String DATE_TO = "2022-07-29";
    private static final Long CAMPAIGN_ID = 45195153L;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(Logins.LOGIN_SUPER).clientLogin(CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Parameterized.Parameter
    public FieldEnum orderByField;

    @Parameterized.Parameter(1)
    public Function<ReportsLine, Comparable> valueGetter;

    @Parameterized.Parameter(2)
    public SortOrderEnum sortOrderEnum;

    @Parameterized.Parameters(name = "{0} {2}")
    public static Collection<Object[]> testData() {
        Collection<Object[]> result = new ArrayList<>();
        Object[][] data = new Object[][]{
                {FieldEnum.AVG_CLICK_POSITION, (Function<ReportsLine, BigDecimal>) ReportsLine::getAvgClickPosition},
                {FieldEnum.AVG_IMPRESSION_POSITION,
                        (Function<ReportsLine, BigDecimal>) ReportsLine::getAvgImpressionPosition},
                {FieldEnum.AVG_CPC, (Function<ReportsLine, BigDecimal>) ReportsLine::getAvgCpc},
                {FieldEnum.COST_PER_CONVERSION, (Function<ReportsLine, BigDecimal>) ReportsLine::getCostPerConversion},
                {FieldEnum.AVG_PAGEVIEWS, (Function<ReportsLine, BigDecimal>) ReportsLine::getAvgPageviews},
                {FieldEnum.BOUNCE_RATE, (Function<ReportsLine, BigDecimal>) ReportsLine::getBounceRate},
                {FieldEnum.PROFIT, (Function<ReportsLine, BigDecimal>) ReportsLine::getProfit},
                // отключаем из-за ошибки в ядре: удаляем значение конверсий при 0 кликов
                //{FieldEnum.CONVERSIONS, (Function<ReportsLine, Integer>) ReportsLine::getConversions},
                // TODO DIRECT-79630 раскомментировать после DIRECT-79630
                // {FieldEnum.CONVERSION_RATE, (Function<ReportsLine, BigDecimal>) ReportsLine::getConversionRate},
                // шумит при скользящем окне
                //{FieldEnum.GOALS_ROI, (Function<ReportsLine, BigDecimal>) ReportsLine::getGoalsRoi}

        };

        for (Object[] line : data) {
            result.add(
                    new Object[]{line[0], line[1], SortOrderEnum.ASCENDING}
            );
            result.add(
                    new Object[]{line[0], line[1], SortOrderEnum.DESCENDING}
            );
        }
        return result;
    }

    @Test
    public void orderByReport() {
        ReportDefinitionMap reportDefinitionMap = new ReportDefinitionMap()
                .withDefaultSearchQueryReportWithUniqueReportName()
                .withFieldNames(FieldEnum.DATE, orderByField)
                .withPage(new PageMap().withLimit(100L))
                .withDateRangeType(DateRangeTypeEnum.CUSTOM_DATE)
                .withOrderBy(new OrderByMap()
                        .withField(orderByField)
                        .withSortOrder(sortOrderEnum))
                .withSelectionCriteria(
                        new SelectionCriteriaMap()
                                .withDateFrom(DATE_FROM)
                                .withDateTo(DATE_TO)
                                .withFilter(
                                        new FilterItemMap()
                                                .withField(orderByField)
                                                .withOperator(FilterOperatorEnum.LESS_THAN)
                                                .withValues(100),
                                        new FilterItemMap()
                                                .withField(FieldEnum.CAMPAIGN_ID)
                                                .withOperator(FilterOperatorEnum.IN)
                                                .withValues(CAMPAIGN_ID))
                );
        ReportsData report = api.userSteps.reportsSteps().callReportsXml(reportDefinitionMap);
        assumeThat("вернулся непустой отчет", report.getReportsLines(), not(empty()));

        Matcher expectedOrder = sortOrderEnum == SortOrderEnum.ASCENDING ? OrderMatcher.isAscendingOrdered()
                : OrderMatcher.isDescendingOrdered();
        List<Comparable> queryList = report.getReportsLines().stream().map(valueGetter).filter(Objects::nonNull).collect(Collectors.toList());
        assertThat("Отчет вернулся с заданной сортировкой", queryList, expectedOrder);
    }
}
