package ru.yandex.autotests.directapi.reports.searchquery;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.function.Function;

import com.yandex.direct.api.v5.general.SortOrderEnum;
import com.yandex.direct.api.v5.reports.FieldEnum;
import org.apache.commons.lang3.math.NumberUtils;
import org.hamcrest.Matcher;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.darkside.Logins;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.model.api5.reports.OrderByMap;
import ru.yandex.autotests.directapi.model.api5.reports.PageMap;
import ru.yandex.autotests.directapi.model.api5.reports.ReportDefinitionMap;
import ru.yandex.autotests.directapi.model.api5.reports.ReportsData;
import ru.yandex.autotests.directapi.model.api5.reports.ReportsLine;
import ru.yandex.autotests.directapi.reports.ReportsFeatures;
import ru.yandex.autotests.directapi.reports.ReportsLogins;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.irt.testutils.matchers.OrderMatcher;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static java.util.stream.Collectors.toList;
import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.not;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by onotole on 08.11.16.
 * https://st.yandex-team.ru/TESTIRT-10416
 */
@Aqua.Test
@Description("базовые тесты на сортировку")
@Issue("https://st.yandex-team.ru/TESTIRT-9244")
@Features(ReportsFeatures.SEARCH_QUERY_PERFORMANCE_REPORT)
@RunWith(Parameterized.class)
public class OrderByPositiveTest {
    private final static String DATE_FROM;
    private final static String DATE_TO;

    static {
        DATE_FROM = "2022-02-08";
        DATE_TO = "2022-02-10";
    }

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(Logins.LOGIN_SUPER)
            .clientLogin(ReportsLogins.SEARCH_QUERY_PERFORMANCE_REPORT_CLIENT_FOR_ORDER_BY);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Parameterized.Parameter(0)
    public FieldEnum orderByField;

    @Parameterized.Parameter(1)
    public Function<ReportsLine, Comparable> valueGetter;

    @Parameterized.Parameter(2)
    public SortOrderEnum sortOrderEnum;

    @Parameterized.Parameters(name = "{0} {2}")
    public static Collection<Object[]> testData() {
        Collection<Object[]> result = new ArrayList<>();
        Object[][] data = new Object[][]{
                {FieldEnum.AD_GROUP_ID, (Function<ReportsLine, Long>) ReportsLine::getAdGroupId},
                {FieldEnum.AD_GROUP_NAME, (Function<ReportsLine, String>) ReportsLine::getAdGroupName},
                {FieldEnum.AD_ID, (Function<ReportsLine, Long>) ReportsLine::getAdId},
                {FieldEnum.BOUNCES, (Function<ReportsLine, Integer>) ReportsLine::getBounces},
                {FieldEnum.CAMPAIGN_ID, (Function<ReportsLine, Long>) ReportsLine::getCampaignId},
                {FieldEnum.CAMPAIGN_NAME, (Function<ReportsLine, String>) ReportsLine::getCampaignName},
                {FieldEnum.CAMPAIGN_TYPE, (Function<ReportsLine, String>) ReportsLine::getCampaignType},
                {FieldEnum.CLICKS, (Function<ReportsLine, Integer>) ReportsLine::getClicks},
                {FieldEnum.COST, (Function<ReportsLine, BigDecimal>) ReportsLine::getCost},
//                {FieldEnum.CRITERIA_ID, (Function<ReportsLine, Long>) ReportsLine::getCriteriaId}, FIXME
                {FieldEnum.CRITERIA_TYPE, (Function<ReportsLine, String>) ReportsLine::getCriteriaType},
                {FieldEnum.CRITERION_TYPE, (Function<ReportsLine, String>) ReportsLine::getCriterionType},
                {FieldEnum.MATCHED_KEYWORD, (Function<ReportsLine, String>) ReportsLine::getMatchedKeyword},
                {FieldEnum.CTR, (Function<ReportsLine, BigDecimal>) ReportsLine::getCtr},
                {FieldEnum.DATE, (Function<ReportsLine, String>) ReportsLine::getDate},
                {FieldEnum.IMPRESSIONS, (Function<ReportsLine, Integer>) ReportsLine::getImpressions},
                {FieldEnum.MONTH, (Function<ReportsLine, String>) ReportsLine::getMonth},
                {FieldEnum.QUARTER, (Function<ReportsLine, String>) ReportsLine::getQuarter},
//                {FieldEnum.QUERY, (Function<ReportsLine, String>) ReportsLine::getQuery}, DIRECT-68806
//                {FieldEnum.PLACEMENT, (Function<ReportsLine, String>) ReportsLine::getPlacement},
//                  надо учитывать, что площадка Яндекс всегда в начале ASCENDING сортировки
                {FieldEnum.REVENUE, (Function<ReportsLine, BigDecimal>) ReportsLine::getRevenue},
                {FieldEnum.WEEK, (Function<ReportsLine, String>) ReportsLine::getWeek},
                {FieldEnum.YEAR, (Function<ReportsLine, String>) ReportsLine::getYear},
                {FieldEnum.TARGETING_CATEGORY, (Function<ReportsLine, String>) ReportsLine::getTargetingCategory},
                {FieldEnum.INCOME_GRADE, (Function<ReportsLine, String>) ReportsLine::getIncomeGrade},
        };

        for (Object[] line : data) {
            result.add(
                    new Object[]{line[0], line[1], SortOrderEnum.ASCENDING}
            );
            result.add(
                    new Object[]{line[0], line[1], SortOrderEnum.DESCENDING}
            );
        }
        return result;
    }

    @Test
    public void orderByReport() {
        ReportDefinitionMap reportDefinitionMap = new ReportDefinitionMap()
                .withDefaultSearchQueryReportWithUniqueReportName()
                .withFieldNames(orderByField)
                .withPage(new PageMap().withLimit(20L))
                .withCustomDatesAndCampaignIds(
                        DATE_FROM, DATE_TO,
                        14282678L, 13782920L
                )
                .withOrderBy(new OrderByMap()
                        .withField(orderByField)
                        .withSortOrder(sortOrderEnum));
        ReportsData report = api.userSteps.reportsSteps().callReportsXml(reportDefinitionMap);
        assumeThat("вернулся непустой отчет", report.getReportsLines(), not(empty()));

        List<Comparable> queryList = report.getReportsLines().stream()
                .map(valueGetter)
                .filter(Objects::nonNull)
                .collect(toList());

        // OrderMatcher.isAscendingOrdered тут некорректно работает со строками которые начинаются с пробела,
        // поэтому было принято решение заменить на след. матчер:
        Matcher expectedOrder = sortOrderEnum == SortOrderEnum.ASCENDING ? equalTo(queryList.stream()
                .sorted(Comparator.comparing(String::valueOf, new CustomComparator()))
                .collect(toList())) : OrderMatcher.isDescendingOrdered();
        assertThat("Отчет вернулся с заданной сортировкой", queryList, expectedOrder);
    }

    private static class CustomComparator implements Comparator<String> {

        @Override
        public int compare(String o1, String o2) {
            if (NumberUtils.isNumber(o1) && NumberUtils.isNumber(o2)) {
                return (new BigDecimal(o1)).compareTo(new BigDecimal(o2));
            } else {
                return o1.compareToIgnoreCase(o2);
            }
        }
    }
}
