package ru.yandex.autotests.directapi.retargetinglists.add;

import com.yandex.direct.api.v5.retargetinglists.RetargetingListRuleOperatorEnum;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.api5.retargetinglists.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.retargetinglists.RetargetingListAddItemMap;
import ru.yandex.autotests.directapi.model.api5.retargetinglists.RetargetingListRuleArgumentItemMap;
import ru.yandex.autotests.directapi.model.api5.retargetinglists.RetargetingListRuleItemMap;
import ru.yandex.autotests.directapi.model.retargeting.RetargetingGoalType;
import ru.yandex.autotests.directapi.retargetinglists.RetargetingListsFeatures;
import ru.yandex.autotests.directapi.retargetinglists.RetargetingListsLogins;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static ru.yandex.autotests.directapi.apiclient.errors.Path.path;

/**
 * Created by buhter on 24/02/16
 * https://st.yandex-team.ru/TESTIRT-8519
 */
@Aqua.Test
@Features(RetargetingListsFeatures.ADD)
@Description("Валидация структуры RetargetingUserLists")
@Issue("https://st.yandex-team.ru/DIRECT-48447")
public class AddRetargetingListsValidationTest {
    private static long goalId;
    private static long segmentId;
    private static long audienceSegmentId;
    private static final String LOGIN = RetargetingListsLogins.LOGIN_RETARGETING26;
    private static final int CORRECT_LIFESPAN = 10;

    @ClassRule
    public static final ApiSteps api = new ApiSteps().as(LOGIN);

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static final SemaphoreRule semaphore = Semaphore.getSemaphore();

    @BeforeClass
    public static void prepare() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
        goalId = api.userSteps.retargetingSteps().getRetargetingGoalIDsByType(LOGIN, RetargetingGoalType.GOAL).get(0);
        segmentId =
                api.userSteps.retargetingSteps().getRetargetingGoalIDsByType(LOGIN, RetargetingGoalType.SEGMENT).get(0);
        audienceSegmentId = api.userSteps.retargetingSteps()
                .getRetargetingGoalIDsByType(LOGIN, RetargetingGoalType.AUDIENCE_SEGMENT).get(0);
    }

    @Test
    public void testAddSuccess() {
        api.userSteps.retargetingListsSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withRetargetingLists(
                                new RetargetingListAddItemMap()
                                        .defaultRetargetingListAddItemMap(goalId)),
                ExpectedResult.success());
    }

    @Test
    public void testAddEmptyRetargetingLists() {
        api.userSteps.retargetingListsSteps().expectErrorOnAdd(
                new AddRequestMap(),
                new Api5Error(8000, Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                        path(AddRequestMap.RETARGETING_LISTS)));
    }

    @Test
    public void testAddWithoutAnyExternalId() {
        api.userSteps.retargetingListsSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withRetargetingLists(
                                new RetargetingListAddItemMap()
                                        .withRandomName()
                                        .withRandomDescription()
                                        .withRules(new RetargetingListRuleItemMap()
                                                .withOperator(RetargetingListRuleOperatorEnum.ANY)
                                                .withArgumentItems(new RetargetingListRuleArgumentItemMap()
                                                        .withMembershipLifeSpan(CORRECT_LIFESPAN))))
                , ExpectedResult.errors(
                        new Notification(5005,
                                Api5ErrorDetailsJava.FIELD_MUST_BE_POSITIVE_INTEGER,
                                path(RetargetingListAddItemMap.RULES,
                                        0,
                                        RetargetingListRuleItemMap.ARGUMENTS,
                                        0,
                                        RetargetingListRuleArgumentItemMap.EXTERNAL_ID)
                        )));
    }

    @Test
    public void testAddGoalWithoutLisfespan() {
        api.userSteps.retargetingListsSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withRetargetingLists(
                                new RetargetingListAddItemMap()
                                        .withRandomName()
                                        .withRandomDescription()
                                        .withRules(new RetargetingListRuleItemMap()
                                                .withOperator(RetargetingListRuleOperatorEnum.ANY)
                                                .withArgumentItems(new RetargetingListRuleArgumentItemMap()
                                                        .withExternalId(goalId)))),
                ExpectedResult.errors(new Notification(5000,
                        Api5ErrorDetailsJava.MEMBERSHIP_LIFESPAN_MUST_BE_SET_FOR_GOAL_AND_SEGMENT,
                        path(RetargetingListAddItemMap.RULES,
                                0,
                                RetargetingListRuleItemMap.ARGUMENTS,
                                0))));
    }

    @Test
    public void testAddSegmentWithoutLisfespan() {
        api.userSteps.retargetingListsSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withRetargetingLists(
                                new RetargetingListAddItemMap()
                                        .withRandomName()
                                        .withRandomDescription()
                                        .withRules(new RetargetingListRuleItemMap()
                                                .withOperator(RetargetingListRuleOperatorEnum.ANY)
                                                .withArgumentItems(new RetargetingListRuleArgumentItemMap()
                                                        .withExternalId(segmentId)))),
                ExpectedResult.success());
    }

    @Test
    public void testAddAudienceSegmentWithoutLisfespan() {
        api.userSteps.retargetingListsSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withRetargetingLists(
                                new RetargetingListAddItemMap()
                                        .withRandomName()
                                        .withRandomDescription()
                                        .withRules(new RetargetingListRuleItemMap()
                                                .withOperator(RetargetingListRuleOperatorEnum.ANY)
                                                .withArgumentItems(new RetargetingListRuleArgumentItemMap()
                                                        .withExternalId(audienceSegmentId)))),
                ExpectedResult.success());
    }

    @Test
    public void testTestAddWithoutOperator() {
        api.userSteps.retargetingListsSteps().expectErrorOnAdd(new AddRequestMap()
                        .withRetargetingLists(
                                new RetargetingListAddItemMap()
                                        .withRandomName()
                                        .withRandomDescription()
                                        .withRules(new RetargetingListRuleItemMap()
                                                .withArgumentItems(new RetargetingListRuleArgumentItemMap()
                                                        .withMembershipLifeSpan(CORRECT_LIFESPAN)
                                                        .withExternalId(goalId)))),
                new Api5Error(8000, Api5ErrorDetailsJava.IN_ELEMENT_OF_ARRAY_MISSING_REQUIRED_FIELD,
                        path(AddRequestMap.RETARGETING_LISTS,
                                RetargetingListAddItemMap.RULES),
                        path(RetargetingListRuleItemMap.OPERATOR)));
    }

    @Test
    public void testAddWithoutArguments() {
        api.userSteps.retargetingListsSteps().expectErrorOnAdd(new AddRequestMap()
                        .withRetargetingLists(
                                new RetargetingListAddItemMap()
                                        .withRandomName()
                                        .withRandomDescription()
                                        .withRules(new RetargetingListRuleItemMap()
                                                .withOperator(RetargetingListRuleOperatorEnum.ANY))),
                new Api5Error(8000, Api5ErrorDetailsJava.IN_ELEMENT_OF_ARRAY_MISSING_REQUIRED_FIELD,
                        path(AddRequestMap.RETARGETING_LISTS,
                                RetargetingListAddItemMap.RULES),
                        path(RetargetingListRuleItemMap.ARGUMENTS)));
    }

    @Test
    public void testAddWithoutRules() {
        api.userSteps.retargetingListsSteps().expectErrorOnAdd(new AddRequestMap()
                        .withRetargetingLists(
                                new RetargetingListAddItemMap()
                                        .withRandomName()
                                        .withRandomDescription()),
                new Api5Error(8000, Api5ErrorDetailsJava.IN_ELEMENT_OF_ARRAY_MISSING_REQUIRED_FIELD,
                        path(AddRequestMap.RETARGETING_LISTS),
                        path(RetargetingListAddItemMap.RULES)));
    }

    @Test
    public void testAddWithoutDescription() {
        api.userSteps.retargetingListsSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withRetargetingLists(
                                new RetargetingListAddItemMap()
                                        .withRandomName()
                                        .withRules(new RetargetingListRuleItemMap()
                                                .withOperator(RetargetingListRuleOperatorEnum.ANY)
                                                .withArgumentItems(new RetargetingListRuleArgumentItemMap()
                                                        .withMembershipLifeSpan(CORRECT_LIFESPAN)
                                                        .withExternalId(goalId)))),
                ExpectedResult.success());
    }

    @Test
    public void testAddWithoutName() {
        api.userSteps.retargetingListsSteps().expectErrorOnAdd(new AddRequestMap()
                        .withRetargetingLists(
                                new RetargetingListAddItemMap()
                                        .withRandomDescription()
                                        .withRules(new RetargetingListRuleItemMap()
                                                .withOperator(RetargetingListRuleOperatorEnum.ANY)
                                                .withArgumentItems(new RetargetingListRuleArgumentItemMap()
                                                        .withMembershipLifeSpan(CORRECT_LIFESPAN)
                                                        .withExternalId(goalId)))),
                new Api5Error(8000, Api5ErrorDetailsJava.IN_ELEMENT_OF_ARRAY_MISSING_REQUIRED_FIELD,
                        path(AddRequestMap.RETARGETING_LISTS),
                        path(RetargetingListAddItemMap.NAME)));
    }
}
