package ru.yandex.autotests.directapi.retargetinglists.add.rules;

import java.util.List;

import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.api5.retargetinglists.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.retargetinglists.RetargetingListAddItemMap;
import ru.yandex.autotests.directapi.model.api5.retargetinglists.RetargetingListRuleArgumentItemMap;
import ru.yandex.autotests.directapi.model.api5.retargetinglists.RetargetingListRuleItemMap;
import ru.yandex.autotests.directapi.model.retargeting.RetargetingGoalType;
import ru.yandex.autotests.directapi.retargetinglists.RetargetingListsFeatures;
import ru.yandex.autotests.directapi.retargetinglists.RetargetingListsLogins;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.greaterThan;
import static org.hamcrest.Matchers.hasSize;
import static ru.yandex.autotests.directapi.apiclient.errors.Path.path;
import static ru.yandex.autotests.directapi.retargetinglists.RetargetingListsLogins.LOGIN_WITH_YUID_SEGMENT;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by buhter on 24/02/16
 * https://st.yandex-team.ru/TESTIRT-8519
 */
@Aqua.Test
@Features(RetargetingListsFeatures.ADD)
@Description("Проверка валидации поля ExternalId")
@Issue("https://st.yandex-team.ru/DIRECT-48447")
public class AddRetargetingListsExternalIdValidationTest {
    private static Long goalId1;
    private static Long goalId2;
    private static final String ANOTHER_LOGIN = RetargetingListsLogins.LOGIN_RETARGETING46;
    private static final String LOGIN = RetargetingListsLogins.LOGIN_RETARGETING45;

    private static final String LOGIN_WITH_ACCESS_TO_FOREIGN_GOAL = RetargetingListsLogins.WITH_ACCESS_TO_FOREIGN_GOAL;
    private static final String LOGIN_WITH_GOAL = RetargetingListsLogins.WITH_GOAL;
    private static final String LOGIN_METRIKA_REPRESENTATIVE = RetargetingListsLogins.METRIKA_REPRESENTATIVE;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);
    private static Long anotherCLientGoalId;

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @BeforeClass
    public static void prepare() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
        anotherCLientGoalId = api.as(ANOTHER_LOGIN).userSteps.retargetingSteps()
                .getRetargetingGoalIDsByType(ANOTHER_LOGIN, RetargetingGoalType.GOAL).get(0);
        List<Long> goalIds =
                api.as(LOGIN).userSteps.retargetingSteps().getRetargetingGoalIDsByType(LOGIN, RetargetingGoalType.GOAL);
        assumeThat("у клиента должно быть > 1 цели", goalIds.size(), greaterThan(1));
        goalId1 = goalIds.get(0);
        goalId2 = goalIds.get(1);
    }

    @Test
    public void testAddWithCorrectGoalId() {
        api.userSteps.retargetingListsSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withRetargetingLists(
                                new RetargetingListAddItemMap()
                                        .defaultRetargetingListAddItemMap(goalId1)),
                ExpectedResult.success());
    }

    @Test
    public void testAddWithZeroGoalId() {
        api.userSteps.retargetingListsSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withRetargetingLists(
                                new RetargetingListAddItemMap()
                                        .defaultRetargetingListAddItemMap(0L)),
                ExpectedResult.errors(
                        new Notification(5005,
                                Api5ErrorDetailsJava.FIELD_MUST_BE_POSITIVE_INTEGER,
                                path(RetargetingListAddItemMap.RULES,
                                        0,
                                        RetargetingListRuleItemMap.ARGUMENTS,
                                        0,
                                        RetargetingListRuleArgumentItemMap.EXTERNAL_ID)
                        )));
    }

    @Test
    public void testAddWithNegativeGoalId() {
        api.userSteps.retargetingListsSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withRetargetingLists(
                                new RetargetingListAddItemMap()
                                        .defaultRetargetingListAddItemMap(-1L)),
                ExpectedResult.errors(
                        new Notification(5005,
                                Api5ErrorDetailsJava.FIELD_MUST_BE_POSITIVE_INTEGER,
                                path(RetargetingListAddItemMap.RULES,
                                        0,
                                        RetargetingListRuleItemMap.ARGUMENTS,
                                        0,
                                        RetargetingListRuleArgumentItemMap.EXTERNAL_ID)
                        )));
    }

    @Test
    public void testAddWithOthersGoalId() {
        api.userSteps.retargetingListsSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withRetargetingLists(
                                new RetargetingListAddItemMap()
                                        .defaultRetargetingListAddItemMap(anotherCLientGoalId)),
                ExpectedResult.errors(new Notification(8800)));
    }

    @Test
    public void testAddWithOthersGoalId_WithAccessToGoal() {
        Long anotherClientGoalId = api.as(LOGIN_WITH_GOAL).userSteps.retargetingSteps()
                .getRetargetingGoalIDsByType(LOGIN_WITH_GOAL, RetargetingGoalType.GOAL).get(0);

        api.as(LOGIN_WITH_ACCESS_TO_FOREIGN_GOAL).userSteps.retargetingListsSteps().shouldGetResultOnAdd(
                new AddRequestMap()
                        .withRetargetingLists(
                                new RetargetingListAddItemMap()
                                        .defaultRetargetingListAddItemMap(anotherClientGoalId)),
                ExpectedResult.success());
    }

    @Test
    public void testAddWithOthersGoalId_MetrikaRepresentative() {
        Long anotherClientGoalId = api.as(LOGIN_WITH_GOAL).userSteps.retargetingSteps()
                .getRetargetingGoalIDsByType(LOGIN_WITH_GOAL, RetargetingGoalType.GOAL).get(0);

        api.as(LOGIN_METRIKA_REPRESENTATIVE).userSteps.retargetingListsSteps().shouldGetResultOnAdd(
                new AddRequestMap()
                        .withRetargetingLists(
                                new RetargetingListAddItemMap()
                                        .defaultRetargetingListAddItemMap(anotherClientGoalId)),
                ExpectedResult.success());
    }

    @Test
    public void testAddWithTwoCorrectGoalIds() {
        api.userSteps.retargetingListsSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withRetargetingLists(
                                new RetargetingListAddItemMap()
                                        .defaultRetargetingListAddItemMap(
                                                new RetargetingListRuleArgumentItemMap()
                                                        .withRandomMembershipLifeSpan()
                                                        .withExternalId(goalId1),
                                                new RetargetingListRuleArgumentItemMap()
                                                        .withRandomMembershipLifeSpan()
                                                        .withExternalId(goalId2)
                                        )),
                ExpectedResult.success());
    }

    @Test
    public void testAddWithSegmentYuid() {
        List<Long> goalIds = api.as(LOGIN_WITH_YUID_SEGMENT).userSteps.retargetingSteps()
                .getRetargetingGoalIDsByType(LOGIN_WITH_YUID_SEGMENT, RetargetingGoalType.AUDIENCE_SEGMENT);

        assumeThat("У пользователя только одна цель", goalIds, hasSize(1));

        Long yuidSegmentGoalId = goalIds.get(0);

        api.as(LOGIN_WITH_YUID_SEGMENT).userSteps.retargetingListsSteps().shouldGetResultOnAdd(
                new AddRequestMap()
                        .withRetargetingLists(
                                new RetargetingListAddItemMap()
                                        .defaultRetargetingListAddItemMap(yuidSegmentGoalId)),
                ExpectedResult.success());
    }

    @Test
    public void testAddWithCorrectAndOthersGoalIds() {
        api.userSteps.retargetingListsSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withRetargetingLists(
                                new RetargetingListAddItemMap()
                                        .defaultRetargetingListAddItemMap(
                                                new RetargetingListRuleArgumentItemMap()
                                                        .withRandomMembershipLifeSpan()
                                                        .withExternalId(goalId1),
                                                new RetargetingListRuleArgumentItemMap()
                                                        .withRandomMembershipLifeSpan()
                                                        .withExternalId(anotherCLientGoalId))),
                ExpectedResult.errors(new Notification(8800)));
    }

    @Test
    public void testAddWithCorrectAndNegativeGoalIds() {
        api.userSteps.retargetingListsSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withRetargetingLists(
                                new RetargetingListAddItemMap()
                                        .defaultRetargetingListAddItemMap(
                                                new RetargetingListRuleArgumentItemMap()
                                                        .withRandomMembershipLifeSpan()
                                                        .withExternalId(goalId2),
                                                new RetargetingListRuleArgumentItemMap()
                                                        .withRandomMembershipLifeSpan()
                                                        .withExternalId(-1L))),
                ExpectedResult.errors(
                        new Notification(5005,
                                Api5ErrorDetailsJava.FIELD_MUST_BE_POSITIVE_INTEGER,
                                path(RetargetingListAddItemMap.RULES,
                                        0,
                                        RetargetingListRuleItemMap.ARGUMENTS,
                                        1,
                                        RetargetingListRuleArgumentItemMap.EXTERNAL_ID)
                        )));
    }

    @Test
    public void testAddWithTwoSameCorrectGoalIds() {
        api.userSteps.retargetingListsSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withRetargetingLists(
                                new RetargetingListAddItemMap()
                                        .defaultRetargetingListAddItemMap(
                                                new RetargetingListRuleArgumentItemMap()
                                                        .withRandomMembershipLifeSpan()
                                                        .withExternalId(goalId1),
                                                new RetargetingListRuleArgumentItemMap()
                                                        .withRandomMembershipLifeSpan()
                                                        .withExternalId(goalId1))),
                ExpectedResult.success());
    }
}
