package ru.yandex.autotests.directapi.retargetinglists.delete;

import java.util.List;

import com.yandex.direct.api.v5.retargetinglists.RetargetingListRuleOperatorEnum;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.api5.audiencetargets.AudienceTargetAddItemMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.IdsCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.api5.retargetinglists.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.retargetinglists.DeleteRequestMap;
import ru.yandex.autotests.directapi.model.api5.retargetinglists.RetargetingListAddItemMap;
import ru.yandex.autotests.directapi.model.api5.retargetinglists.RetargetingListRuleArgumentItemMap;
import ru.yandex.autotests.directapi.model.api5.retargetinglists.RetargetingListRuleItemMap;
import ru.yandex.autotests.directapi.model.retargeting.RetargetingGoalType;
import ru.yandex.autotests.directapi.retargetinglists.RetargetingListsFeatures;
import ru.yandex.autotests.directapi.retargetinglists.RetargetingListsLogins;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.greaterThan;
import static org.hamcrest.Matchers.iterableWithSize;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by buhter on 02/03/16
 * https://st.yandex-team.ru/TESTIRT-8519
 */
@Aqua.Test
@Features(RetargetingListsFeatures.DELETE)
@Description("Проверка корректности работы метода с валидными данными")
@Issue("https://st.yandex-team.ru/DIRECT-48447")
public class DeleteRetargetingListsTest {
    @ClassRule
    public static final SemaphoreRule semaphore = Semaphore.getSemaphore();
    private static final String LOGIN = RetargetingListsLogins.LOGIN_RETARGETING42;
    @ClassRule
    public static final ApiSteps api = new ApiSteps().as(LOGIN);
    @Rule
    public Trashman trashman = new Trashman(api);
    private long goalId1;
    private long goalId2;

    @Before
    public void prepare() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
        List<Long> goalIds =
                api.userSteps.retargetingSteps().getRetargetingGoalIDsByType(LOGIN, RetargetingGoalType.GOAL);
        assumeThat("у клиента должно быть > 1 цели", goalIds.size(), greaterThan(1));
        goalId1 = goalIds.get(0);
        goalId2 = goalIds.get(1);

        List<Long> loginRetargetingListIds = api.userSteps.retargetingListsSteps().getIdsByLogin(LOGIN);
        if (!loginRetargetingListIds.isEmpty()) {
            api.userSteps.retargetingListsSteps().delete(loginRetargetingListIds);
        }
    }

    @Test
    public void successDeleteOneRetargetingLists() {
        Long id = api.userSteps.retargetingListsSteps().addDefaultRetargetingListWithoutSearch(goalId1);
        api.userSteps.retargetingListsSteps().shouldGetResultOnDelete(
                new DeleteRequestMap().withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(id)),
                ExpectedResult.success()
        );
    }

    @Test
    public void successDeleteTwoRetargetingLists() {
        List<Long> idList = api.userSteps.retargetingListsSteps().add(new AddRequestMap()
                .withRetargetingLists(new RetargetingListAddItemMap()
                                .withRandomName()
                                .withRules(new RetargetingListRuleItemMap()
                                        .withOperator(RetargetingListRuleOperatorEnum.ALL)
                                        .withArgumentItems(new RetargetingListRuleArgumentItemMap()
                                                .withRandomMembershipLifeSpan()
                                                .withExternalId(goalId1))),
                        new RetargetingListAddItemMap()
                                .withRandomName()
                                .withRules(new RetargetingListRuleItemMap()
                                        .withOperator(RetargetingListRuleOperatorEnum.ANY)
                                        .withArgumentItems(new RetargetingListRuleArgumentItemMap()
                                                .withRandomMembershipLifeSpan()
                                                .withExternalId(goalId2))))
        );

        assumeThat("добавилось два объекта", idList, iterableWithSize(2));
        api.userSteps.retargetingListsSteps().shouldGetResultOnDelete(
                new DeleteRequestMap().withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(idList.get(0), idList.get(1))),
                ExpectedResult.success(),
                ExpectedResult.success()
        );
    }

    @Test
    public void tryDeleteRetargetingListsInAudienceTargets() {
        Long id = api.userSteps.retargetingListsSteps().addDefaultRetargetingLists(goalId1);
        Long cid = api.userSteps.campaignSteps().addDefaultTextCampaign();
        Long pid = api.userSteps.adGroupsSteps().addDefaultGroup(cid);
        api.userSteps.audienceTargetsSteps()
                .shouldGetResultOnAdd(new ru.yandex.autotests.directapi.model.api5.audiencetargets.AddRequestMap()
                                .withAudienceTargets(new AudienceTargetAddItemMap()
                                        .withAdGroupId(pid)
                                        .withRetargetingListId(id)
                                        .withContextBid(MoneyCurrency.get(User.get(LOGIN)
                                                .getCurrency()).getLongMinPrice().longValue())),
                        ExpectedResult.success());
        api.userSteps.retargetingListsSteps().shouldGetResultOnDelete(
                new DeleteRequestMap().withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(id)),
                ExpectedResult.errors(new Notification(8301,
                        Api5ErrorDetailsJava.CAN_NOT_DELETE_VALUE_ONLY, id))
        );
    }

    @Test
    public void tryDeleteRetargetingListsInBidModifier() {
        Long id = api.userSteps.retargetingListsSteps().addDefaultRetargetingLists(goalId1);
        Long cid = api.userSteps.campaignSteps().addDefaultTextCampaign();
        api.userSteps.adGroupsSteps().addDefaultGroup(cid);
        api.userSteps.bidModifiersSteps().addBidModifierRetargetingToCampaign(id, cid.intValue());
        api.userSteps.retargetingListsSteps().shouldGetResultOnDelete(
                new DeleteRequestMap().withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(id)),
                ExpectedResult.errors(new Notification(8301,
                        Api5ErrorDetailsJava.CAN_NOT_DELETE_VALUE_ONLY, id))
        );
    }

    @Test
    public void tryDeleteTwoRetargetingListsOneInAudienceTargets() {
        Long id1 = api.userSteps.retargetingListsSteps().addDefaultRetargetingLists(goalId1);
        Long id2 = api.userSteps.retargetingListsSteps().addDefaultRetargetingLists(goalId2);
        Long cid = api.userSteps.campaignSteps().addDefaultTextCampaign();
        Long pid = api.userSteps.adGroupsSteps().addDefaultGroup(cid);
        api.userSteps.audienceTargetsSteps()
                .shouldGetResultOnAdd(new ru.yandex.autotests.directapi.model.api5.audiencetargets.AddRequestMap()
                                .withAudienceTargets(new AudienceTargetAddItemMap()
                                        .withAdGroupId(pid)
                                        .withRetargetingListId(id1)
                                        .withContextBid(MoneyCurrency.get(User.get(LOGIN)
                                                .getCurrency()).getLongMinPrice().longValue())),
                        ExpectedResult.success());
        api.userSteps.retargetingListsSteps().shouldGetResultOnDelete(
                new DeleteRequestMap().withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(id1, id2)),
                ExpectedResult.errors(new Notification(8301,
                        Api5ErrorDetailsJava.CAN_NOT_DELETE_VALUE_ONLY, id1)),
                ExpectedResult.success()
        );
    }

    @Test
    public void tryDeleteTwoRetargetingListsOneInBidModifier() {
        Long id1 = api.userSteps.retargetingListsSteps().addDefaultRetargetingLists(goalId1);
        Long id2 = api.userSteps.retargetingListsSteps().addDefaultRetargetingLists(goalId2);
        Long cid = api.userSteps.campaignSteps().addDefaultTextCampaign();
        api.userSteps.adGroupsSteps().addDefaultGroup(cid);
        api.userSteps.bidModifiersSteps().addBidModifierRetargetingToCampaign(id1, cid.intValue());
        DeleteRequestMap deleteRequestMap = new DeleteRequestMap().withSelectionCriteria(new IdsCriteriaMap()
                .withIds(id1, id2));
        api.userSteps.retargetingListsSteps().shouldGetResultOnDelete(
                deleteRequestMap,
                ExpectedResult.errors(new Notification(8301,
                        Api5ErrorDetailsJava.CAN_NOT_DELETE_VALUE_ONLY, id1)),
                ExpectedResult.success()
        );
    }

    @Test
    public void successDeleteRetargetingListsInAudienceTargetsAfterDeleteCampaign() {
        Long id = api.userSteps.retargetingListsSteps().addDefaultRetargetingListWithoutSearch(goalId1);
        Long cid = api.userSteps.campaignSteps().addDefaultTextCampaign();
        Long pid = api.userSteps.adGroupsSteps().addDefaultGroup(cid);
        api.userSteps.audienceTargetsSteps()
                .shouldGetResultOnAdd(new ru.yandex.autotests.directapi.model.api5.audiencetargets.AddRequestMap()
                                .withAudienceTargets(new AudienceTargetAddItemMap()
                                        .withAdGroupId(pid)
                                        .withRetargetingListId(id)
                                        .withContextBid(MoneyCurrency.get(User.get(LOGIN)
                                                .getCurrency()).getLongMinPrice().longValue())),
                        ExpectedResult.success());
        api.userSteps.campaignSteps().campaignsDelete(cid);
        api.userSteps.retargetingListsSteps().shouldGetResultOnDelete(
                new DeleteRequestMap().withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(id)),
                ExpectedResult.success()
        );
    }

//    @Test
//    public void tryDeleteRetargetingListsInBidModifierAfterDeleteCampaign() {
//        //DIRECT-58166
//        Long id = api.userSteps.retargetingListsSteps().addDefaultRetargetingLists(goalId1);
//        Long cid = api.userSteps.campaignSteps().addDefaultTextCampaign();
//        api.userSteps.adGroupsSteps().addDefaultGroup(cid);
//        api.userSteps.bidModifiersSteps().addBidModifierRetargetingToCampaign(id, cid.intValue());
//        api.userSteps.campaignSteps().campaignsDelete(cid);
//        api.userSteps.retargetingListsSteps().shouldGetResultOnDelete(
//                new DeleteRequestMap().withSelectionCriteria(new IdsCriteriaMap()
//                        .withIds(id)),
//                ExpectedResult.success()
//        );
//    }
}
