package ru.yandex.autotests.directapi.retargetinglists.get;

import java.util.List;

import com.yandex.direct.api.v5.retargetinglists.RetargetingListFieldEnum;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.retargetinglists.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.retargetinglists.RetargetingListSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.retargeting.RetargetingGoalType;
import ru.yandex.autotests.directapi.retargetinglists.RetargetingListsFeatures;
import ru.yandex.autotests.directapi.retargetinglists.RetargetingListsLogins;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.greaterThan;
import static ru.yandex.autotests.directapi.apiclient.errors.Path.path;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * https://st.yandex-team.ru/TESTIRT-8519
 */
@Aqua.Test
@Features(RetargetingListsFeatures.GET)
@Description("Проверка списания баллов API")
@Issue("https://st.yandex-team.ru/DIRECT-48447")
public class GetRetargetingListUnitTest {
    private static final int GET_COST = 1;
    private static final int ITEM_COST_ON_GET = 1;
    private static final int GENERAL_FAULT_COST = 50;
    private static Long id1;
    private static Long goalId1;
    private static Long goalId2;
    private Long id2;
    private static final String LOGIN = RetargetingListsLogins.LOGIN_RETARGETING15;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @BeforeClass
    public static void prepare() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
        List<Long> goalIds =
                api.userSteps.retargetingSteps().getRetargetingGoalIDsByType(LOGIN, RetargetingGoalType.GOAL);
        assumeThat("у клиента должно быть > 1 цели", goalIds.size(), greaterThan(1));
        goalId1 = goalIds.get(0);
        goalId2 = goalIds.get(1);
        id1 = api.userSteps.retargetingListsSteps().addDefaultRetargetingLists(goalId1);
    }

    @Before
    public void init() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
    }

    @Test
    public void getSuccess() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.retargetingListsSteps().get(
                new GetRequestMap()
                        .withFieldNames(RetargetingListFieldEnum.ID)
                        .withSelectionCriteria(new RetargetingListSelectionCriteriaMap()
                                .withIds(id1)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        assertThat("стоимость получения одного объекта", unitsBefore - unitsAfter,
                equalTo(GET_COST + ITEM_COST_ON_GET));
    }

    @Test
    public void getSomeRetargetingLists() {
        id2 = api.userSteps.retargetingListsSteps().addDefaultRetargetingLists(goalId2);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.retargetingListsSteps().get(
                new GetRequestMap()
                        .withFieldNames(RetargetingListFieldEnum.ID)
                        .withSelectionCriteria(new RetargetingListSelectionCriteriaMap()
                                .withIds(id1, id2)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        assertThat("стоимость получения двух объектов", unitsBefore - unitsAfter,
                equalTo(GET_COST + ITEM_COST_ON_GET * 2));
    }

    @Test
    public void getNonExistingRetargetingLists() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.retargetingListsSteps().get(
                new GetRequestMap()
                        .withFieldNames(RetargetingListFieldEnum.ID)
                        .withSelectionCriteria(new RetargetingListSelectionCriteriaMap()
                                .withIds(1L)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        assertThat("списалось верное кол-во баллов", unitsBefore - unitsAfter, equalTo(GET_COST));
    }

    @Test
    public void generalErrorOnGetRetargetingLists() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.retargetingListsSteps().expectErrorOnGet(
                new GetRequestMap()
                        .withSelectionCriteria(new RetargetingListSelectionCriteriaMap()),
                new Api5Error(8000, Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                        path(GetRequestMap.FIELD_NAMES))
        );
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        assertThat("списалось верное кол-во баллов", unitsBefore - unitsAfter, equalTo(GENERAL_FAULT_COST));
    }

    @Test
    public void notEnoughUnitsOnGetRetargetingLists() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.clientFakeSteps().fakeWithdrawClientUnits(LOGIN, unitsBefore);
        unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        assumeThat("доступные баллы пользователя", unitsBefore, equalTo(0));

        api.userSteps.retargetingListsSteps().expectErrorOnGet(
                new GetRequestMap()
                        .withFieldNames(RetargetingListFieldEnum.ID)
                        .withSelectionCriteria(new RetargetingListSelectionCriteriaMap()
                                .withIds(id1)),
                new Api5Error(152, Api5ErrorDetails.NOT_ENOUGH_UNITS_FOR_OPERATION));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        assertThat("доступные баллы пользователя после ошибки", unitsAfter, equalTo(0));
    }

}
