package ru.yandex.autotests.directapi.retargetinglists.update;

import java.util.List;

import com.yandex.direct.api.v5.retargetinglists.RetargetingListRuleOperatorEnum;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.api5.retargetinglists.RetargetingListRuleArgumentItemMap;
import ru.yandex.autotests.directapi.model.api5.retargetinglists.RetargetingListRuleItemMap;
import ru.yandex.autotests.directapi.model.api5.retargetinglists.RetargetingListUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.retargetinglists.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.retargeting.RetargetingGoalType;
import ru.yandex.autotests.directapi.retargetinglists.RetargetingListsFeatures;
import ru.yandex.autotests.directapi.retargetinglists.RetargetingListsLogins;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.greaterThan;
import static ru.yandex.autotests.directapi.apiclient.errors.Path.path;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * https://st.yandex-team.ru/TESTIRT-8519
 */
@Aqua.Test
@Features(RetargetingListsFeatures.UPDATE)
@Description("Проверка списания баллов")
@Issue("https://st.yandex-team.ru/DIRECT-48447")
public class UpdateRetargetingListsUnitTest {
    private static long correctGoalId;
    private static long anotherCorrectGoalId;
    private static final String LOGIN = RetargetingListsLogins.LOGIN_RETARGETING10;
    private static Long id1;
    private static Long id2;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    private static final int CALL_COST = 10;
    private static final int RETARGETING_LIST_COST = 2;
    private static final int ERROR_COST = 20;
    private static final int COMMON_REQUEST_ERROR_COST = 50;

    @BeforeClass
    public static void addRetargetingLists() {
        List<Long> goalIds =
                api.userSteps.retargetingSteps().getRetargetingGoalIDsByType(LOGIN, RetargetingGoalType.GOAL);
        assumeThat("получено больше 1 объекта", goalIds.size(), greaterThan(1));
        correctGoalId = goalIds.get(0);
        anotherCorrectGoalId = goalIds.get(1);
        id1 = api.userSteps.retargetingListsSteps().addDefaultRetargetingLists(correctGoalId);
        id2 = api.userSteps.retargetingListsSteps().addDefaultRetargetingLists(anotherCorrectGoalId);
    }

    @Before
    public void addUnitsToClient() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
    }

    @Test
    public void updateOneRetargetingList() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.retargetingListsSteps().shouldGetResultOnUpdate(new UpdateRequestMap()
                        .withRetargetingLists(
                                new RetargetingListUpdateItemMap()
                                        .withId(id1)
                                        .withRules(new RetargetingListRuleItemMap()
                                                .withOperator(RetargetingListRuleOperatorEnum.ANY)
                                                .withArgumentItems(new RetargetingListRuleArgumentItemMap()
                                                        .withRandomMembershipLifeSpan()
                                                        .withExternalId(correctGoalId)))),
                ExpectedResult.success());
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + RETARGETING_LIST_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    public void updateTwoRetargetingLists() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.retargetingListsSteps().shouldGetResultOnUpdate(new UpdateRequestMap()
                        .withRetargetingLists(
                                new RetargetingListUpdateItemMap()
                                        .withId(id1)
                                        .withRules(new RetargetingListRuleItemMap()
                                                .withOperator(RetargetingListRuleOperatorEnum.ALL)
                                                .withArgumentItems(new RetargetingListRuleArgumentItemMap()
                                                        .withRandomMembershipLifeSpan()
                                                        .withExternalId(correctGoalId))),
                                new RetargetingListUpdateItemMap()
                                        .withId(id2)
                                        .withRules(new RetargetingListRuleItemMap()
                                                .withOperator(RetargetingListRuleOperatorEnum.ANY)
                                                .withArgumentItems(new RetargetingListRuleArgumentItemMap()
                                                        .withMembershipLifeSpan(
                                                                RetargetingListRuleArgumentItemMap.DEFAULT_MEMBERSHIP_LIFE_SPAN
                                                                        + 1)
                                                        .withExternalId(anotherCorrectGoalId)))),
                ExpectedResult.success(), ExpectedResult.success());
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + RETARGETING_LIST_COST * 2;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }


    @Test
    public void updateOneInvalidRetargetingLists() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.retargetingListsSteps().shouldGetResultOnUpdate(new UpdateRequestMap()
                        .withRetargetingLists(
                                new RetargetingListUpdateItemMap()
                                        .withId(id2)
                                        .withRules(new RetargetingListRuleItemMap()
                                                .withOperator(RetargetingListRuleOperatorEnum.ANY)
                                                .withArgumentItems(new RetargetingListRuleArgumentItemMap()
                                                        .withRandomMembershipLifeSpan()))),
                ExpectedResult.errors(
                        new Notification(5005,
                                Api5ErrorDetailsJava.FIELD_MUST_BE_POSITIVE_INTEGER,
                                path(RetargetingListUpdateItemMap.RULES,
                                        0,
                                        RetargetingListRuleItemMap.ARGUMENTS,
                                        0,
                                        RetargetingListRuleArgumentItemMap.EXTERNAL_ID)
                        )));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + ERROR_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    public void updateTwoInvalidRetargetingLists() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.retargetingListsSteps().shouldGetResultOnUpdate(new UpdateRequestMap()
                        .withRetargetingLists(
                                new RetargetingListUpdateItemMap()
                                        .withId(id1)
                                        .withRules(new RetargetingListRuleItemMap()
                                                .withOperator(RetargetingListRuleOperatorEnum.ALL)
                                                .withArgumentItems(new RetargetingListRuleArgumentItemMap()
                                                        .withRandomMembershipLifeSpan())),
                                new RetargetingListUpdateItemMap()
                                        .withId(id2)
                                        .withRules(new RetargetingListRuleItemMap()
                                                .withOperator(RetargetingListRuleOperatorEnum.ANY)
                                                .withArgumentItems(new RetargetingListRuleArgumentItemMap()
                                                        .withRandomMembershipLifeSpan()))),

                ExpectedResult.errors(
                        new Notification(5005,
                                Api5ErrorDetailsJava.FIELD_MUST_BE_POSITIVE_INTEGER,
                                path(RetargetingListUpdateItemMap.RULES,
                                        0,
                                        RetargetingListRuleItemMap.ARGUMENTS,
                                        0,
                                        RetargetingListRuleArgumentItemMap.EXTERNAL_ID)
                        )),
                ExpectedResult.errors(
                        new Notification(5005,
                                Api5ErrorDetailsJava.FIELD_MUST_BE_POSITIVE_INTEGER,
                                path(RetargetingListUpdateItemMap.RULES,
                                        0,
                                        RetargetingListRuleItemMap.ARGUMENTS,
                                        0,
                                        RetargetingListRuleArgumentItemMap.EXTERNAL_ID)
                        )));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + ERROR_COST * 2;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    public void updateOneValidAndOneInvalidRetargetingLists() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.retargetingListsSteps().shouldGetResultOnUpdate(new UpdateRequestMap()
                        .withRetargetingLists(
                                new RetargetingListUpdateItemMap()
                                        .withId(id1)
                                        .withRules(new RetargetingListRuleItemMap()
                                                .withOperator(RetargetingListRuleOperatorEnum.ANY)
                                                .withArgumentItems(new RetargetingListRuleArgumentItemMap()
                                                        .withRandomMembershipLifeSpan()
                                                        .withExternalId(correctGoalId))),
                                new RetargetingListUpdateItemMap()
                                        .withId(id2)
                                        .withRules(new RetargetingListRuleItemMap()
                                                .withOperator(RetargetingListRuleOperatorEnum.ALL)
                                                .withArgumentItems(new RetargetingListRuleArgumentItemMap()
                                                        .withRandomMembershipLifeSpan()))),

                ExpectedResult.success(),
                ExpectedResult.errors(
                        new Notification(5005,
                                Api5ErrorDetailsJava.FIELD_MUST_BE_POSITIVE_INTEGER,
                                path(RetargetingListUpdateItemMap.RULES,
                                        0,
                                        RetargetingListRuleItemMap.ARGUMENTS,
                                        0,
                                        RetargetingListRuleArgumentItemMap.EXTERNAL_ID)
                        )));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + ERROR_COST + RETARGETING_LIST_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    public void invalidRequest() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.retargetingListsSteps().expectErrorOnUpdate(new UpdateRequestMap()
                        .withRetargetingLists(
                                new RetargetingListUpdateItemMap()
                                        .withId(id1)
                                        .withRules(new RetargetingListRuleItemMap()
                                                .withOperator(RetargetingListRuleOperatorEnum.ANY))),
                new Api5Error(8000, Api5ErrorDetailsJava.IN_ELEMENT_OF_ARRAY_MISSING_REQUIRED_FIELD,
                        path(UpdateRequestMap.RETARGETING_LISTS,
                                RetargetingListUpdateItemMap.RULES),
                        path(RetargetingListRuleItemMap.ARGUMENTS)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = COMMON_REQUEST_ERROR_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    public void updateRetargetingListsWhenClientHasZeroUnits() {
        int units = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.clientFakeSteps().fakeWithdrawClientUnits(LOGIN, units);
        units = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        assumeThat("у клиента нету баллов", units, equalTo(0));
        api.userSteps.retargetingListsSteps().expectErrorOnUpdate(
                new UpdateRequestMap()
                        .withRetargetingLists(
                                new RetargetingListUpdateItemMap()
                                        .withId(id1)
                                        .withRules(new RetargetingListRuleItemMap()
                                                .withOperator(RetargetingListRuleOperatorEnum.ANY)
                                                .withArgumentItems(new RetargetingListRuleArgumentItemMap()
                                                        .withRandomMembershipLifeSpan()
                                                        .withExternalId(correctGoalId)))),
                new Api5Error(152, Api5ErrorDetailsJava.NOT_ENOUGH_UNITS));
    }
}
