package ru.yandex.autotests.directapi.retargetinglists.update;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import com.yandex.direct.api.v5.retargetinglists.RetargetingListFieldEnum;
import com.yandex.direct.api.v5.retargetinglists.RetargetingListGetItem;
import com.yandex.direct.api.v5.retargetinglists.RetargetingListRuleOperatorEnum;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.api5.retargetinglists.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.retargetinglists.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.retargetinglists.RetargetingListAddItemMap;
import ru.yandex.autotests.directapi.model.api5.retargetinglists.RetargetingListRuleArgumentItemMap;
import ru.yandex.autotests.directapi.model.api5.retargetinglists.RetargetingListRuleItemMap;
import ru.yandex.autotests.directapi.model.api5.retargetinglists.RetargetingListUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.retargetinglists.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.retargeting.RetargetingGoalType;
import ru.yandex.autotests.directapi.retargetinglists.RetargetingListsFeatures;
import ru.yandex.autotests.directapi.retargetinglists.RetargetingListsLogins;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.iterableWithSize;
import static ru.yandex.autotests.directapi.apiclient.errors.Path.path;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by buhter on 24/02/16
 * https://st.yandex-team.ru/TESTIRT-8519
 */
@Aqua.Test
@Features(RetargetingListsFeatures.UPDATE)
@Description("Общие проверки метода")
@Issue("https://st.yandex-team.ru/DIRECT-48447")
public class UpdateRetargetingListsValidationTest {
    @ClassRule
    public static final SemaphoreRule semaphore = Semaphore.getSemaphore();
    private static final String LOGIN = RetargetingListsLogins.LOGIN_RETARGETING9;
    @ClassRule
    public static final ApiSteps api = new ApiSteps().as(LOGIN);
    private static final int CORRECT_LIFESPAN = 10;
    private static final String NAME = "Name";
    private static final String DESCRIPTION = "Description";
    private static final RetargetingListRuleOperatorEnum OPERATOR_ENUM = RetargetingListRuleOperatorEnum.ANY;
    private static long goalId;
    private static long segmentId;
    private static long audienceSegmentId;
    @Rule
    public Trashman trashman = new Trashman(api);
    private List<Long> idList;

    @BeforeClass
    public static void prepare() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
        goalId = api.userSteps.retargetingSteps().getRetargetingGoalIDsByType(LOGIN, RetargetingGoalType.GOAL).get(0);
        segmentId = api.userSteps.retargetingSteps()
                .getRetargetingGoalIDsByType(LOGIN, RetargetingGoalType.SEGMENT).get(0);
        audienceSegmentId = api.userSteps.retargetingSteps()
                .getRetargetingGoalIDsByType(LOGIN, RetargetingGoalType.AUDIENCE_SEGMENT).get(0);
    }

    @Before
    public void init() {
        List<RetargetingListGetItem> listsGetItems = api.userSteps.retargetingListsSteps().get(
                new GetRequestMap()
                        .withFieldNames(
                                RetargetingListFieldEnum.ID,
                                RetargetingListFieldEnum.NAME
                        ));
        api.userSteps.retargetingListsSteps().deleteByIds(
                LOGIN, listsGetItems.stream()
                        .filter(i -> i.getName().equals(NAME))
                        .map(RetargetingListGetItem::getId)
                        .collect(Collectors.toList())
        );
        idList = api.userSteps.retargetingListsSteps().add(new AddRequestMap()
                .withRetargetingLists(
                        new RetargetingListAddItemMap()
                                .withDescription(DESCRIPTION)
                                .withName(NAME)
                                .withRules(new RetargetingListRuleItemMap()
                                        .withOperator(OPERATOR_ENUM)
                                        .withArgumentItems(new RetargetingListRuleArgumentItemMap()
                                                .withMembershipLifeSpan(CORRECT_LIFESPAN)
                                                .withExternalId(goalId))))
        );
        assumeThat("добавился один объект", idList, iterableWithSize(1));
    }

    @Test
    public void testUpdateSuccess() {
        api.userSteps.retargetingListsSteps().shouldGetResultOnUpdate(new UpdateRequestMap()
                        .withRetargetingLists(
                                new RetargetingListUpdateItemMap()
                                        .withId(idList.get(0))
                                        .withDescription(DESCRIPTION)
                                        .withName(NAME)
                                        .withRules(new RetargetingListRuleItemMap()
                                                .withOperator(OPERATOR_ENUM)
                                                .withArgumentItems(new RetargetingListRuleArgumentItemMap()
                                                        .withMembershipLifeSpan(CORRECT_LIFESPAN)
                                                        .withExternalId(goalId)))),
                ExpectedResult.success());
    }

    @Test
    public void testUpdateWithEmptyRetargetingLists() {
        api.userSteps.retargetingListsSteps().expectErrorOnUpdate(
                new UpdateRequestMap(),
                new Api5Error(8000, Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                        path(UpdateRequestMap.RETARGETING_LISTS)));
    }

    @Test
    public void testUpdateWithEmptyName() {
        api.userSteps.retargetingListsSteps().shouldGetResultOnUpdate(new UpdateRequestMap()
                        .withRetargetingLists(
                                new RetargetingListUpdateItemMap()
                                        .withId(idList.get(0))
                                        .withName("")
                        ),
                ExpectedResult.errors(new Notification(5003, Api5ErrorDetailsJava.EMPTY_VALUE,
                        path(RetargetingListUpdateItemMap.NAME))));
    }

    @Test
    public void testUpdateWithoutAnyExternalId() {
        api.userSteps.retargetingListsSteps().shouldGetResultOnUpdate(new UpdateRequestMap()
                        .withRetargetingLists(
                                new RetargetingListUpdateItemMap()
                                        .withId(idList.get(0))
                                        .withRules(new RetargetingListRuleItemMap()
                                                .withOperator(RetargetingListRuleOperatorEnum.ANY)
                                                .withArgumentItems(new RetargetingListRuleArgumentItemMap()
                                                        .withMembershipLifeSpan(CORRECT_LIFESPAN)))),
                ExpectedResult.errors(
                        new Notification(5005,
                                Api5ErrorDetailsJava.FIELD_MUST_BE_POSITIVE_INTEGER,
                                path(RetargetingListUpdateItemMap.RULES,
                                        0,
                                        RetargetingListRuleItemMap.ARGUMENTS,
                                        0,
                                        RetargetingListRuleArgumentItemMap.EXTERNAL_ID)
                        )));
    }

    @Test
    public void testUpdateGoalWithoutLisfespan() {
        api.userSteps.retargetingListsSteps().shouldGetResultOnUpdate(new UpdateRequestMap()
                        .withRetargetingLists(
                                new RetargetingListUpdateItemMap()
                                        .withId(idList.get(0))
                                        .withRules(new RetargetingListRuleItemMap()
                                                .withOperator(RetargetingListRuleOperatorEnum.ANY)
                                                .withArgumentItems(new RetargetingListRuleArgumentItemMap()
                                                        .withExternalId(goalId)))),
                ExpectedResult.errors(
                        new Notification(5000,
                                Api5ErrorDetailsJava.MEMBERSHIP_LIFESPAN_MUST_BE_SET_FOR_GOAL_AND_SEGMENT,
                                path(RetargetingListUpdateItemMap.RULES,
                                        0,
                                        RetargetingListRuleItemMap.ARGUMENTS,
                                        0))));
    }

    @Test
    public void testUpdateSegmentWithoutLisfespan() {
        api.userSteps.retargetingListsSteps().shouldGetResultOnUpdate(new UpdateRequestMap()
                        .withRetargetingLists(
                                new RetargetingListUpdateItemMap()
                                        .withId(idList.get(0))
                                        .withRules(new RetargetingListRuleItemMap()
                                                .withOperator(RetargetingListRuleOperatorEnum.ANY)
                                                .withArgumentItems(new RetargetingListRuleArgumentItemMap()
                                                        .withExternalId(segmentId)))),
                ExpectedResult.success());
    }

    @Test
    public void testUpdateAudienceSegmentWithoutLisfespan() {
        api.userSteps.retargetingListsSteps().shouldGetResultOnUpdate(new UpdateRequestMap()
                        .withRetargetingLists(
                                new RetargetingListUpdateItemMap()
                                        .withId(idList.get(0))
                                        .withRules(new RetargetingListRuleItemMap()
                                                .withOperator(RetargetingListRuleOperatorEnum.ANY)
                                                .withArgumentItems(new RetargetingListRuleArgumentItemMap()
                                                        .withExternalId(audienceSegmentId)))),
                ExpectedResult.success());
    }

    @Test
    public void testtestUpdateWithoutOperator() {
        api.userSteps.retargetingListsSteps().expectErrorOnUpdate(new UpdateRequestMap()
                        .withRetargetingLists(
                                new RetargetingListUpdateItemMap()
                                        .withId(idList.get(0))
                                        .withRules(new RetargetingListRuleItemMap()
                                                .withArgumentItems(new RetargetingListRuleArgumentItemMap()
                                                        .withMembershipLifeSpan(CORRECT_LIFESPAN)
                                                        .withExternalId(goalId)))),
                new Api5Error(8000,
                        Api5ErrorDetailsJava.IN_ELEMENT_OF_ARRAY_MISSING_REQUIRED_FIELD,
                        path(UpdateRequestMap.RETARGETING_LISTS,
                                RetargetingListAddItemMap.RULES),
                        path(RetargetingListRuleItemMap.OPERATOR)));
    }

    @Test
    public void testUpdateWithoutArguments() {
        api.userSteps.retargetingListsSteps().expectErrorOnUpdate(new UpdateRequestMap()
                        .withRetargetingLists(
                                new RetargetingListUpdateItemMap()
                                        .withId(idList.get(0))
                                        .withRules(new RetargetingListRuleItemMap()
                                                .withOperator(RetargetingListRuleOperatorEnum.ANY))),
                new Api5Error(8000,
                        Api5ErrorDetailsJava.IN_ELEMENT_OF_ARRAY_MISSING_REQUIRED_FIELD,
                        path(UpdateRequestMap.RETARGETING_LISTS,
                                RetargetingListAddItemMap.RULES),
                        path(RetargetingListRuleItemMap.ARGUMENTS)));
    }
}
