package ru.yandex.autotests.directapi.sitelinks.add;

import java.util.Arrays;
import java.util.Comparator;
import java.util.List;

import javax.xml.bind.JAXBElement;
import javax.xml.namespace.QName;

import com.yandex.direct.api.v5.sitelinks.SitelinksSetGetItem;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.ProtocolType;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.api5.sitelinks.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.sitelinks.SitelinkMap;
import ru.yandex.autotests.directapi.model.api5.sitelinks.SitelinksSetAddItemMap;
import ru.yandex.autotests.directapi.model.api5.sitelinks.SitelinksSetGetItemMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.sitelinks.SitelinksFeatures;
import ru.yandex.autotests.directapi.sitelinks.SitelinksLogins;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static com.google.common.primitives.Longs.asList;
import static org.hamcrest.Matchers.hasSize;
import static ru.yandex.autotests.directapi.matchers.beans.version5.BeanDifferMatcherV5.beanDifferV5;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;
import static ru.yandex.autotests.irt.testutils.beandiffer.BeanDifferMatcher.beanEquivalent;

/**
 * Created by semkagtn on 1/29/15.
 * https://st.yandex-team.ru/TESTIRT-4100
 */
@Aqua.Test
@Features(SitelinksFeatures.ADD)
@Description("Проверка корректности сохранения полей сайтлинков после их добавления")
@Tag(TagDictionary.TRUNK)
public class AddAndCheckFieldsTest {

    private static final String CLIENT = SitelinksLogins.CLIENT;
    @ClassRule
    public static ApiSteps api = new ApiSteps().as(CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static Long turboPageId;

    @BeforeClass
    public static void createTurboPages() {
        DirectJooqDbSteps jooqSteps = api.userSteps.getDirectJooqDbSteps().useShardForLogin(CLIENT);
        long clientId = jooqSteps.usersSteps().getUser(CLIENT).getClientid();
        turboPageId = jooqSteps.turboLandingsSteps().createDefaultTurbolanding(clientId);
    }

    @AfterClass
    public static void deleteTurboPages() {
        api.userSteps.getDirectJooqDbSteps().useShardForLogin(CLIENT).turboLandingsSteps()
                .deleteTurboLandings(asList(turboPageId));
    }

    @Before
    public void addUnitsToUser() {
        api.protocol(ProtocolType.JSON);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
    }

    @Test
    @Description("Добавить один набор сайтлинков с одним сайтлинком и проверить записанные поля в результате")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3786")
    public void addSitelinksSetWithOneSitelink() {
        SitelinkMap sitelinkMap = new SitelinkMap()
                .randomSitelink();

        List<Long> ids = api.userSteps.sitelinksSteps().add(new AddRequestMap()
                .withSitelinksSets(new SitelinksSetAddItemMap()
                        .withSitelinks(sitelinkMap)));
        assumeThat("добавился один набор сайтлинков", ids, hasSize(1));
        long id = ids.get(0);

        List<SitelinksSetGetItem> actualResult = api.userSteps.sitelinksSteps().get(id);
        assertThat("набор сайтлинков сохранился верно", actualResult, beanEquivalent(Arrays.asList(
                (SitelinksSetGetItem) new SitelinksSetGetItemMap()
                        .withSitelinks(sitelinkMap.asSitelinkGetMap())
                        .withId(id).getBean())));
    }

    @Test
    @Description("Добавить один набор сайтлинков с двумя сайтлинками и проверить записанные поля в результате")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3787")
    public void addSitelinksSetWithTwoSitelinks() {
        SitelinkMap firstSitelinkMap = new SitelinkMap()
                .randomSitelink().withTurboPageId(turboPageId);
        SitelinkMap secondSitelinkMap = new SitelinkMap()
                .randomSitelink();

        List<Long> ids = api.userSteps.sitelinksSteps().add(new AddRequestMap()
                .withSitelinksSets(new SitelinksSetAddItemMap()
                        .withSitelinks(firstSitelinkMap, secondSitelinkMap)));
        assumeThat("добавился один набор сайтлинков", ids, hasSize(1));
        long id = ids.get(0);

        List<SitelinksSetGetItem> actualResult = api.userSteps.sitelinksSteps().getWithTurboPageId(id);
        assertThat("набор сайтлинков сохранился верно", actualResult, beanEquivalent(Arrays.asList(
                (SitelinksSetGetItem) new SitelinksSetGetItemMap()
                        .withSitelinks(firstSitelinkMap.asSitelinkGetMap(), secondSitelinkMap.asSitelinkGetMap())
                        .withId(id).getBean())));
    }

    @Test
    @Description("Добавить два набора сайтлинков и проверить записанные поля в результате")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3788")
    public void addTwoSitelinksSets() {
        SitelinkMap firstSitelinkMap = new SitelinkMap()
                .randomSitelink().withTurboPageId(turboPageId);
        SitelinkMap secondSitelinkMap = new SitelinkMap()
                .randomSitelink();

        List<Long> ids = api.userSteps.sitelinksSteps().add(new AddRequestMap()
                .withSitelinksSets(
                        new SitelinksSetAddItemMap()
                                .withSitelinks(firstSitelinkMap),
                        new SitelinksSetAddItemMap()
                                .withSitelinks(secondSitelinkMap)));
        assumeThat("добавилось два набора сайтлинков", ids, hasSize(2));

        List<SitelinksSetGetItem> actualResult = api.userSteps.sitelinksSteps().getWithTurboPageId(ids);
        List<SitelinksSetGetItem> expected = Arrays.asList(
                (SitelinksSetGetItem) new SitelinksSetGetItemMap()
                        .withSitelinks(firstSitelinkMap.asSitelinkGetMap())
                        .withId(ids.get(0)).getBean(),
                (SitelinksSetGetItem) new SitelinksSetGetItemMap()
                        .withSitelinks(secondSitelinkMap.asSitelinkGetMap())
                        .withId(ids.get(1)).getBean());
        expected.sort(Comparator.comparingLong(SitelinksSetGetItem::getId));

        assertThat("набор сайтлинков сохранился верно", actualResult, beanEquivalent(expected));
    }

    @Test
    @Description("При сохранении сайтлика без href в ответе должен вернуться сайтлик с null в href")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3789")
    public void addSitelinksSetWithOneSitelinkWithoutHref() {
        SitelinkMap sitelinkMap = new SitelinkMap()
                .withRandomTitle()
                .withTurboPageId(turboPageId);

        List<Long> ids = api.userSteps.sitelinksSteps().add(new AddRequestMap()
                .withSitelinksSets(new SitelinksSetAddItemMap()
                        .withSitelinks(sitelinkMap)));
        assumeThat("добавился один набор сайтлинков", ids, hasSize(1));
        long id = ids.get(0);

        List<SitelinksSetGetItem> actualResult = api.userSteps.sitelinksSteps().get(id);
        //Турбо-страницы в ответе не будет, т.к. запрошен стандартный набор полей, поэтому сбрасываем TurboPageId в ожидаемом результате
        assertThat("пустой href вернулся как null", actualResult, beanDifferV5(Arrays.asList(
                (SitelinksSetGetItem) new SitelinksSetGetItemMap()
                        .withSitelinks(sitelinkMap.asSitelinkGetMap()
                                .withHref(null)
                                .withoutTurboPageId()
                        )
                        .withId(id).getBean())));
    }

    @Test
    @Description("При сохранении сайтлика с без href в ответе должен вернуться сайтлик с null в href")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3790")
    public void addSitelinksSetWithOneSitelinkWithoutHrefUsingSOAP() {
        api.protocol(ProtocolType.SOAP);
        SitelinkMap sitelinkMap = new SitelinkMap()
                .withRandomTitle()
                .withTurboPageId(turboPageId);

        List<Long> ids = api.userSteps.sitelinksSteps().add(new AddRequestMap()
                .withSitelinksSets(new SitelinksSetAddItemMap()
                        .withSitelinks(sitelinkMap)));
        assumeThat("добавился один набор сайтлинков", ids, hasSize(1));
        long id = ids.get(0);

        List<SitelinksSetGetItem> actualResult = api.userSteps.sitelinksSteps().get(id);
        //Турбо-страницы в ответе не будет, т.к. запрошен стандартный набор полей, поэтому сбрасываем TurboPageId в ожидаемом результате
        assertThat("пустой href вернулся как null", actualResult, beanDifferV5(Arrays.asList(
                (SitelinksSetGetItem) new SitelinksSetGetItemMap()
                        .withSitelinks(sitelinkMap.asSitelinkGetMap()
                                .withHref(null)
                                .withoutTurboPageId()
                        )
                        .withId(id).getBean())));
    }

    @Test
    @Description("При сохранении сайтлика с пустым href получаем ошибку")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3791")
    public void negativeAddSitelinksSetWithOneSitelinkWithEmptyHref() {
        SitelinkMap sitelinkMap = new SitelinkMap()
                .randomSitelink()
                .withHref("")
                .withTurboPageId(turboPageId);


        api.userSteps.sitelinksSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withSitelinksSets(new SitelinksSetAddItemMap().withSitelinks(sitelinkMap)),
                ExpectedResult.errors(new Notification(5005,
                        Api5ErrorDetailsJava.EMPTY_SITELINK_HREF)));
    }

}
