package ru.yandex.autotests.directapi.sitelinks.add;

import java.util.Arrays;
import java.util.Collection;

import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.api5.sitelinks.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.sitelinks.SitelinkMap;
import ru.yandex.autotests.directapi.model.api5.sitelinks.SitelinksSetAddItemMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.sitelinks.SitelinksFeatures;
import ru.yandex.autotests.directapi.sitelinks.SitelinksLogins;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang3.RandomStringUtils.randomAlphabetic;

/**
 * Created by semkagtn on 2/4/15.
 * https://st.yandex-team.ru/TESTIRT-4100
 */
@Aqua.Test
@Features(SitelinksFeatures.ADD)
@Description("Валидация ссылки в сайтлинках: негативные сценарии")
@Tag(TagDictionary.TRUNK)
@RunWith(Parameterized.class)
public class AddHrefValidationNegativeTest {

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(SitelinksLogins.CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameter(0)
    public String description;

    @Parameterized.Parameter(1)
    public String href;

    @Parameterized.Parameter(2)
    public Notification expectedNotification;

    @Parameterized.Parameters(name = "{0}")
    public static Collection<Object[]> data() {
        String wrongProtocolHref = "ssh://semkagtn.com";
        String onlyNumericHref = "http://1234.56";
        String hrefWithSpecificSymbol = "http://hello!world.ru";
        String lessThanMinDomainLengthHref = "http://" + randomAlphabetic(SitelinkMap.MIN_DOMAIN_LENGTH - 1) + ".ru";
        String greaterThanMaxDomainLengthHref = "http://" + randomAlphabetic(SitelinkMap.MAX_DOMAIN_LENGTH + 1) +
                "." + randomAlphabetic(SitelinkMap.MAX_DOMAIN_LENGTH) + ".ru";
        String lessThanMinTopDomainLengthHref =
                "http://aaaaa." + randomAlphabetic(SitelinkMap.MIN_TOP_DOMAIN_LENGTH - 1);
        String greaterThanMaxTopDomainLengthHref =
                "http://tester." + randomAlphabetic(SitelinkMap.MAX_TOP_DOMAIN_LENGTH + 1);
        String greaterThanMaxLengthHref =
                "http://" + generateValidHrefWithoutProtocolOfLength(SitelinkMap.MAX_HREF_LENGTH - 6);

        return Arrays.asList(new Object[][]{
                {"Создать сайтлинк со ссылкой, протокол которой отличен от http или https", wrongProtocolHref,
                        new Notification(5005, Api5ErrorDetails.INVALID_QUICK_LINK_FORMAT, wrongProtocolHref)},

                {"Создать сайтлинк со ссылкой, состоящей только из цифр", onlyNumericHref,
                        new Notification(5005, Api5ErrorDetails.INVALID_QUICK_LINK_FORMAT, onlyNumericHref)},

                {"Создать сайтлинк со ссылкой, содержащей специфичный символ", hrefWithSpecificSymbol,
                        new Notification(5005, Api5ErrorDetails.INVALID_QUICK_LINK_FORMAT, hrefWithSpecificSymbol)},

                {"Создать сайтлинк со ссылкой с внутренним доменом длиной меньшей, чем " +
                        SitelinkMap.MIN_DOMAIN_LENGTH, lessThanMinDomainLengthHref, new Notification(5005,
                        Api5ErrorDetails.INVALID_QUICK_LINK_FORMAT, lessThanMinDomainLengthHref)},

                {"Создать сайтлинк со ссылкой с внутренними доменами длиной большей, чем " +
                        SitelinkMap.MAX_DOMAIN_LENGTH, greaterThanMaxDomainLengthHref, new Notification(5005,
                        Api5ErrorDetails.INVALID_QUICK_LINK_FORMAT, greaterThanMaxDomainLengthHref)},

                {"Создать сайтлинк со ссылкой с доменом верхнего уровня длиной меньшей, чем" +
                        SitelinkMap.MIN_TOP_DOMAIN_LENGTH, lessThanMinTopDomainLengthHref, new Notification(5005,
                        Api5ErrorDetails.INVALID_QUICK_LINK_FORMAT, lessThanMinTopDomainLengthHref)},

                {"Создать сайтлинк со ссылкой с доменом верхнего уровня длиной большей, чем" +
                        SitelinkMap.MAX_TOP_DOMAIN_LENGTH, greaterThanMaxTopDomainLengthHref, new Notification(5005,
                        Api5ErrorDetails.INVALID_QUICK_LINK_FORMAT, greaterThanMaxTopDomainLengthHref)},

                {"Создать сайтлинк со ссылкой, длиной большей, чем " + SitelinkMap.MAX_HREF_LENGTH,
                        greaterThanMaxLengthHref, new Notification(5001,
                        Api5ErrorDetails.LENGTH_OF_SITELINK_HREF_EXCEEDED, SitelinkMap.MAX_HREF_LENGTH)}
        });
    }

    @Before
    public void addUnitsToUser() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(SitelinksLogins.CLIENT);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3791")
    public void addSitelinkWithSpecificHref() {
        api.userSteps.sitelinksSteps().shouldGetResultOnAdd(new AddRequestMap()
                .withSitelinksSets(new SitelinksSetAddItemMap()
                        .withSitelinks(new SitelinkMap()
                                .withRandomTitle()
                                .withHref(href))),
                ExpectedResult.errors(expectedNotification));
    }

    private static String generateValidHrefWithoutProtocolOfLength(int length) {
        StringBuilder stringBuilder = new StringBuilder(randomAlphabetic(length));
        int domainLength = Math.min(
                (SitelinkMap.MAX_DOMAIN_LENGTH - 1) / 2, (SitelinkMap.MAX_TOP_DOMAIN_LENGTH - 1) / 2);
        for (int i = domainLength; i < length - domainLength; i += domainLength) {
            if (i == domainLength) {
                stringBuilder.replace(i, i + 4, ".ru/");
            } else {
                stringBuilder.setCharAt(i, '/');
            }
        }
        return stringBuilder.toString();
    }
}
