package ru.yandex.autotests.directapi.sitelinks.add;

import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.ApiStories;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.api5.sitelinks.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.sitelinks.SitelinksSetAddItemMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.sitelinks.SitelinksFeatures;
import ru.yandex.autotests.directapi.sitelinks.SitelinksLogins;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.directapi.apiclient.errors.Path.path;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by semkagtn on 2/9/15.
 * https://st.yandex-team.ru/TESTIRT-4100
 */
@Aqua.Test
@Features(SitelinksFeatures.ADD)
@Stories(ApiStories.UNITS)
@Description("Проверка снятия баллов при добавлении сайтлинков")
public class AddUnitsTest {

    private static final String LOGIN = SitelinksLogins.UNITS_ADD_CLIENT;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Before
    public void addUnitsToUser() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
    }

    private static final int CALL_COST = 20;
    private static final int SET_COST = 20;
    private static final int ERROR_COST = 20;
    private static final int COMMON_REQUEST_ERROR_COST = 50;

    @Test
    @Description("Снятие баллов при добавление одного набора сайтлинков")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3834")
    public void addOneSitelinksSet() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.sitelinksSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withSitelinksSets(new SitelinksSetAddItemMap()
                                .defaultSitelinksSet(2)),
                ExpectedResult.success());
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + SET_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Снятие баллов при добавление двух наборов сайтлинков")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3840")
    public void addTwoSitelinksSets() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.sitelinksSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withSitelinksSets(
                                new SitelinksSetAddItemMap()
                                        .defaultSitelinksSet(1),
                                new SitelinksSetAddItemMap()
                                        .defaultSitelinksSet(1)),
                ExpectedResult.success(),
                ExpectedResult.success());
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + 2 * SET_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Один невалидный набор сайтлинков в запросе на добавление")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3835")
    public void oneInvalidSitelinksSet() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.sitelinksSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withSitelinksSets(new SitelinksSetAddItemMap()
                                .defaultSitelinksSet(SitelinksSetAddItemMap.MAX_SITELINKS_AMOUNT + 1)),
                ExpectedResult.errors(
                        new Notification(7000, Api5ErrorDetails.NUMBER_OF_QUICK_LINKS_SHOULD_BE_FROM_TO,
                                SitelinksSetAddItemMap.MIN_SITELINKS_AMOUNT, SitelinksSetAddItemMap.MAX_SITELINKS_AMOUNT)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + ERROR_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Два невалидных набора сайтлинков в запросе на добавление")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3836")
    public void twoInvalidSitelinksSets() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.sitelinksSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withSitelinksSets(
                                new SitelinksSetAddItemMap()
                                        .defaultSitelinksSet(SitelinksSetAddItemMap.MAX_SITELINKS_AMOUNT + 1),
                                new SitelinksSetAddItemMap()
                                        .defaultSitelinksSet(SitelinksSetAddItemMap.MAX_SITELINKS_AMOUNT + 1)),
                ExpectedResult.errors(
                        new Notification(7000, Api5ErrorDetails.NUMBER_OF_QUICK_LINKS_SHOULD_BE_FROM_TO,
                                SitelinksSetAddItemMap.MIN_SITELINKS_AMOUNT, SitelinksSetAddItemMap.MAX_SITELINKS_AMOUNT)),
                ExpectedResult.errors(
                        new Notification(7000, Api5ErrorDetails.NUMBER_OF_QUICK_LINKS_SHOULD_BE_FROM_TO,
                                SitelinksSetAddItemMap.MIN_SITELINKS_AMOUNT, SitelinksSetAddItemMap.MAX_SITELINKS_AMOUNT)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + 2 * ERROR_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Один валидный и один невалидный набор сайтлинков в запросе на добавление")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3837")
    public void oneValidAndOneInvalidSitelinksSets() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.sitelinksSteps().shouldGetResultOnAdd(new AddRequestMap()
                .withSitelinksSets(
                        new SitelinksSetAddItemMap()
                                .defaultSitelinksSet(1),
                        new SitelinksSetAddItemMap()
                                .defaultSitelinksSet(SitelinksSetAddItemMap.MAX_SITELINKS_AMOUNT + 1)),
                ExpectedResult.success(),
                ExpectedResult.errors(
                        new Notification(7000, Api5ErrorDetails.NUMBER_OF_QUICK_LINKS_SHOULD_BE_FROM_TO,
                                SitelinksSetAddItemMap.MIN_SITELINKS_AMOUNT, SitelinksSetAddItemMap.MAX_SITELINKS_AMOUNT)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + SET_COST + ERROR_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Некорректный запрос на добавление сайтлинков")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3838")
    public void invalidRequest() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.sitelinksSteps().expectErrorOnAdd(new AddRequestMap()
                .withSitelinksSets(null),
                new Api5Error(8000, Api5ErrorDetailsJava.FIELD_MUST_CONTAIN_NOT_LESS_ELEMENTS,
                        path(AddRequestMap.SITELINKS_SETS), 1));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = COMMON_REQUEST_ERROR_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Добавление наборов сайтлинков при отсутствии баллов у клиента")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3839")
    public void requestWhenClientHasZeroUnits() {
        int units = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.clientFakeSteps().fakeWithdrawClientUnits(LOGIN, units);
        units = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        assumeThat("у клиента нету баллов", units, equalTo(0));

        api.userSteps.sitelinksSteps().expectErrorOnAdd(new AddRequestMap()
                .withSitelinksSets(new SitelinksSetAddItemMap()
                        .defaultSitelinksSet(1)),
                new Api5Error(152, Api5ErrorDetails.NOT_ENOUGH_UNITS_FOR_OPERATION));
    }
}
