package ru.yandex.autotests.directapi.sitelinks.get;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import com.yandex.direct.api.v5.sitelinks.SitelinkFieldEnum;
import com.yandex.direct.api.v5.sitelinks.SitelinkGetItem;
import com.yandex.direct.api.v5.sitelinks.SitelinksSetFieldEnum;
import com.yandex.direct.api.v5.sitelinks.SitelinksSetGetItem;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.general.IdsCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.sitelinks.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.sitelinks.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.sitelinks.SitelinkGetMap;
import ru.yandex.autotests.directapi.model.api5.sitelinks.SitelinkMap;
import ru.yandex.autotests.directapi.model.api5.sitelinks.SitelinksSetAddItemMap;
import ru.yandex.autotests.directapi.model.api5.sitelinks.SitelinksSetGetItemMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.sitelinks.SitelinksFeatures;
import ru.yandex.autotests.directapi.sitelinks.SitelinksLogins;
import ru.yandex.autotests.irt.testutils.beandiffer.matchvariation.DefaultMatchVariation;
import ru.yandex.autotests.irt.testutils.beandiffer.matchvariation.MatchVariation;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static com.google.common.primitives.Longs.asList;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.core.IsNull.nullValue;
import static ru.yandex.autotests.directapi.apiclient.errors.Path.path;
import static ru.yandex.autotests.directapi.matchers.beans.version5.BeanDifferMatcherV5.beanDifferV5;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;
import static ru.yandex.autotests.irt.testutils.beandiffer.BeanDifferMatcher.beanDiffer;

/**
 * Created by semkagtn on 2/4/15.
 * https://st.yandex-team.ru/TESTIRT-4099
 */
@Aqua.Test
@Features(SitelinksFeatures.GET)
@Description("Проверка получения различных полей для наборов сайтлинков")
@Tag(TagDictionary.TRUNK)
public class GetFieldNamesTest {
    private static String client = SitelinksLogins.CLIENT;
    private static Long turboPageId;
    private static SitelinkMap addedSitelink;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(client);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static long id;

    @BeforeClass
    public static void createSitelinksSet() {
        DirectJooqDbSteps jooqSteps = api.userSteps.getDirectJooqDbSteps().useShardForLogin(client);
        long clientId = jooqSteps.usersSteps().getUser(client).getClientid();
        turboPageId = jooqSteps.turboLandingsSteps().createDefaultTurbolanding(clientId);
        addedSitelink = new SitelinkMap()
                .randomSitelink()
                .withTurboPageId(turboPageId);

        List<Long> ids = api.userSteps.sitelinksSteps().add(new AddRequestMap()
                .withSitelinksSets(new SitelinksSetAddItemMap()
                        .withSitelinks(addedSitelink)));
        assumeThat("добавился один набор сайтлинков", ids, hasSize(1));
        id = ids.get(0);
    }

    @AfterClass
    public static void deleteTurboPages() {
        api.userSteps.getDirectJooqDbSteps().useShardForLogin(client).turboLandingsSteps()
                .deleteTurboLandings(asList(turboPageId));
    }

    @Before
    public void addUnitsToUser() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
    }

    @Test
    @Description("Запрос на получение только идентификаторов наборов сайтлинков")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3875")
    public void onlyIdInFieldNames() {
        List<SitelinksSetGetItem> result = api.userSteps.sitelinksSteps().get(new GetRequestMap()
                .withFieldNames(SitelinksSetFieldEnum.ID)
                .withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(id)));
        assertThat("вернулись только идентификаторы", result, beanDiffer(Arrays.asList(
                (SitelinksSetGetItem) new SitelinksSetGetItemMap()
                        .withId(id)
                        .withSitelinks(null).getBean())));
    }

    @Test
    @Description("Запрос на получение наборов сайтлинков без идентификаторов")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3876")
    public void onlySitelinksInFieldNames() {
        List<SitelinksSetGetItem> result = api.userSteps.sitelinksSteps().get(new GetRequestMap()
                .withFieldNames(SitelinksSetFieldEnum.SITELINKS)
                .withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(id)));

        MatchVariation variation = new DefaultMatchVariation()
                .forClasses(SitelinkGetItem.class).useMatcher(notNullValue());

        assertThat("вернулись сайтлинки без идентификаторов", result, beanDiffer(Arrays.asList(
                (SitelinksSetGetItem) new SitelinksSetGetItemMap()
                        .withSitelinks(new SitelinkGetMap())
                        .withId(null).getBean())).withVariation(variation));
    }

    @Test
    @Description("Запрос на получение наборов сайтлинков со всеми полями")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3877")
    public void idAndSitelinksInFieldNames() {
        List<SitelinksSetGetItem> result = api.userSteps.sitelinksSteps().get(new GetRequestMap()
                .withFieldNames(SitelinksSetFieldEnum.ID)
                .withSitelinkFieldNames(SitelinkFieldEnum.values())
                .withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(id)));

        MatchVariation variation = new DefaultMatchVariation()
                .forClasses(SitelinkGetItem.class).useMatcher(notNullValue());

        assertThat("вернулись все поля", result, beanDiffer(Arrays.asList(
                (SitelinksSetGetItem) new SitelinksSetGetItemMap()
                        .withSitelinks(new SitelinkGetMap())
                        .withId(id).getBean())).withVariation(variation));
    }

    @Test
    @Description("Запрос на получение наборов сайтлинков с невалидным значением в FieldNames")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3878")
    public void invalidFieldNameInRequest() {
        String invalidFieldName = "SomeInvalidValue";
        api.userSteps.sitelinksSteps().expectErrorOnGet(new GetRequestMap()
                        .withFieldNames(invalidFieldName),
                new Api5Error(8000, Api5ErrorDetailsJava.ELEMENT_OF_ARRAY_CONTAINS_INCORRECT_ENUMERATE_VALUE,
                        path(GetRequestMap.FIELD_NAMES),
                        Api5Error.enumAsParam(SitelinksSetFieldEnum.class)
                ));
    }

    @Test
    @Description("Запрос на получение наборов сайтлинков - по умолчанию не возвращается TurboPageId")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3879")
    public void getWithoutTurboPageid() {
        List<SitelinksSetGetItem> result = api.userSteps.sitelinksSteps().get(new GetRequestMap()
                .withFieldNames(SitelinksSetFieldEnum.ID, SitelinksSetFieldEnum.SITELINKS)
                .withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(id)));

        assertThat("в ответе нет TurboPageId", result.get(0).getSitelinks().get(0).getTurboPageId(), nullValue());
    }

    @Test
    @Description("Запрос на получение наборов сайтлинков - при явном запросе возвращается TurboPageId")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3880")
    public void getWithTurboPageid() {
        List<SitelinksSetGetItem> result = api.userSteps.sitelinksSteps().get(new GetRequestMap()
                .withFieldNames(SitelinksSetFieldEnum.ID)
                .withSitelinkFieldNames(SitelinkFieldEnum.TURBO_PAGE_ID)
                .withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(id)));
        Long storedTurboPageId = result.get(0).getSitelinks().get(0).getTurboPageId().getValue();

        assertThat("в ответе присутствует сохраненный TurboPageId", storedTurboPageId, equalTo(turboPageId));
    }

    @Test
    @Description("Запрос на получение наборов сайтлинков - запрашиваем только SitelinkFieldNames")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3881")
    public void getOnlySitelinkFieldNames() {
        List<SitelinksSetGetItem> result = api.userSteps.sitelinksSteps().get(new GetRequestMap()
                .withSitelinkFieldNames(SitelinkFieldEnum.values())
                .withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(id)));
        List<SitelinksSetGetItem> expectedResult = Collections.singletonList((SitelinksSetGetItem) new SitelinksSetGetItemMap()
                .withSitelinks(addedSitelink.asSitelinkGetMap()).getBean());
        assertThat("в ответе вернулись все поля сайтлинка", result,
                beanDifferV5(expectedResult));
    }
}
