package ru.yandex.autotests.directapi.sitelinks.get;

import com.yandex.direct.api.v5.sitelinks.GetResponse;
import com.yandex.direct.api.v5.sitelinks.SitelinksSetFieldEnum;
import com.yandex.direct.api.v5.sitelinks.SitelinksSetGetItem;
import org.junit.*;
import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.general.IdsCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.general.LimitOffsetMap;
import ru.yandex.autotests.directapi.model.api5.sitelinks.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.sitelinks.SitelinksSetGetItemMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.sitelinks.SitelinksFeatures;
import ru.yandex.autotests.directapi.sitelinks.SitelinksLogins;
import ru.yandex.autotests.irt.testutils.allure.TestSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import java.util.Arrays;
import java.util.List;

import static ch.lambdaj.Lambda.on;
import static ch.lambdaj.Lambda.sort;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;
import static ru.yandex.autotests.irt.testutils.beandiffer.BeanDifferMatcher.beanEquivalent;

/**
 * Created by semkagtn on 2/3/15.
 * https://st.yandex-team.ru/TESTIRT-4099
 */
@Aqua.Test
@Features(SitelinksFeatures.GET)
@Description("Проверка получения сайтлинков клиента")
@Tag(TagDictionary.TRUNK)
public class GetTest {

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(SitelinksLogins.CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private GetRequestMap requestMap;

    private static long firstId;
    private static long secondId;
    private static long thirdId;

    @BeforeClass
    public static void createSitelinksSets() {
        firstId = api.userSteps.sitelinksSteps().addDefaultSet(1);
        secondId = api.userSteps.sitelinksSteps().addDefaultSet(1);
        thirdId = api.userSteps.sitelinksSteps().addDefaultSet(1);
    }

    @Before
    public void initRequestMap() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(SitelinksLogins.CLIENT);
        requestMap = new GetRequestMap()
                .withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(firstId, secondId, thirdId))
                .withFieldNames(SitelinksSetFieldEnum.ID);
    }

    @Test
    @Description("Получение сайтлинков без использования дополнительных параметров")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3904")
    public void getSitelinksSets() {
        List<SitelinksSetGetItem> actualResult = api.userSteps.sitelinksSteps().get(requestMap);
        actualResult = sort(actualResult, on(SitelinksSetGetItem.class).getId());

        assertThat("вернулись предполагаемые наборы сайтлинков", actualResult, beanEquivalent(Arrays.asList(
                (SitelinksSetGetItem) new SitelinksSetGetItemMap()
                        .withId(firstId).getBean(),
                (SitelinksSetGetItem) new SitelinksSetGetItemMap()
                        .withId(secondId).getBean(),
                (SitelinksSetGetItem) new SitelinksSetGetItemMap()
                        .withId(thirdId).getBean())));
    }

    @Test
    @Description("Получение сайтлинков с использованием лимита")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3905")
    public void getSitelinksSetsUsingLimit() {
        List<SitelinksSetGetItem> resultWithoutLimit = api.userSteps.sitelinksSteps().get(requestMap);
        assumeThat("вернулось три набора сайтлинков", resultWithoutLimit, hasSize(3));

        List<SitelinksSetGetItem> actualResult = api.userSteps.sitelinksSteps().get(requestMap
                .withPage(new LimitOffsetMap()
                        .withLimit(2l)));

        assertThat("вернулись предполагаемые наборы сайтлинков", actualResult, beanEquivalent(Arrays.asList(
                resultWithoutLimit.get(0), resultWithoutLimit.get(1))));
    }

    @Test
    @Description("Получение сайтлинков с использованием смещения")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3906")
    public void getSitelinksSetsUsingOffset() {
        List<SitelinksSetGetItem> resultWithoutOffset = api.userSteps.sitelinksSteps().get(requestMap);
        assumeThat("вернулось три набора сайтлинков", resultWithoutOffset, hasSize(3));

        List<SitelinksSetGetItem> actualResult = api.userSteps.sitelinksSteps().get(requestMap
                .withPage(new LimitOffsetMap()
                        .withOffset(1l)));

        assertThat("вернулись предполагаемые наборы сайтлинков", actualResult, beanEquivalent(Arrays.asList(
                resultWithoutOffset.get(1), resultWithoutOffset.get(2))));
    }

    @Test
    public void checkLimitedByIfLimitIsWorked() {
        GetResponse response = api.userSteps.sitelinksSteps().sitelinksGet(requestMap
                .withPage(new LimitOffsetMap()
                        .withLimit(1l)));
        TestSteps.assertThat("значение LimitedBy верное", response.getLimitedBy(), equalTo(1l));
    }

    @Test
    public void checkLimitedByIfLimitIsWorkedWithOffset() {
        //DIRECT-44575
        GetResponse response = api.userSteps.sitelinksSteps().sitelinksGet(requestMap
                .withPage(new LimitOffsetMap()
                        .withLimit(1l)
                        .withOffset(1l)));
        TestSteps.assertThat("значение LimitedBy верное", response.getLimitedBy(), equalTo(2l));
    }
}
