package ru.yandex.autotests.directapi.vcards.add;

import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.ApiStories;
import ru.yandex.autotests.directapi.VCardsFeatures;
import ru.yandex.autotests.directapi.VCardsLogins;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.vcards.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.vcards.VCardAddMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by chicos on 12.12.14.
 */
@Aqua.Test
@Features(VCardsFeatures.ADD)
@Stories(ApiStories.UNITS)
public class AddVCardsUnitsTest {
    private static LogSteps log = LogSteps.getLogger(AddVCardsUnitsTest.class);

    private static final String singleClient = VCardsLogins.CLIENT_UNITS_ADD;

    private static final String agency = VCardsLogins.AGENCY_UNITS;
    private static final String subclient = VCardsLogins.SUBCLIENT_UNITS;

    private static final int ADD_COST = 20;
    private static final int ITEM_COST_ON_ADD = 20;
    private static final int ADD_FAULT_COST = 20;
    private static final int GENERAL_FAULT_COST = 50;

    @ClassRule
    public static ApiSteps api = new ApiSteps();

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trashman = new Trashman(api);

    @BeforeClass
    public static void prepareClient() {
        //reset client units
        api.as(Logins.SUPER_LOGIN).userSteps.clientFakeSteps().fakeClearClientSpentUnits(singleClient);
        api.as(Logins.SUPER_LOGIN).userSteps.clientFakeSteps().fakeClearClientSpentUnits(agency);
        api.as(Logins.SUPER_LOGIN).userSteps.clientFakeSteps().fakeClearClientSpentUnits(subclient);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3956")
    public void addSingleVCardTest() {
        Long campaignID = api.as(singleClient).userSteps.campaignSteps().addDefaultTextCampaign();
        VCardAddMap item = new VCardAddMap().shortContacts().withCampaignId(campaignID);

        int unitsBefore = api.as(Logins.SUPER_LOGIN).userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        api.as(singleClient).userSteps.vCardsSteps().vCardsAdd(item);
        int unitsAfter = api.as(Logins.SUPER_LOGIN).userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);

        assertThat("стоимость добавления одной визитки", (unitsBefore - unitsAfter),
                equalTo(ADD_COST + ITEM_COST_ON_ADD));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3957")
    public void addSeveralVCardsTest() {
        Long campaignID = api.as(singleClient).userSteps.campaignSteps().addDefaultTextCampaign();
        VCardAddMap item1 = new VCardAddMap().shortContacts().withCampaignId(campaignID).withCompanyName("Компания 1");
        VCardAddMap item2 = new VCardAddMap().shortContacts().withCampaignId(campaignID).withCompanyName("Компания 2");

        int unitsBefore = api.as(Logins.SUPER_LOGIN).userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        api.as(singleClient).userSteps.vCardsSteps().vCardsAdd(item1, item2);
        int unitsAfter = api.as(Logins.SUPER_LOGIN).userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);

        assertThat("стоимость добавления двух визиток", (unitsBefore - unitsAfter),
                equalTo(ADD_COST + 2 * ITEM_COST_ON_ADD));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3958")
    public void addInvalidVCardItemTest() {
        //https://st.yandex-team.ru/DIRECT-39224
        int unitsBefore = api.as(Logins.SUPER_LOGIN).userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        api.as(singleClient).userSteps.vCardsSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap()
                        .withVCards(new VCardAddMap()
                                .shortContacts()
                                .withCampaignId(12345)),
                ExpectedResult.errorWithCode(8800));
        int unitsAfter = api.as(Logins.SUPER_LOGIN).userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);

        assertThat("стоимость ошибки при добавлении визитки", (unitsBefore - unitsAfter),
                equalTo(ADD_COST + ADD_FAULT_COST));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3959")
    public void addFewVCardsOneInvalidItemTest() {
        //https://st.yandex-team.ru/DIRECT-39224
        Long campaignID = api.as(singleClient).userSteps.campaignSteps().addDefaultTextCampaign();
        VCardAddMap item = new VCardAddMap().shortContacts().withCampaignId(campaignID).withCompanyName("Компания 1");

        int unitsBefore = api.as(Logins.SUPER_LOGIN).userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        api.as(singleClient).userSteps.vCardsSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap()
                        .withVCards(item, new VCardAddMap()
                                .shortContacts()
                                .withCampaignId(12345)),
                ExpectedResult.success(),
                ExpectedResult.errorWithCode(8800));
        int unitsAfter = api.as(Logins.SUPER_LOGIN).userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);

        assertThat("стоимость ошибки при добавлении нескольких визиток", (unitsBefore - unitsAfter),
                equalTo(ADD_COST + ITEM_COST_ON_ADD + ADD_FAULT_COST));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3960")
    public void addVCardUnitsCostByAgencyTest() {
        //https://st.yandex-team.ru/DIRECT-39224
        Long campaignID = api.as(agency).userSteps.campaignSteps().addDefaultTextCampaign(subclient);
        VCardAddMap item1 = new VCardAddMap().shortContacts().withCampaignId(campaignID).withCompanyName("Компания 1");
        VCardAddMap item2 = new VCardAddMap().shortContacts().withCampaignId(campaignID).withCompanyName("Компания 2");

        int agencyUnitsBefore = api.as(Logins.SUPER_LOGIN).userSteps.clientFakeSteps().fakeClientUnitsBalance(agency);
        int scUnitsBefore = api.as(Logins.SUPER_LOGIN).userSteps.clientFakeSteps().fakeClientUnitsBalance(subclient);
        api.as(agency).userSteps.vCardsSteps().vCardsAdd(subclient, item1, item2);
        int scUnitsAfter = api.as(Logins.SUPER_LOGIN).userSteps.clientFakeSteps().fakeClientUnitsBalance(subclient);
        int agencyUunitsAfter = api.as(Logins.SUPER_LOGIN).userSteps.clientFakeSteps().fakeClientUnitsBalance(agency);

        assertThat("списаные юниты у агентства", (agencyUnitsBefore - agencyUunitsAfter), equalTo(0));
        assertThat("списаные юниты у субклиента", (scUnitsBefore - scUnitsAfter),
                equalTo(ADD_COST + 2 * ITEM_COST_ON_ADD));
    }

    @Test
    @Description("Списание баллов у агентства при вызове метода без указания логина субклиента")
    @ru.yandex.qatools.allure.annotations.TestCaseId("3961")
    public void unknownClientUnitsCostByAgencyTest() {
        Long campaignID = api.as(agency).userSteps.campaignSteps().addDefaultTextCampaign(subclient);

        int agencyUnitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(agency);
        api.userSteps.vCardsSteps().shouldGetVCardErrorOn(
                Action.ADD,
                new AddRequestMap()
                        .withVCards(new VCardAddMap()
                                .shortContacts()
                                .withCampaignId(campaignID)),
                new Api5Error(8000, Api5ErrorDetailsJava.NOT_CLIENT_IN_CLIENT_LOGIN));
        int agencyUnitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(agency);

        assertThat("списание баллов у агентства", agencyUnitsBefore - agencyUnitsAfter, equalTo(GENERAL_FAULT_COST));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3962")
    public void noRequiredFieldTest() {
        int unitsBefore = api.as(Logins.SUPER_LOGIN).userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);
        api.as(singleClient).userSteps.vCardsSteps().shouldGetVCardErrorWithErrorCodeOn(
                Action.ADD,
                new AddRequestMap()
                        .withVCards(new VCardAddMap()),
                8000);
        int unitsAfter = api.as(Logins.SUPER_LOGIN).userSteps.clientFakeSteps().fakeClientUnitsBalance(singleClient);

        assertThat("стоимость ошибки при добавлении визитки", (unitsBefore - unitsAfter), equalTo(GENERAL_FAULT_COST));
    }
}
