package ru.yandex.autotests.directapi.vcards.get;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;

import com.yandex.direct.api.v5.vcards.VCardFieldEnum;
import com.yandex.direct.api.v5.vcards.VCardGetItem;
import org.apache.commons.lang3.StringUtils;
import org.junit.AfterClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.ReflectionUtils;
import ru.yandex.autotests.direct.utils.converter.EnumToStringValueConverter;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.VCardsFeatures;
import ru.yandex.autotests.directapi.VCardsLogins;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.vcards.MapPointMap;
import ru.yandex.autotests.directapi.model.api5.vcards.VCardAddMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static ch.lambdaj.Lambda.convert;
import static org.hamcrest.Matchers.containsInAnyOrder;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by chicos on 17.07.14.
 */
@Aqua.Test
@Features(VCardsFeatures.GET)
@Tag(TagDictionary.TRUNK)
@RunWith(Parameterized.class)
public class GetVCardsFieldNamesTest {
    private static LogSteps log = LogSteps.getLogger(GetVCardsFieldNamesTest.class);

    private static String login = VCardsLogins.SINGLE_CLIENT_2;

    private static Long campaignId;
    private static long firstCardId;
    private static long secondCardId;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(login);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trashman = new Trashman(api);

    @Parameterized.Parameter(value = 0)
    public String description;

    @Parameterized.Parameter(value = 1)
    public Long[] vCardIds;

    @Parameterized.Parameter(value = 2)
    public List<VCardFieldEnum> fieldNames;

    @Parameterized.Parameters(name = "name = {0}")
    public static Collection fieldNames() {
        prepareVCards();
        Object[][] data = new Object[][]{
                {"Обязательное составное поле", new Long[]{firstCardId}, Arrays.asList(VCardFieldEnum.PHONE)},
                {"Опциональное простое поле, заданное", new Long[]{secondCardId}, Arrays.asList(VCardFieldEnum.COMPANY_NAME)},
                {"Опциональное простое поле, не заданное", new Long[]{firstCardId}, Arrays.asList(VCardFieldEnum.METRO_STATION_ID)},
                {"Опциональное составное поле, заданное", new Long[]{firstCardId}, Arrays.asList(VCardFieldEnum.POINT_ON_MAP)},
                {"Опциональное составное поле, не заданное", new Long[]{secondCardId}, Arrays.asList(VCardFieldEnum.POINT_ON_MAP)},
                {"Повторяющееся опциональное поле", new Long[]{firstCardId}, Arrays.asList(VCardFieldEnum.STREET, VCardFieldEnum.STREET)},
                {"Обязательные и опциональные поля", new Long[]{firstCardId}, Arrays.asList(
                        VCardFieldEnum.CAMPAIGN_ID,
                        VCardFieldEnum.COUNTRY,
                        VCardFieldEnum.PHONE,
                        VCardFieldEnum.COMPANY_NAME,
                        VCardFieldEnum.CONTACT_PERSON)},
                {"Несколько ID, обязательное составное поле", new Long[]{secondCardId, firstCardId}, Arrays.asList(VCardFieldEnum.PHONE)},
                {"Обязательные поля", new Long[]{secondCardId}, Arrays.asList(
                        VCardFieldEnum.CAMPAIGN_ID,
                        VCardFieldEnum.COUNTRY,
                        VCardFieldEnum.CITY,
                        VCardFieldEnum.WORK_TIME,
                        VCardFieldEnum.PHONE)},
                {"Несколько ID, обязательные поля", new Long[]{firstCardId, secondCardId}, Arrays.asList(
                        VCardFieldEnum.CAMPAIGN_ID,
                        VCardFieldEnum.COUNTRY,
                        VCardFieldEnum.CITY,
                        VCardFieldEnum.WORK_TIME,
                        VCardFieldEnum.PHONE)}
        };
        return Arrays.asList(data);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("4012")
    public void getVCardFieldNamesTest() {
        List<VCardGetItem> vCards = api.as(login).userSteps.vCardsSteps().vCardsGet(fieldNames, vCardIds);

        log.info("Проверим набор визиток на наличие запрошенных полей");
        for (VCardGetItem item : vCards){
            checkFieldNames(item, fieldNames);
        }
    }

    @AfterClass
    public static void putCampaignToRemove(){
        //workaround для корректроного создания кампаний в параметризованном тесте
        api.userSteps.campaignSteps().putCampaignToQueue(login, campaignId);
    }

    private static void prepareVCards() {
        log.info("Создадим кампанию и визитки");
        campaignId = api.as(login).userSteps.campaignSteps().addDefaultTextCampaign(login);

        //workaround для создания кампаний в параметризованном тесте
        api.userSteps.campaignSteps().removeCampaignsFromQueue(Arrays.asList(campaignId));

        VCardAddMap item = new VCardAddMap().defaultContacts().withCampaignId(campaignId)
                .withPointOnMap(new MapPointMap().defaultPointOnMap())
                .withCompanyName("Кампания для визитки 1");
        firstCardId = api.as(login).userSteps.vCardsSteps().vCardsAdd(item).get(0);

        item.withCompanyName("Кампания для визитки 2");
        secondCardId = api.as(login).userSteps.vCardsSteps().vCardsAdd(item).get(0);
    }

    private void checkFieldNames(VCardGetItem item, List<VCardFieldEnum> fields) {
        List<String> gotFileds = getNonNullFieldsNames(item);

        //exclude field duplicates
        HashSet<VCardFieldEnum> fieldsSet = new HashSet<>(fields);
        List<VCardFieldEnum> nonDuplicatedFields = new ArrayList<>(fieldsSet);

        String[] expectedFields = convert(new ArrayList<>(nonDuplicatedFields), new EnumToStringValueConverter())
                .toArray(new String[nonDuplicatedFields.size()]);

        assertThat("вернулся объект с корректным набором запрошенных полей", gotFileds, containsInAnyOrder(expectedFields));
    }


    private List<String> getNonNullFieldsNames(Object object) {
        if (object == null) {
            throw new IllegalArgumentException("Необходимо передать объект для извлечения имен инициализированных полей");
        }
        log.info("Извлечем имена инициализированных полей объекта");
        List<String> nonNullFileds = new ArrayList<>();
        for (Field field : object.getClass().getDeclaredFields()) {
            if (ReflectionUtils.invokeGetter(object, StringUtils.capitalize(field.getName())) != null) {
                nonNullFileds.add(StringUtils.capitalize(field.getName()));
            }
        }
        return nonNullFileds;
    }
}
