package ru.yandex.autotests.directapi.vcards.get;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import com.yandex.direct.api.v5.vcards.GetResponse;
import com.yandex.direct.api.v5.vcards.VCardGetItem;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.VCardsFeatures;
import ru.yandex.autotests.directapi.VCardsLogins;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.general.IdsCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.general.LimitOffsetMap;
import ru.yandex.autotests.directapi.model.api5.vcards.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.vcards.VCardAddMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static ch.lambdaj.Lambda.extract;
import static ch.lambdaj.Lambda.on;
import static org.hamcrest.Matchers.containsInAnyOrder;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.notNullValue;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by chicos on 17.07.14.
 */
@Aqua.Test
@Features(VCardsFeatures.GET)
@Tag(TagDictionary.TRUNK)
@RunWith(Parameterized.class)
public class GetVCardsLimitOffsetTest {
    private static LogSteps log = LogSteps.getLogger(GetVCardsLimitOffsetTest.class);

    private static String login = VCardsLogins.SINGLE_CLIENT_2;

    private static Long campaignId;
    private static long firstCardId;
    private static long secondCardId;
    private static long thirdCardId;
    private static long fourthCardId;
    private static long fifthCardId;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(login);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trashman = new Trashman(api);

    @Parameterized.Parameter(value = 0)
    public Long limit;

    @Parameterized.Parameter(value = 1)
    public Long offset;

    @Parameterized.Parameter(value = 2)
    public Long[] vCardIds;

    @Parameterized.Parameter(value = 3)
    public Long[] expectedIds;

    @Parameterized.Parameter(value = 4)
    public Long expectedLimitedBy;

    private GetResponse getResponse;

    @Parameterized.Parameters(name = "limit = {0}, offset = {1}")
    public static Collection limitsAndOffsets() {
        prepareVCards();
        Long[] defaultVCardIds = new Long[]{firstCardId, secondCardId, thirdCardId, fourthCardId, fifthCardId};
        Object[][] data = new Object[][]{
                {null, null,
                        defaultVCardIds,
                        defaultVCardIds, null},
                {7L, null,
                        defaultVCardIds,
                        defaultVCardIds, null},
                {4L, null,
                        defaultVCardIds,
                        new Long[]{firstCardId, secondCardId, thirdCardId, fourthCardId}, 4L},
                {null, 2L,
                        defaultVCardIds,
                        new Long[]{thirdCardId, fourthCardId, fifthCardId}, null},
                {null, 6L,
                        defaultVCardIds,
                        new Long[0], null},
                {null, 5L,
                        defaultVCardIds,
                        new Long[0], null},
                {null, 0L,
                        defaultVCardIds,
                        defaultVCardIds, null},
                {3L, 1L,
                        defaultVCardIds,
                        new Long[]{secondCardId, thirdCardId, fourthCardId}, 4L},
                {1L, 5L,
                        defaultVCardIds,
                        new Long[0], null},
                {4L, 3L,
                        defaultVCardIds,
                        new Long[]{fourthCardId, fifthCardId}, null},
                {4L, 2L,
                        new Long[]{firstCardId, secondCardId, thirdCardId, fifthCardId, fourthCardId},
                        new Long[]{thirdCardId, fifthCardId, fourthCardId}, null}
        };
        return Arrays.asList(data);
    }

    @Before
    public void getVCards() {
        getResponse = api.userSteps.vCardsSteps().get(
                new GetRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap().withIds(firstCardId, secondCardId, thirdCardId, fourthCardId, fifthCardId))
                        .withAllFieldNames()
                        .withPage(new LimitOffsetMap()
                                .withLimit(limit)
                                .withOffset(offset)));
        assumeThat("получили результат запроса визиток", getResponse.getVCards(), notNullValue());
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("4014")
    public void getVCardLimitOffsetTest() {
        //https://st.yandex-team.ru/DIRECT-34422
        List<VCardGetItem> vCards = api.as(login).userSteps.vCardsSteps().vCardsGet(
                new LimitOffsetMap()
                        .withLimit(limit)
                        .withOffset(offset),
                vCardIds
        );

        assertThat("вернулся корректный набор визиток",
                extract(getResponse.getVCards(), on(VCardGetItem.class).getId()),
                containsInAnyOrder(expectedIds)
        );
    }

    @Test
    public void limitedByTest() {
        assertThat("значение параметра LimitedBy", getResponse.getLimitedBy(), equalTo(expectedLimitedBy));
    }

    @AfterClass
    public static void putCampaingToRemove() {
        //workaround для корректроного создания кампаний в параметризованном тесте
        api.userSteps.campaignSteps().putCampaignToQueue(login, campaignId);
    }

    private static void prepareVCards() {
        log.info("Создадим кампании и визитки");
        campaignId = api.as(login).userSteps.campaignSteps().addDefaultTextCampaign(login);

        //workaround для создания кампаний в параметризованном тесте
        api.userSteps.campaignSteps().removeCampaignsFromQueue(Arrays.asList(campaignId));

        VCardAddMap item = new VCardAddMap().defaultContacts().withCampaignId(campaignId)
                .withCompanyName("Кампания для визитки 1");
        firstCardId = api.as(login).userSteps.vCardsSteps().vCardsAdd(item).get(0);

        item.withCompanyName("Кампания для визитки 2");
        secondCardId = api.as(login).userSteps.vCardsSteps().vCardsAdd(item).get(0);

        item.withCompanyName("Кампания для визитки 3");
        thirdCardId = api.as(login).userSteps.vCardsSteps().vCardsAdd(item).get(0);

        item.withCompanyName("Кампания для визитки 4");
        fourthCardId = api.as(login).userSteps.vCardsSteps().vCardsAdd(item).get(0);

        item.withCompanyName("Кампания для визитки 5");
        fifthCardId = api.as(login).userSteps.vCardsSteps().vCardsAdd(item).get(0);
    }
}
