package ru.yandex.autotests.directintapi.bstransport;

import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import com.google.gson.reflect.TypeToken;
import org.apache.commons.lang.StringUtils;
import org.hamcrest.Matcher;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.junit.ComparisonFailure;

import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersMobileContentPrimaryAction;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.AdgroupsMobileContentRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BannerImagesFormatsRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BannersMobileContentRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BsExportQueueRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.CampaignsMobileContentRecord;
import ru.yandex.autotests.direct.utils.model.RegionIDValues;
import ru.yandex.autotests.directapi.darkside.model.Status;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.ExpressionCoefItem;
import ru.yandex.autotests.directapi.darkside.model.bslogs.mobile.MobileContentData;
import ru.yandex.autotests.directapi.darkside.model.bslogs.mobile.OSVersions;
import ru.yandex.autotests.directapi.darkside.model.multipliers.DirectAtom;
import ru.yandex.autotests.directapi.enums.AgeGroup;
import ru.yandex.autotests.directapi.enums.DeviceTargeting;
import ru.yandex.autotests.directapi.enums.Gender;
import ru.yandex.autotests.directapi.enums.NetworkTargeting;
import ru.yandex.autotests.directapi.enums.ReflectedAttributes;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.irt.testutils.beandiffer2.comparestrategy.defaultcomparestrategy.DefaultCompareStrategies;

import static org.hamcrest.CoreMatchers.equalTo;
import static org.junit.Assert.assertEquals;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;
import static ru.yandex.autotests.irt.testutils.beandiffer2.BeanDifferMatcher.beanDiffer;

/**
 * Вспомогательные методы тестов транспорта
 * <p/>
 * User: mexicano
 * Date: 25.08.14
 */
public class TransportTestUtils {
    private TransportTestUtils() {}

    private static final Type COLLECTION_TYPE = new TypeToken<List<MobileContentData>>() {
    }.getType();
    private static final DateTimeFormatter formatter = DateTimeFormat.forPattern("yyyy-MM-dd HH:mm:ss");
    public static final String DEFAULT_TRACKING_LINK
            = "https://play.google.com/store/apps/details?id=ru.yandex.searchplugin";


    public static Matcher<Map<String, Object>> beanDifferForBsExportQueue(BsExportQueueRecord bsExportQueueRecord) {
        return beanDiffer(bsExportQueueRecord.intoMap()).useCompareStrategy(DefaultCompareStrategies.onlyExpectedFields());
    }

    public static BsExportQueueRecord getDefaultBsExportQueueRecord(long cid) {
        return new BsExportQueueRecord()
                .setCid(cid)
                .setIsFullExport(0);
    }

    public static <T> List<T> getMapValues(Map<?, T> map) {
        if (map == null) {
            return null;
        }
        return new ArrayList<>(map.values());
    }

    public static String randomDomain() {
        return org.apache.commons.lang3.RandomStringUtils.randomAlphabetic(5).toLowerCase()
                + System.currentTimeMillis() + ".com";
    }

    public static Map<String, Map> getDefaultSocdemCoef() {
        Map<String, Map> socdemCoef = new HashMap<>();
        Map<String, Integer> unknown = new HashMap<>();
        Map<String, Integer> male = new HashMap<>();
        Map<String, Integer> female = new HashMap<>();
        Arrays.asList(AgeGroup.values()).forEach(ageGroup -> unknown.put(ageGroup.toString(), null));
        Arrays.asList(AgeGroup.values()).forEach(ageGroup -> female.put(ageGroup.toString(), null));
        Arrays.asList(AgeGroup.values()).forEach(ageGroup -> male.put(ageGroup.toString(), null));
        female.put("unknown", null);
        male.put("unknown", null);
        socdemCoef.put(Gender.FEMALE.toString(), female);
        socdemCoef.put(Gender.MALE.toString(), male);
        socdemCoef.put("unknown", unknown);
        return socdemCoef;
    }

    public static void checkBsUnifiedMultipliers(List<ExpressionCoefItem> coefs, List<Object> expectedCoefs,
                                                 List<List<List<DirectAtom>>> expectedExpressions) {
        assumeThat("Количество ожидаемых значений совпадает с количеством проверяемых объектов",
                coefs.size(), equalTo(expectedCoefs.size()));
        assumeThat("Количество ожидаемых значений совпадает с количеством проверяемых объектов",
                coefs.size(), equalTo(expectedExpressions.size()));

        int exprectedCount = expectedCoefs.size();
        assumeThat("SocdemCoefs содержит ожидаемое количество корректировок",
                coefs.size(), equalTo(exprectedCount));

        for (int i = 0; i < coefs.size(); i++) {
            assumeThat("Значение Coef соответствует ожидаемому значению",
                    coefs.get(i).getCoef(), equalTo(expectedCoefs.get(i)));

            assertThat("Выражение корректировки соответсвует ожиданиям",
                    coefs.get(i).getExpression(), beanDiffer(expectedExpressions.get(i)));
        }

    }

    public static CampaignsMobileContentRecord getDefaultCampaignsMobileContent(int cid) {
        CampaignsMobileContentRecord mobCamp = new CampaignsMobileContentRecord()
                .setCid((long) cid)
                .setDeviceTypeTargeting(DeviceTargeting.PHONE.toString())
                .setIsInstalledApp(0)
                .setNetworkTargeting(NetworkTargeting.CELL.toString());
        return mobCamp;
    }

    public static CampaignsMobileContentRecord getDefaultCampaignsMobileContent(Long cid) {
        return getDefaultCampaignsMobileContent(cid.intValue());
    }

    public static AdgroupsMobileContentRecord getDefaultAdgroupsMobileContent(Long pid, long mobileContentId) {
        AdgroupsMobileContentRecord adgrMob = new AdgroupsMobileContentRecord()
                .setPid(pid)
                .setMobileContentId(mobileContentId)
                .setMinOsVersion(OSVersions.IOS_VERSIONS[1])
                .setNetworkTargeting(NetworkTargeting.WIFI.toString())
                .setDeviceTypeTargeting(DeviceTargeting.PHONE.toString())
                .setStoreContentHref(DEFAULT_TRACKING_LINK);
        return adgrMob;
    }

    @Deprecated
    public static AdgroupsMobileContentRecord getDefaultAdgroupsMobileContent(Integer pid, long mobileContentId) {
        return getDefaultAdgroupsMobileContent(Long.valueOf(pid), mobileContentId);
    }

    public static BannersMobileContentRecord getDefaultBannersMobileContent(Long bid) {
        BannersMobileContentRecord banMob = new BannersMobileContentRecord()
                .setBid(bid)
                .setPrimaryAction(BannersMobileContentPrimaryAction.download)
                .setReflectedAttrs(ReflectedAttributes.ICON.toString());
        return banMob;
    }

    @Deprecated
    public static BannersMobileContentRecord getDefaultBannersMobileContent(Integer bid) {
        return getDefaultBannersMobileContent(Long.valueOf(bid));
    }

    // Преобразование списка регионов в строку вида 1,2,3
    public static String regionListToString(List<RegionIDValues> regionsList) {
        return regionListToString(regionsList, Collections.EMPTY_LIST);
    }

    // Преобразование списка плюс-регионов и минус-регионов в строку вида 1,2,-3
    public static String regionListToString(List<RegionIDValues> regionsList, List<RegionIDValues> minusRegionsList) {
        List regionStrList = new ArrayList<String>();
        regionStrList.addAll(regionsList.stream().map(s -> String.valueOf(s.getId())).collect(Collectors.toList()));
        regionStrList.addAll(minusRegionsList.stream().map(s -> "-" + String.valueOf(s.getId())).collect(Collectors.toList()));
        return StringUtils.join(regionStrList, ",");
    }

    public static String generateImageUrl(BannerImagesFormatsRecord formatsRecord) {
        return generateImageUrl(formatsRecord, "orig");
    }

    public static String generateImageUrl(BannerImagesFormatsRecord formatsRecord, String formatId) {
        String host = formatsRecord.getAvatarsHost().getLiteral();
        String namespace = formatsRecord.getNamespace().getLiteral();
        return String.format("https://%s/get-%s/%s/%s/%s", host, namespace, formatsRecord.getMdsGroupId(), formatsRecord.getImageHash(), formatId);
    }

    /**
     * assert сравнения Float с точностью до 1e-6, считающий, что null == null
     */
    public static void assertNullsOrEquals(String message, Float expected, Float actual) {
        if (actual == null && expected == null) {return;}
        if (actual == null || expected == null) {
            String cleanMessage = message == null ? "" : message;
            throw new ComparisonFailure(cleanMessage, Objects.toString(expected),
                    Objects.toString(actual));
        }
        assertEquals(expected, actual, 1e-6f);
    }

    /**
     * Вычисляет статус кампании, который должен быть установлен на ней после того,
     * как статус модерации единственного баннера будет установлен в bannerStatusModerate
     * Возвращает null, если статус кампании менять не нужно
     */
    public static String calcCampaignStatusModerate(String bannerStatusModerate) {
        if (bannerStatusModerate.equals(Status.NO)) {
            return Status.NO;
        }
        if (bannerStatusModerate.equals(Status.YES)) {
            return Status.YES;
        }
        // в других случаях не меняем
        return null;
    }

    /**
     * Устанавливает статус модерации баннера с последующим изменением статуса модерации кампании
     * (раньше это пересчитывалось при вызове ручки FakeBannerParams, теперь нужно задавать явно)
     */
    public static void setBannerStatuses(ApiSteps api, long bid, String bannerStatusModerate,
                                         String bannerStatusPostModerate, long cid) {
        api.userSteps.bannersFakeSteps().setStatusModerate(bid, bannerStatusModerate);
        api.userSteps.bannersFakeSteps().setStatusPostModerate(bid, bannerStatusPostModerate);
        String campaignStatusModerate = calcCampaignStatusModerate(bannerStatusModerate);
        if (campaignStatusModerate != null) {
            api.userSteps.campaignFakeSteps().setStatusModerate(cid, campaignStatusModerate);
        }
    }
    public static boolean getDoNotSendExtraBannerImageFormatToBsProperty(ApiSteps api) {
        String dontSendExtraBannerImageFormatsRaw = api.userSteps.getDarkSideSteps().getDirectJooqDbSteps().ppcPropertiesSteps()
                .getPropertyValue("DO_NOT_SEND_EXTRA_BANNER_IMAGE_FORMATS_TO_BS");
        boolean dontSendExtraBannerImageFormats = false;
        if (dontSendExtraBannerImageFormatsRaw != null) {
            dontSendExtraBannerImageFormats = Long.parseLong(dontSendExtraBannerImageFormatsRaw) == 1;
        }
        return dontSendExtraBannerImageFormats;
    }
}
