package ru.yandex.autotests.directintapi.bstransport.b2b;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.junit.Rule;
import org.junit.Test;
import org.junit.internal.AssumptionViolatedException;
import org.junit.rules.TestWatcher;
import org.junit.runner.Description;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.direct.utils.config.DirectTestRunProperties;
import ru.yandex.autotests.directapi.darkside.exceptions.DarkSideException;
import ru.yandex.autotests.directapi.darkside.model.RunBsTransportScriptResponse;
import ru.yandex.autotests.directapi.darkside.steps.DarkSideSteps;
import ru.yandex.autotests.directapi.darkside.steps.ScriptParamsProfiles;
import ru.yandex.autotests.directapi.darkside.steps.TransportSteps;
import ru.yandex.autotests.directintapi.bstransport.b2b.utils.TransportB2BBeanMongoHelper;
import ru.yandex.autotests.directintapi.bstransport.b2b.utils.TransportB2BMongoBean;
import ru.yandex.autotests.directintapi.bstransport.b2b.utils.TransportB2BProperties;
import ru.yandex.autotests.directintapi.bstransport.b2b.utils.TransportMethodEnum;
import ru.yandex.autotests.directintapi.bstransport.b2b.utils.UpdateDataBeanDiffer;
import ru.yandex.autotests.irt.testutils.allure.AssumptionException;
import ru.yandex.autotests.irt.testutils.beandiffer2.beanfield.BeanFieldPath;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.allure.annotations.Title;

import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.greaterThanOrEqualTo;
import static org.hamcrest.Matchers.iterableWithSize;
import static org.hamcrest.Matchers.notNullValue;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * https://st.yandex-team.ru/TESTIRT-10501
 * https://st.yandex-team.ru/TESTIRT-10504
 */
@RunWith(Parameterized.class)
public abstract class BaseBsTransportB2BTest {

    private static final TransportB2BBeanMongoHelper mongoHelper = new TransportB2BBeanMongoHelper();

    /**
     * Настройки b2b-тестов
     */
    protected static TransportB2BProperties properties;

    /**
     * Степы для работы с проверяемой средой
     */
    private static DarkSideSteps darkSideSteps;

    /**
     * Степы для работы с эталонной средой
     */
    private static DarkSideSteps etalonDarkSideSteps;

    /**
     * DB-степы для работы с базой, общей для двух сред
     */
    private static DirectJooqDbSteps jooqSteps;
    /**
     * Игнорируемые поля
     */
    private static List<BeanFieldPath> ignoredFields = new ArrayList<>();

    static {
        init();
    }

    @Rule
    public TestWatcher watcher = new TestWatcher() {
        @Override
        protected void succeeded(Description description) {
            if (campaignBean.isLastFailed()) {
                mongoHelper.setLastFailedForCampaigns(false, campaignBean);
            }
            super.succeeded(description);
        }

        @Override
        protected void failed(Throwable e, Description description) {
            if (!campaignBean.isLastFailed()) {
                mongoHelper.setLastFailedForCampaigns(true, campaignBean);
            }
            super.failed(e, description);
        }

        @Override
        protected void skipped(AssumptionViolatedException e, Description description) {
            if (!campaignBean.isLastFailed()) {
                mongoHelper.setLastFailedForCampaigns(true, campaignBean);
            }
            super.skipped(e, description);
        }
    };


    @Parameterized.Parameter
    public TransportB2BMongoBean campaignBean;

    private static void init() {
        properties = new TransportB2BProperties();

        DirectTestRunProperties etalonStageProperties = DirectTestRunProperties
                .newInstance()
                .withDirectStage(properties.getEtalonStageType());
        etalonDarkSideSteps = new DarkSideSteps(etalonStageProperties);

        DirectTestRunProperties testStageProperties = DirectTestRunProperties
                .newInstance()
                .withDirectStage(properties.getTestStageType());
        darkSideSteps = new DarkSideSteps(testStageProperties);

        etalonDarkSideSteps.getTransportSteps()
                .setCheckFullExport(properties.getEtalonMethod() == TransportMethodEnum.LB);
        darkSideSteps.getTransportSteps()
                .setCheckFullExport(properties.getTestMethod() == TransportMethodEnum.LB);

        jooqSteps = etalonDarkSideSteps.getDirectJooqDbSteps();

        checkDatabaseIdentity();
        initIgnoredFields();
    }

    protected static Collection<Object[]> buildTestParameters(Class testClass) {
        return buildTestParameters(testClass, properties.getCampaignsQuantity());
    }

    protected static Collection<Object[]> buildTestParameters(Class testClass, int campaignQuantity) {
        List<TransportB2BMongoBean> campaignBeans = mongoHelper.getCampaingsForTestClass(testClass);
        if (properties.isFailedOnly()) {
            campaignBeans = campaignBeans
                    .stream()
                    .filter(TransportB2BMongoBean::isLastFailed)
                    .collect(Collectors.toList());
        } else {
            //Проверяем, что набрано достаточно кампаний для запуска тестов только, если не идет перезапуск упавших
            try {
                assumeThat("В монге нашлось нужноe количество кампаний(" + campaignQuantity
                                + ") " + "для теста", campaignBeans,
                        iterableWithSize(greaterThanOrEqualTo(campaignQuantity)));
            } catch (AssumptionException e) {
                mongoHelper.setLastFailedForCampaigns(true, campaignBeans.toArray(new TransportB2BMongoBean[]{}));
                throw e;
            }
        }
        return campaignBeans
                .stream()
                .limit(campaignQuantity)
                .map(transportB2BMongoBean -> new Object[]{transportB2BMongoBean})
                .collect(Collectors.toList());
    }

    @Test
    @Title("Сравнение логов отправки в БК с 2х сред")
    public void campaignB2BTest() {
        Map expectedResult = sendCampaignToBSAndGetLog(
                etalonDarkSideSteps,
                "эталонной среде: " + etalonDarkSideSteps.getHost());
        Map actualResult = sendCampaignToBSAndGetLog(
                darkSideSteps,
                "тестовом стенде: " + darkSideSteps.getHost());

        // Проверяем что лог вообще есть.
        // Без этой проверки тест будет зеленым, если, например, с обоих стендов получили пустые ответы
        assertThat("Не получили лог эталонной среды", expectedResult.values(), contains(notNullValue()));
        assertThat("Не получили лог тестового стенда", actualResult.values(), contains(notNullValue()));

//         Что делать когда тест падает - смотри
//         https://wiki.yandex-team.ru/testirovanie/functesting/direkt/automatization/transport/b2b
//         /#chtodelatkogdatestypadajut
        assertThat("Данные транспорта на тестовом стенде отличаются от эталонного",
                actualResult, new UpdateDataBeanDiffer(expectedResult, ignoredFields.toArray(new BeanFieldPath[]{})));
    }

    @SuppressWarnings("unused")
    @Step("Запускаем весь цикл транспорта на {1}")
    private Map sendCampaignToBSAndGetLog(DarkSideSteps darkSideSteps, String stageDescription) {
        TransportSteps transportSteps = darkSideSteps.getTransportSteps();
        darkSideSteps.getBsExportSteps().addCampaignToFullExportQueue(campaignBean.getCid());
        RunBsTransportScriptResponse response = transportSteps.runBsClientDataInDebugMode(campaignBean.getShard(),
                campaignBean.getCid());
        Map<String, Map> rawLogEntriesForCampaign = transportSteps
                .getRawClientDataLogEntriesForCampaignWithOmittedMdsMeta(response, campaignBean.getCid());
        return buildLogEntriesWithIterations(response, rawLogEntriesForCampaign);
    }

    private static void checkDatabaseIdentity() {
        String productionDBHost = etalonDarkSideSteps.getDirectConfigurationSteps().getDBInfo().getHost();
        String actualDBHost = darkSideSteps.getDirectConfigurationSteps().getDBInfo().getHost();
        if (!productionDBHost.equals(actualDBHost)) {
            throw new DarkSideException(
                    String.format("Сравниваемые стенды смотрят на разные базы (эталонная: %s, проверяемая: %s)",
                            productionDBHost, actualDBHost));
        }
    }

    private static void initIgnoredFields() {
        ignoredFields.addAll(Arrays.asList(
                BeanFieldPath.newPath(".+", "request", "ORDER", ".+", "QueueSetTime"),
                //Для LB response не существует(что выгрузили - то выгрузили), решили на него не смотреть,
                //так как в его проверке есть смысл только для интеграционных B2B, но в ближайшей переспективе из
                BeanFieldPath.newPath(".+", "response"),
                BeanFieldPath.newPath(".+", "request", "ORDER", ".+", "AutoBudgetRestartTime"),
                BeanFieldPath.newPath(".+", "request", "ORDER", ".+", "AutoBudgetSoftRestartTime")
        ));


        String ignoreList = properties.getB2bIgnoreFieldsList();
        if (ignoreList != null) {
            for (String ignoreItem : ignoreList.replaceAll(" ", "").split(",")) {
                ignoredFields.add(BeanFieldPath.newPath(ignoreItem.split("/")));
            }

        }
    }

    private static Map<String, Map> buildLogEntriesWithIterations(RunBsTransportScriptResponse resp,
                                                                  Map<String, Map> rawLogEntries) {
        Map<String, Map> logEntriesWithIterations = new LinkedHashMap<>();
        for (int i = 0; i < resp.getUuid().length; i++) {
            String uuid = resp.getUuid()[i];
            logEntriesWithIterations.put("request " + (i + 1), rawLogEntries.get(uuid));
        }
        return logEntriesWithIterations;
    }
}

