package ru.yandex.autotests.directintapi.bstransport.main.banner.parameters.permalink;

import java.util.concurrent.atomic.AtomicLong;

import org.apache.commons.lang.math.RandomUtils;
import org.apache.commons.lang3.StringUtils;
import org.junit.After;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannerPermalinksPermalinkAssignType;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.OrganizationsStatusPublish;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.ClientPhonesRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.OrganizationsRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.directapi.darkside.Logins;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.darkside.model.RunBsTransportScriptResponse;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.Banner;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.Campaign;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.Context;
import ru.yandex.autotests.directapi.darkside.steps.TransportSteps;
import ru.yandex.autotests.directapi.model.api5.ads.AdAddItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdAddMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directintapi.bstransport.FeatureNames;
import ru.yandex.autotests.directintapi.bstransport.StoriesNames;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.allure.annotations.Title;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static com.google.common.base.Preconditions.checkState;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.Matchers.nullValue;
import static org.junit.Assert.assertThat;
import static ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannerPermalinksPermalinkAssignType.auto;
import static ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannerPermalinksPermalinkAssignType.manual;
import static ru.yandex.autotests.direct.db.models.jooq.ppc.enums.OrganizationsStatusPublish.published;
import static ru.yandex.autotests.direct.db.models.jooq.ppc.enums.OrganizationsStatusPublish.unpublished;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

@Aqua.Test
@Issue("https://st.yandex-team.ru/DIRECT-118565")
@Title("Транспорт: проверка отправки Banner.PermalinkAdPhone")
@Stories(StoriesNames.BANNER_PARAMS_PERMALINK)
@Features(FeatureNames.BANNERS)
public class BsTransportOfPermalinkAdPhoneTextAdTest {
    private static final String login = Logins.LOGIN_TRANSPORT;
    private static long clientId;
    private static TransportSteps transportSteps;
    private static int shard;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(login);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();
    private static DirectJooqDbSteps jooqDbSteps;

    private long cid;
    private long pid;
    private long bid;
    private Long permalink;
    private Long clientPhoneId;
    private String phone;
    private Context context;

    @BeforeClass
    public static void beforeClass() {
        shard = api.userSteps.clientFakeSteps().getUserShard(login);
        clientId = Long.parseLong(api.userSteps.clientFakeSteps().getClientData(login).getClientID());
        transportSteps = api.userSteps.getDarkSideSteps().getTransportSteps();
        jooqDbSteps = api.userSteps.getDirectJooqDbSteps().useShard(shard);
    }

    @Before
    public void before() {
        cid = api.userSteps.campaignSteps().addDefaultTextCampaign();
        pid = api.userSteps.adGroupsSteps().addDefaultGroup(cid);
        long keywordId = api.userSteps.keywordsSteps().addDefaultKeyword(pid);
        api.userSteps.groupFakeSteps().makeGroupFullyModerated(pid);
        api.userSteps.phrasesFakeSteps().makeKeywordModerated(keywordId);
        TextAdAddMap textAdAddMap = new TextAdAddMap().defaultTextAd()
                .withHref("http://ya.ru");
        AdAddItemMap adAddItemMap = new AdAddItemMap()
                .withAdGroupId(pid)
                .withTextAd(textAdAddMap);
        bid = api.userSteps.adsSteps().addAd(adAddItemMap);
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(bid);

        clientPhoneId = (long) RandomUtils.nextInt();
        phone = buildUniquePhoneWithExt();
        ClientPhonesRecord record = new ClientPhonesRecord()
                .setClientPhoneId(clientPhoneId)
                .setPhone(phone)
                .setClientid(clientId);
        jooqDbSteps.bannerAdditionsSteps().addClientPhone(record);
    }

    @After
    @Step("Утилизация тестовых данных")
    public void after() {
        jooqDbSteps.bannerAdditionsSteps().deleteClientPhone(clientPhoneId);
    }

    @Title("Ручной опубликованный пермалинк")
    @Test
    public void permalinkAdPhone_permalinkAdPhoneSentToBs() {
        permalink = assignPermalinkOrChain(manual, published, false);
        context = sendToBsAndGetContext();

        assertThat("Группа должна быть отправлена в БК", context, notNullValue());
        Banner banner = context.getBanner(bid);
        assumeThat("баннер должен быть отправлен в БК", banner, notNullValue());
        assertThat("баннер должен иметь корректный Permalink", banner.getPermalink(), equalTo(permalink));
        assertThat("баннер должен иметь пустой PermalinkChainId", banner.getChainIds(), empty());
        assertThat("баннер должен иметь корректный PermalinkAdPhone", banner.getPermalinkAdPhone(), equalTo(phone));
    }

    @Title("Корректно отправляем номер подменника из Телефонии")
    @Test
    public void permalinkAdPhone_whenTelephonyPhoneIsSet_sendItToBs() {
        // В DIRECT-125269 изменили схему хранения номера Телефонии. Если он задан, лежит в telephony_phone
        Long clientPhoneId = (long) RandomUtils.nextInt();
        String userPhone = buildUniquePhone();
        String telephonyPhone = buildUniquePhone();
        ClientPhonesRecord record =
                new ClientPhonesRecord()
                        .setClientPhoneId(clientPhoneId)
                        .setClientid(clientId)
                        .setPhone(userPhone)
                        .setTelephonyPhone(telephonyPhone);
        jooqDbSteps.bannerAdditionsSteps().addClientPhone(record);

        permalink = assignPermalinkOrChain(manual, published, false, clientPhoneId);
        context = sendToBsAndGetContext();

        assertThat("Группа должна быть отправлена в БК", context, notNullValue());
        Banner banner = context.getBanner(bid);
        assumeThat("баннер должен быть отправлен в БК", banner, notNullValue());
        assertThat("баннер должен иметь корректный PermalinkAdPhone", banner.getPermalinkAdPhone(),
                equalTo(telephonyPhone));
    }

    @Title("Ручной опубликованный пермалинк без подменного телефона")
    @Test
    public void withoutPermalinkAdPhone_permalinkAdPhoneNotSentToBs() {
        permalink = assignPermalinkOrChain(manual, published, false, null);
        context = sendToBsAndGetContext();

        assertThat("Группа должна быть отправлена в БК", context, notNullValue());
        Banner banner = context.getBanner(bid);
        assumeThat("баннер должен быть отправлен в БК", banner, notNullValue());
        assertThat("баннер должен иметь корректный Permalink", banner.getPermalink(), equalTo(permalink));
        assertThat("баннер должен иметь пустой PermalinkChainId", banner.getChainIds(), empty());
        assertThat("баннер должен иметь пустой PermalinkAdPhone", banner.getPermalinkAdPhone(), nullValue());
    }

    @Title("Ручной неопубликованный пермалинк")
    @Test
    public void permalinkAdPhone_notPublished_permalinkAdPhoneNotSentToBs() {
        permalink = assignPermalinkOrChain(manual, unpublished, false);
        context = sendToBsAndGetContext();

        assertThat("Группа должна быть отправлена в БК", context, notNullValue());
        Banner banner = context.getBanner(bid);
        assumeThat("баннер должен быть отправлен в БК", banner, notNullValue());
        assertThat("баннер должен иметь пустой Permalink", banner.getPermalink(), nullValue());
        assertThat("баннер должен иметь пустой PermalinkChainId", banner.getChainIds(), empty());
        assertThat("баннер должен иметь пустой PermalinkAdPhone", banner.getPermalinkAdPhone(), nullValue());
    }

    @Title("Автопривязанный опубликованный пермалинк")
    @Test
    public void permalinkAdPhone_autoAssignedType_permalinkAdPhoneNotSentToBs() {
        permalink = assignPermalinkOrChain(auto, published, false);
        context = sendToBsAndGetContext();

        assertThat("Группа должна быть отправлена в БК", context, notNullValue());
        Banner banner = context.getBanner(bid);
        assumeThat("баннер должен быть отправлен в БК", banner, notNullValue());
        assertThat("баннер должен иметь корректный Permalink", banner.getPermalink(), equalTo(permalink));
        assertThat("баннер должен иметь пустой PermalinkChainId", banner.getChainIds(), empty());
        assertThat("баннер должен иметь пустой PermalinkAdPhone", banner.getPermalinkAdPhone(), nullValue());
    }

    @Title("Ручная опубликованная сеть")
    @Test
    public void permalinkAdPhone_permalinkChain_permalinkAdPhoneNotSentToBs() {
        permalink = assignPermalinkOrChain(manual, published, true);
        context = sendToBsAndGetContext();

        assertThat("Группа должна быть отправлена в БК", context, notNullValue());
        Banner banner = context.getBanner(bid);
        assumeThat("баннер должен быть отправлен в БК", banner, notNullValue());
        assertThat("баннер должен иметь пустой Permalink", banner.getPermalink(), nullValue());
        assertThat("баннер должен иметь корректный PermalinkChainId", banner.getChainIds(), contains(permalink));
        assertThat("баннер должен иметь пустой PermalinkAdPhone", banner.getPermalinkAdPhone(), nullValue());
    }

    /**
     * Отправка в БК и получение контекста
     */
    private Context sendToBsAndGetContext() {
        api.userSteps.campaignFakeSteps().makeNewCampaignReadyForSendingToBS(cid);
        RunBsTransportScriptResponse resp = transportSteps.sendNewCampaign(shard, cid);
        Campaign campaign = transportSteps.getClientDataRequestCampaign(resp, 0, cid);
        assumeThat("новая кампания отправлена в БК", campaign, notNullValue());
        return campaign.getContext(pid);
    }

    private Long assignPermalinkOrChain(
            BannerPermalinksPermalinkAssignType permalinkType,
            OrganizationsStatusPublish pubStatus,
            boolean addChain
    ) {
        return assignPermalinkOrChain(permalinkType, pubStatus, addChain, clientPhoneId);
    }

    private Long assignPermalinkOrChain(
            BannerPermalinksPermalinkAssignType permalinkType,
            OrganizationsStatusPublish pubStatus,
            boolean addChain,
            Long clientPhoneId
    ) {
        Long permalink = addChain ? 0 : RandomUtils.nextLong();
        Long chainId = addChain ? RandomUtils.nextLong() : null;
        jooqDbSteps.bannerAdditionsSteps().setPermalink(bid, permalink, chainId, permalinkType, false);

        if (clientPhoneId != null) {
            jooqDbSteps.bannerAdditionsSteps().setPhone(bid, clientPhoneId);
        }

        jooqDbSteps.bannerAdditionsSteps()
                .addOrUpdateOrganization(new OrganizationsRecord()
                        .setPermalinkId(permalink)
                        .setStatusPublish(pubStatus)
                        .setClientid(clientId));

        return permalink != 0 ? permalink : chainId;
    }

    private static String buildUniquePhone() {
        return buildUniquePhone(false);
    }

    private static String buildUniquePhoneWithExt() {
        return buildUniquePhone(true);
    }

    private static String buildUniquePhone(boolean withExt) {
        String phone = ClientPhoneGenerator.getUniqPhoneE164();
        // значение добавочного номера выбрано произвольно
        String ext = withExt ? "123" : "";
        return phone.substring(0, 2) + "#"
                + phone.substring(2, 5) + "#"
                + phone.substring(5, 8) + "-" + phone.substring(8, 10) + "-" + phone.substring(10, 12) + "#"
                + ext;
    }

}

class ClientPhoneGenerator {
    // Чтобы минимизировать коллизии между запусками, инициализируем счётчик псевдослучайным числом
    private static final AtomicLong COUNTER = new AtomicLong(System.currentTimeMillis() % 1_000_000L);

    /**
     * Генерирует строку с номером в формате E164. Без добавочных
     * <p>
     * Копия
     * <a href="https://a.yandex-team.ru/arc/trunk/arcadia/direct/core/src/test/java/ru/yandex/direct/core/entity/clientphone/ClientPhoneTestUtils.java?blame=true&rev=6902776#L18">ClientPhoneTestUtils#getUniqPhone</a>
     */
    static String getUniqPhoneE164() {
        String prefix = "+79";
        int digitsLeft = 9;
        String nextNumber = Long.toString(COUNTER.incrementAndGet());
        checkState(nextNumber.length() < digitsLeft);
        return prefix + StringUtils.leftPad(nextNumber, digitsLeft, '0');
    }
}
