package ru.yandex.autotests.directintapi.bstransport.main.banner.parameters.sitelinks;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.darkside.Logins;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.darkside.model.RunBsTransportScriptResponse;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.Banner;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.Campaign;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.Context;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.Sitelink;
import ru.yandex.autotests.directapi.enums.AdGroupType;
import ru.yandex.autotests.directapi.enums.CampaignType;
import ru.yandex.autotests.directapi.model.api5.ads.AdAddItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AdUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.DynamicTextAdAddMap;
import ru.yandex.autotests.directapi.model.api5.ads.DynamicTextAdUpdateMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdAddMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdUpdateMap;
import ru.yandex.autotests.directapi.model.api5.ads.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.api5.sitelinks.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.sitelinks.SitelinkMap;
import ru.yandex.autotests.directapi.model.api5.sitelinks.SitelinksSetAddItemMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directintapi.bstransport.FeatureNames;
import ru.yandex.autotests.directintapi.bstransport.StoriesNames;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.allure.annotations.Title;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.containsInAnyOrder;
import static org.hamcrest.Matchers.hasSize;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

@Aqua.Test
@Tag(TagDictionary.RELEASE)
@Title("Транспорт: проверка отправки ссылки в сайтлинках при наличии шаблона для utm-параметров на уровне группы или кампании")
@Stories(StoriesNames.BANNER_ACTION_CREATION)
@Features(FeatureNames.BANNERS)
@RunWith(Parameterized.class)
public class BsTransportOfSitelinksWithHrefParamsTest {
    private static final String LOGIN = Logins.LOGIN_TRANSPORT;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Parameterized.Parameter(0)
    public String paramsOnCampaign;

    @Parameterized.Parameter(1)
    public CampaignType campaignType;

    @Parameterized.Parameter(2)
    public AdGroupType adGroupType;

    @Parameterized.Parameter(3)
    public String expectedParams;

    private static int shard;

    @BeforeClass
    public static void beforeClass() {
        shard = api.userSteps.clientFakeSteps().getUserShard(LOGIN);
    }

    private Long cid;
    private Long pid;
    private Long bid;

    private String firstHref;
    private String secondHref;

    @Parameterized.Parameters(name = "Params on campaign: {0}, campaign type:: {1}")
    public static Iterable<Object[]> data() {
        return Arrays.asList(new Object[][]{
                {null, CampaignType.TEXT, AdGroupType.BASE, null},
                {"utm_param=val", CampaignType.TEXT, AdGroupType.BASE, "utm_param=val"},
                {"utm_param=val", CampaignType.DYNAMIC, AdGroupType.DYNAMIC, null},
                {null, CampaignType.DYNAMIC, AdGroupType.DYNAMIC, null}
        });
    }

    @Before
    @Step("Подготовка данных для теста")
    public void prepareData() {
        cid = api.userSteps.campaignSteps().addDefaultCampaign(campaignType);
        pid = api.userSteps.adGroupsSteps().addDefaultGroup(cid, adGroupType);
        api.userSteps.getDirectJooqDbSteps()
                .useShard(shard)
                .campaignsSteps()
                .updateHrefParams(cid, paramsOnCampaign);
        if (campaignType == CampaignType.TEXT) {
            Long bidsId = api.userSteps.keywordsSteps().addDefaultKeyword(pid);
            api.userSteps.phrasesFakeSteps().makeKeywordModerated(bidsId);
        }
        if (campaignType == CampaignType.DYNAMIC) {
            api.userSteps.dynamicTextAdTargetsSteps().addDefaultWebpage(pid);
        }

        SitelinkMap firstSitelinkMap = new SitelinkMap().randomSitelinkWithDescription();
        SitelinkMap secondSitelinkMap = new SitelinkMap().randomSitelinkWithDescription();
        firstHref = ((com.yandex.direct.api.v5.sitelinks.SitelinkAddItem) firstSitelinkMap.getBean())
                .getHref();
        secondHref = ((com.yandex.direct.api.v5.sitelinks.SitelinkAddItem) secondSitelinkMap.getBean())
                .getHref();
        List<Long> sitelinksSetIds = api.userSteps.sitelinksSteps().add(new AddRequestMap()
                .withSitelinksSets(new SitelinksSetAddItemMap()
                        .withSitelinks(firstSitelinkMap, secondSitelinkMap)));
        assumeThat("был добавлен один набор сайтлинков", sitelinksSetIds, hasSize(1));

        Long sitelinksSetId = sitelinksSetIds.get(0);

        bid = createBannerWithSitelinks(adGroupType, sitelinksSetId);
        api.userSteps.campaignFakeSteps().makeNewCampaignReadyForSendingToBS(cid);
        api.userSteps.groupFakeSteps().makeGroupFullyModerated(pid);
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(bid);
    }

    @Test
    @Description("Отправка сайтлинков при создании объявления")
    public void sendNewBannerWithSitelinksWithDescription() {
        RunBsTransportScriptResponse response = api.userSteps.getDarkSideSteps().getTransportSteps()
                .sendNewCampaign(shard, cid);
        List<String> hrefs = getSitelinksHrefs(response);

        assertThat("ссылки сайтлинков были отправлены верно", hrefs,
                containsInAnyOrder(getExpectedHref(firstHref), getExpectedHref(secondHref)));
    }

    @Test
    @Description("Отправка сайтлинков при обновлении объявления")
    public void sendUpdatedSitelinksWithDescription() {
        api.userSteps.getDarkSideSteps().getTransportSteps().sendNewCampaign(shard, cid);

        SitelinkMap firstSitelinkMap = new SitelinkMap().randomSitelinkWithDescription();
        SitelinkMap secondSitelinkMap = new SitelinkMap().randomSitelinkWithDescription();
        firstHref = ((com.yandex.direct.api.v5.sitelinks.SitelinkAddItem) firstSitelinkMap.getBean())
                .getHref();
        secondHref = ((com.yandex.direct.api.v5.sitelinks.SitelinkAddItem) secondSitelinkMap.getBean())
                .getHref();
        List<Long> sitelinksSetIds = api.userSteps.sitelinksSteps().add(new AddRequestMap()
                .withSitelinksSets(new SitelinksSetAddItemMap()
                        .withSitelinks(firstSitelinkMap, secondSitelinkMap)));
        assumeThat("был добавлен один набор сайтлинков", sitelinksSetIds, hasSize(1));
        long sitelinksSetId = sitelinksSetIds.get(0);

        updateBannerWithSitelinks(adGroupType, bid, sitelinksSetId);
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(bid);

        RunBsTransportScriptResponse response = api.userSteps.getDarkSideSteps().getTransportSteps()
                .sendSyncedCampaign(shard, cid);
        List<String> hrefs = getSitelinksHrefs(response);

        assertThat("описаниия сайтлинков были отправлены верно", hrefs,
                containsInAnyOrder(getExpectedHref(firstHref), getExpectedHref(secondHref)));
    }

    private Long createBannerWithSitelinks(AdGroupType type, Long sitelinksSetId) {
        switch (type) {
            case BASE:
                return api.userSteps.adsSteps().addAd(new AdAddItemMap()
                        .withAdGroupId(pid)
                        .withTextAd(new TextAdAddMap()
                            .defaultTextAd()
                            .withSitelinkSetId(sitelinksSetId)));
            case DYNAMIC:
                return api.userSteps.adsSteps().addAd(new AdAddItemMap()
                        .withAdGroupId(pid)
                        .withDynamicTextAd(new DynamicTextAdAddMap()
                                .defaultDynamicTextAd()
                                .withSitelinkSetId(sitelinksSetId)));
        }
        return -1L;
    }

    private void updateBannerWithSitelinks(AdGroupType type, Long bid, Long sitelinksSetId) {
        switch (type) {
            case BASE:
                api.userSteps.adsSteps().adsUpdate(new UpdateRequestMap().withAds(new AdUpdateItemMap()
                        .withId(bid)
                        .withTextAd(new TextAdUpdateMap()
                                .withSitelinkSetId(sitelinksSetId))));
                return;
            case DYNAMIC:
                api.userSteps.adsSteps().adsUpdate(new UpdateRequestMap().withAds(new AdUpdateItemMap()
                        .withId(bid)
                        .withDynamicTextAd(new DynamicTextAdUpdateMap()
                                .withSitelinkSetId(sitelinksSetId))));
        }
    }

    private List<String> getSitelinksHrefs(RunBsTransportScriptResponse response) {
        Campaign campaign = api.userSteps.getDarkSideSteps().getTransportSteps()
                .getClientDataRequestCampaign(response, 0, cid);
        Context context = campaign != null ? campaign.getContext(pid) : null;
        Banner banner = context != null ? context.getBanner(bid) : null;
        List<Sitelink> sitelinks = banner != null ? banner.getSitelinks() : Collections.emptyList();
        return sitelinks.stream()
                .map(Sitelink::getHref)
                .collect(Collectors.toList());
    }

    private String getExpectedHref(String href) {
        return  expectedParams == null ? href : href + "?" + expectedParams;
    }
}
