package ru.yandex.autotests.directintapi.bstransport.main.campaign.actions.migratingsums.transmit;

import org.hamcrest.core.IsNull;
import org.junit.After;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BsExportQueueRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.CampaignsRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.darkside.Logins;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.TakeRedisLockResult;
import ru.yandex.autotests.directapi.darkside.model.RunBsTransportScriptResponse;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.Campaign;
import ru.yandex.autotests.directapi.darkside.steps.TransportSteps;
import ru.yandex.autotests.directapi.darkside.steps.fakesteps.CampaignFakeSteps;
import ru.yandex.autotests.directapi.darkside.steps.fakesteps.FakeAdminSteps;
import ru.yandex.autotests.directapi.darkside.tags.TestDomains;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directintapi.bstransport.FeatureNames;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.allure.annotations.Title;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.core.Is.is;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

@Aqua.Test
@Tag(TagDictionary.RELEASE)
@Tag(TagDictionary.TRUNK)
@Tag(TestDomains.BsTransport.UPDATE_DATA)
@Title("Транспорт: неотправки в БК кампаний под общим счетом, который заблокирован Redis локом для миграции зачислений")
@Features({FeatureNames.SUMS_MIGRATION, FeatureNames.NOT_FOR_FULL_EXPORT})
@Issue("https://st.yandex-team.ru/DIRECT-86131")
/*
  Тест тесно связан с Redis локами на общий счет, и может падать, если
  лок параллельно взял кто-то еще. Это можно диагностировать по логам bsClientData
  Также тест будет падать, если на тестовой среде не включен флаг возможности
  миграции в новую схему зачислений, см. ExportWorker::_is_sum_migration_possible
  direct-sql dt:ppcdict 'select value from ppc_properties where name = "WALLET_SUMS_MIGRATION_STATE"'
 */
public class BsTransportOfCampaignSumMigrationLockTest {

    private static final String LOGIN = Logins.LOGIN_TRANSPORT_SUM_MIGRATION;
    private static final long REDIS_LOCK_TTL = 2L * 60L;

    @ClassRule
    public static final ApiSteps api = new ApiSteps().as(LOGIN);

    @ClassRule
    public static final SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trashman = new Trashman(api);

    private CampaignFakeSteps campaignFakeSteps;
    private TransportSteps transportSteps;
    private static int shard;
    private long cid;
    private long walletCid;
    private DirectJooqDbSteps dbSteps;
    private Object redisLock = null;
    private FakeAdminSteps fakeAdminSteps;

    @BeforeClass
    public static void beforeClass() {
        shard = api.userSteps.clientFakeSteps().getUserShard(LOGIN);
    }

    @Before
    @Step("Подготовка тестовых данных")
    public void init() {
        campaignFakeSteps = api.userSteps.campaignFakeSteps();
        transportSteps = api.userSteps.getDarkSideSteps().getTransportSteps();
        fakeAdminSteps = api.userSteps.getDarkSideSteps().fakeAdminSteps();
        dbSteps = api.userSteps.getDarkSideSteps().getDirectJooqDbSteps(shard);
        cid = api.userSteps.campaignSteps().addDefaultTextCampaign();
        CampaignsRecord campaign = dbSteps.campaignsSteps().getCampaignById(cid);
        walletCid = campaign.getWalletCid();
    }

    @After
    public void cleanup() {
        if (redisLock != null) {
            fakeAdminSteps.dropRedisLock(redisLock);
        }
    }

    @Test
    public void testNotLockedWalletIsSent() {
        assertCampIsSent(walletCid, false);
    }

    @Test
    public void testCampaignUnderNotLockedWalletIsSent() {
        assertCampIsSent(cid, true);
    }

    @Test
    public void testLockedWalletIsQueuedNotSent() {
        lockWallet();
        assertCampQueuedNotSent(walletCid, false);
    }

    @Test
    public void testCampaignUnderLockedWalletIsQueuedNotSent() {
        lockWallet();
        assertCampQueuedNotSent(cid, true);
    }

    private void lockWallet() {
        String lockKey = String.format("sum-migration-%d{group%d}1", walletCid, walletCid % 3);
        TakeRedisLockResult result = fakeAdminSteps.takeRedisLock(lockKey,REDIS_LOCK_TTL);
        assumeThat("удалось взять Redis лок на общий счет: "+lockKey, result.getLocks().isEmpty(), is(false));
        redisLock = result.getLocks();
    }

    private void assertCampIsSent(long cid, boolean isNew) {
        if (isNew) {
            campaignFakeSteps.makeNewCampaignReadyForSendingToBS(cid);
        } else {
            campaignFakeSteps.setBSSynced(cid, false);
        }

        RunBsTransportScriptResponse resp;
        if (isNew) {
            resp = transportSteps.sendNewCampaign(shard, cid);
        } else {
            resp = transportSteps.sendSyncedCampaign(shard, cid);
        }
        Campaign campaign = transportSteps.getClientDataRequestCampaign(resp, 0, cid);
        assertThat("в БК отправлена кампания", campaign, IsNull.notNullValue());
    }

    private void assertCampQueuedNotSent(long cid, boolean isNew) {
        if (isNew) {
            campaignFakeSteps.makeNewCampaignReadyForSendingToBS(cid);
        } else {
            campaignFakeSteps.setBSSynced(cid, false);
        }
        transportSteps.runBsExportMasterScript(shard, cid);
        BsExportQueueRecord bsExportQueueRecord = dbSteps.transportSteps().getBsExportQueueRecord(cid);
        assumeThat("кампания попала в очередь ppc.bs_export_queue",
                bsExportQueueRecord, notNullValue());

        RunBsTransportScriptResponse resp = transportSteps.runBsClientDataScript(shard, cid);
        assertThat("в БК не отправлена кампания", resp, IsNull.nullValue());
    }
}
