package ru.yandex.autotests.directintapi.bstransport.main.campaign.parameters.createtime;

import java.util.Arrays;
import java.util.Collection;
import java.util.Map;

import org.jooq.impl.DSL;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.CampOptions;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.darkside.Logins;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.darkside.model.RunBsTransportScriptResponse;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directintapi.bstransport.FeatureNames;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Title;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.notNullValue;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

@Aqua.Test
@Tag(TagDictionary.TRUNK)
@Tag(TagDictionary.RELEASE)
@Title("Транспорт: проверка отправки параметра OrderCreateTime (дата создания кампании)")
@Features({FeatureNames.CAMPAIGNS, FeatureNames.NOT_FOR_FULL_EXPORT})
@RunWith(Parameterized.class)
public class BsTransportOfOrderCreateTimeTest {

    private static final String login = Logins.LOGIN_TRANSPORT;
    private static final String ZERO_TIME = "0000-00-00 00:00:00";

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(login);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    private static int shard;

    @Parameterized.Parameter(0)
    public String createTime;

    @Parameterized.Parameter(1)
    public String expectedUnixTime;

    @Parameterized.Parameters(name = "Дата создания кампании = {0}")
    public static Collection<Object[]> data() {
        Object[][] data = new Object[][]{
                {ZERO_TIME, null}, // ничего не отпрвляем для нулевого timestamp
                {"1970-01-01 03:00:01", "1"},  // работаем в московской таймзоне (UTC +3)
                {"2017-01-25 05:23:56", "1485311036"},
        };
        return Arrays.asList(data);
    }

    @BeforeClass
    public static void beforeClass() {
        shard = api.userSteps.clientFakeSteps().getUserShard(login);
    }

    @Test
    @Title("Отправка в БК параметра OrderCreateTime (дата создания кампании)")
    public void testTransportOfStopTime() {
        Long cid = api.userSteps.campaignSteps().addDefaultTextCampaign();
        api.userSteps.getDirectJooqDbSteps().useShard(shard).campaignsSteps().setCampaignOption(cid,
                CampOptions.CAMP_OPTIONS.CREATE_TIME, DSL.value(createTime).cast(java.sql.Timestamp.class));
        api.userSteps.campaignFakeSteps().makeNewCampaignReadyForSendingToBS(cid);

        RunBsTransportScriptResponse resp =
                api.userSteps.getDarkSideSteps().getTransportSteps().sendNewCampaign(shard, cid);

        Map<String, Map> campaignRaw =
                api.userSteps.getDarkSideSteps().getTransportSteps().getRawClientDataLogEntriesForCampaignWithOmittedMdsMeta(resp, cid);
        assumeThat("в запросе к БК присутствует объект campaign", campaignRaw, notNullValue());

        // Значение OrderCreateTime берем из сырых логов, чтобы убедиться что ключ отсутсвует при нулевой дате
        // создания кампании
        Map campaign =
                (Map) ((Map) ((Map) (campaignRaw.get(resp.getUuid()[0])).get("request")).get("ORDER")).get(cid.toString());
        if (createTime == ZERO_TIME) {
            assertThat("в БК не отправлен параметр OrderCreateTime", campaign.containsKey("OrderCreateTime"),
                    equalTo(false));
        } else {
            assertThat("отправленный в БК параметр OrderCreateTime соответствует ожидаемому",
                    campaign.get("OrderCreateTime"), equalTo(expectedUnixTime));
        }
    }
}
