package ru.yandex.autotests.directintapi.bstransport.main.campaign.parameters.deals;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;

import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.AgencyOptionsRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.direct.web.api.core.DirectRule;
import ru.yandex.autotests.direct.web.api.models.Deal;
import ru.yandex.autotests.direct.web.api.rules.DealRule;
import ru.yandex.autotests.directapi.darkside.Logins;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.darkside.model.CampaignsType;
import ru.yandex.autotests.directapi.darkside.model.RunBsTransportScriptResponse;
import ru.yandex.autotests.directapi.darkside.model.bslogs.UpdateInfo;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.Campaign;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.ItemDeal;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directintapi.bstransport.FeatureNames;
import ru.yandex.autotests.directintapi.bstransport.StoriesNames;
import ru.yandex.autotests.directintapi.bstransport.TransportHelpSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.allure.annotations.Title;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.core.IsNull.nullValue;
import static ru.yandex.autotests.direct.db.models.jooq.ppc.tables.AgencyOptions.AGENCY_OPTIONS;
import static ru.yandex.autotests.directapi.model.Logins.AGENCY_DEALS;
import static ru.yandex.autotests.directapi.model.Logins.SUPER_LOGIN;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

@Aqua.Test
@Tag(TagDictionary.TRUNK)
@Tag(TagDictionary.RELEASE)
@Title("Транспорт: проверка отправки параметров сделок (ItemDeal, AgencyCoef, PrivateDeal)")
@Stories(StoriesNames.CAMPAIGN_PARAMS_DEALS)
@Features({FeatureNames.CAMPAIGNS, FeatureNames.NOT_FOR_FULL_EXPORT})
@RunWith(Parameterized.class)
public class DirectDealsTest {

    private static final String login = Logins.LOGIN_TRANSPORT;

    private static final String agencyLogin = AGENCY_DEALS;

    private static DirectJooqDbSteps jooqDbSteps;

    private static int shard;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(login);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    public DealRule dealRule = new DealRule().withUlogin(agencyLogin);

    @Rule
    public DirectRule directRule =
            DirectRule.defaultRule().as(SUPER_LOGIN).withRules(dealRule);

    @Parameterized.Parameter()
    public Integer dealsCount;

    @Parameterized.Parameter(1)
    public Double agencyCoef;

    @Parameterized.Parameter(2)
    public Boolean isDealActive;

    @Parameterized.Parameter(3)
    public Boolean isCampaignsDealsActive;

    @Parameterized.Parameter(4)
    public Integer expectedAgencyCoef;

    @Parameterized.Parameters(name = "Количество сделок = {0}, AgencyCoef = {1}, Будет запущена = {2}, Коэффициент ожидается = {3}")
    public static Collection<Object[]> data() {
        Object[][] data = new Object[][]{
                //Сделок нет, агентский коэффициент выставлен
                {0, 10.01, false, true, null},
                //Есть одна активная сделка, агентский коэффициент выставлен
                {1, 10.01, true, true, 100_100},
                //Есть одна неактивная сделка, агентский коэффициент выставлен
                {1, 10.01, false, true, null},
                //Есть одна сделка, агентский коэффициент не выставлен, используется дефолт - 15%
                {1, 0.0, true, true, 150_000},
                //Есть несколько сделок, агентский коэффициент выставлен
                {3, 0.1, true, true, 1000},
                //Есть сделка, агентский коэффициент 10%
                {1, 10.0, true, true, 100_000},
                //Есть сделка, агентский коэффициент 100%
                {1, 100.0, true, true, 1000_000},
                //Есть несколько сделок, привязка к кампаниям удалена
                {3, 10.0, true, false, 100_000},
                //Есть сделка, привязка к кампании удалена
                {1, 10.0, true, false, 100_000},
        };
        return Arrays.asList(data);
    }

    @BeforeClass
    public static void beforeClass() {
        shard = api.userSteps.clientFakeSteps().getUserShard(login);
        jooqDbSteps = api.userSteps.getDirectJooqDbSteps();
    }

    @Test
    @Title("Отправка в БК частных сделок")
    public void testTransportOfDirectDeals() {

        Long cid = createNewCamapaignAndReturnCid();
        api.userSteps.campaignFakeSteps().setType(cid, CampaignsType.CPM_DEALS);

        List<ItemDeal> expectedDeals = new ArrayList<>();

        long agencyId = Long.decode(User.get(agencyLogin).getClientID());
        jooqDbSteps.useShard(shard).campaignsSteps().setAgencyId(cid, agencyId);

        int agencyShard = api.userSteps.clientFakeSteps().getUserShard(agencyLogin);

        jooqDbSteps.useShard(agencyShard).agencyOptionsSteps().add(new
                AgencyOptionsRecord()
                .with(AGENCY_OPTIONS.CLIENTID, agencyId)
                .with(AGENCY_OPTIONS.PREFERRED_DEAL_FEE_PERCENT, BigDecimal.valueOf(agencyCoef))
                .with(AGENCY_OPTIONS.ALLOW_CLIENTS_WITHOUT_WALLET, 0)
                .with(AGENCY_OPTIONS.DEFAULT_CLIENTS_WITH_WALLET, 0)
        );

        TransportHelpSteps helpSteps = new TransportHelpSteps(api);
        Campaign expectedCampaign = helpSteps.buildExpectedCampaignObjFromDb(cid, shard, UpdateInfo.UPDATE);
        expectedCampaign.setPrivateDealFlag(1);

        List<Long> dealIds = new ArrayList<>();

        if (dealsCount > 0) {
            List<Deal> deals =
                    dealRule.getDirectWebApiSteps().dealTestSteps().addRandomDeals(dealsCount.longValue(), agencyLogin);
            deals.forEach(deal -> {
                jooqDbSteps.useShard(shard).campaignDealsSteps()
                        .addCampaignDeal(deal.getId(), cid, isCampaignsDealsActive ? 0 : 1);
                expectedDeals.add(new ItemDeal().withId(deal.getId()));
                dealIds.add(deal.getId());
            });

            if (isDealActive) {
                dealRule.getDirectWebApiSteps().dealSteps().activateDeals(dealIds, agencyLogin);
                if (isCampaignsDealsActive) {
                    expectedCampaign.setAgencyCoef(expectedAgencyCoef);
                    expectedCampaign.setDirectDeals(expectedDeals);
                    expectedCampaign.getDirectDeals().sort(Comparator.comparing(ItemDeal::getId));
                }
            }
        }

        RunBsTransportScriptResponse resp =
                api.userSteps.getDarkSideSteps().getTransportSteps().sendNewCampaign(shard, cid);

        Campaign campaign =
                api.userSteps.getDarkSideSteps().getTransportSteps().getClientDataRequestCampaign(resp, cid);

        assertThat("значения параметров, связанных со сделками, соответствуют ожидаемым",
                campaign,
                helpSteps.campaignReq1Matcher(expectedCampaign)
        );
        if (!(isDealActive && isCampaignsDealsActive)) {
            assertThat("в БК не отправились сделки", campaign.getDirectDeals(), nullValue());
        }
    }

    private Long createNewCamapaignAndReturnCid() {
        Long cid = api.userSteps.campaignSteps().addDefaultTextCampaign();
        api.userSteps.campaignFakeSteps().makeNewCampaignReadyForSendingToBS(cid);

        return cid;
    }

}
