package ru.yandex.autotests.directintapi.bstransport.main.campaign.parameters.finance;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Collection;

import com.jayway.awaitility.Duration;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.ClientsOptionsStatusbalancebanned;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.common.api45mng.APIPort_PortType;
import ru.yandex.autotests.directapi.common.api45mng.CreateNewSubclientResponse;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.darkside.model.RunBsTransportScriptResponse;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.Campaign;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.steps.ConditionFactories;
import ru.yandex.autotests.directintapi.bstransport.FeatureNames;
import ru.yandex.autotests.directintapi.bstransport.StoriesNames;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.allure.annotations.Title;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.lessThan;
import static ru.yandex.autotests.directapi.darkside.Logins.LOGIN_MNGR;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

@Aqua.Test
@Tag(TagDictionary.TRUNK)
@Tag(TagDictionary.RELEASE)
@Title("Транспорт: проверка отправки в БК суммы общего счёта клиента, у которого включён автоовердрафт")
@Stories(StoriesNames.CAMPAIGN_PARAMS_FINANCE)
@Features({FeatureNames.CAMPAIGNS, FeatureNames.NOT_FOR_FULL_EXPORT})
@RunWith(Parameterized.class)
public class BsTransportOfWalletSumWithAutoOverdraftLimitTest {
    private static final LogSteps logger = LogSteps.getLogger(BsTransportOfWalletSumWithAutoOverdraftLimitTest.class);

    private static final double EPSILON = 0.000001;

    // Инициализируем степы для заданного менеджера
    @ClassRule
    public static final ApiSteps api = new ApiSteps().wsdl(APIPort_PortType.class).as(LOGIN_MNGR);

    @ClassRule
    public static final SemaphoreRule semaphore = Semaphore.getSemaphore();

    // Пока эта штука работает только для рублей
    // В будущем будут добавляться и другие валюты
    public final static Currency currency = Currency.RUB;

    private static int shard;
    private static DirectJooqDbSteps jooqDbSteps;
    private static long clientID;
    private static long cid;
    private static long walletCid;

    private Campaign wallet;

    @Parameterized.Parameter(0)
    public BigDecimal walletSum;

    @Parameterized.Parameter(1)
    public BigDecimal campSumSpent;

    @Parameterized.Parameter(2)
    public BigDecimal overdraftLimit;

    @Parameterized.Parameter(3)
    public BigDecimal autoOverdraftLimit;

    @Parameterized.Parameter(4)
    public BigDecimal clientDebt;

    @Parameterized.Parameter(5)
    public ClientsOptionsStatusbalancebanned statusBalanceBanned;

    @Parameterized.Parameter(6)
    public BigDecimal expectedSumCur;

    @Parameterized.Parameter(7)
    public String name;

    @Parameterized.Parameters(name = "walletSum={0}, sumSpent={1}, overdraftLim={2},"
            + " autoOverdraftLim={3}, clientDebt={4}, statusBalanceBanned={5}, name={7}")
    public static Collection<Object[]> data() {
        Object[][] data = new Object[][]{
                {
                        BigDecimal.valueOf(345000), BigDecimal.ZERO,
                        BigDecimal.valueOf(100000), BigDecimal.valueOf(17000.57), BigDecimal.ZERO,
                        ClientsOptionsStatusbalancebanned.No,
                        BigDecimal.valueOf(362000.57),
                        "Сумма автоовердрафта добавляется к wallet.sum"
                },
                {
                        BigDecimal.valueOf(1000), BigDecimal.valueOf(1200),
                        BigDecimal.valueOf(500), BigDecimal.valueOf(500), BigDecimal.ZERO,
                        ClientsOptionsStatusbalancebanned.Yes,
                        // Клиент открутился уже на 1200, но успел внести только 1000
                        // Баланс его "забанил", а значит, AutoOverdraftLim далее ему недоступен,
                        // но он уже открутился на 1200, поэтому в качестве SUMCur мы должны
                        // отправить 1200 (если отправим меньше, то потраченные "кредитные" деньги
                        // будут откачены как технические перекруты)
                        BigDecimal.valueOf(1200),
                        "Автоовердрафт фиксируется на использованном (statusBalanceBanned='Yes')"
                },
                {
                        BigDecimal.valueOf(1000), BigDecimal.valueOf(1700),
                        BigDecimal.valueOf(500), BigDecimal.valueOf(500), BigDecimal.ZERO,
                        ClientsOptionsStatusbalancebanned.Yes,
                        // По какой-то причине клиент открутился на 1700, что даже больше чем Sum+AutoOverdraftLim
                        // (например, из-за перекрутов в БК). Баланс его "забанил", значит,
                        // AutoOverdraftLim далее ему недоступен. Из-за ограничения сверху в Sum+AutoOverdraftLim
                        // (чтобы не уйти в таких случаях в бесконечность) отправим 1000+500,
                        // лишние 200 будут списаны как технические перекруты (что фактически и произошло)
                        BigDecimal.valueOf(1500),
                        "Не отправляем в БК значения, превышающие Sum+AutoOverdraftLim (statusBalanceBanned='Yes')"
                },
                {
                        BigDecimal.valueOf(1000), BigDecimal.valueOf(800),
                        BigDecimal.valueOf(50), BigDecimal.valueOf(50), BigDecimal.ZERO,
                        ClientsOptionsStatusbalancebanned.Yes,
                        // Клиент открутился на 1000, а заплатил уже 800. Баланс его "забанил", а значит,
                        // AutoOverdraftLim далее ему недоступен. Но он ещё и не докрутил свои деньги,
                        // а значит, мы должны отправить 1000.
                        BigDecimal.valueOf(1000),
                        "Деньги ещё есть, не ограничиваем клиента (statusBalanceBanned='Yes')"
                },
                {
                        BigDecimal.valueOf(1000), BigDecimal.valueOf(1200),
                        BigDecimal.valueOf(0), BigDecimal.valueOf(500), BigDecimal.ZERO,
                        ClientsOptionsStatusbalancebanned.No,
                        // Случай 1, но в этом месте мы узнаём о том, что клиент "забанен" в Балансе
                        // косвенно -- из-за того, что overdraftLim = 0, при том что autoOverdraftLim > 0
                        BigDecimal.valueOf(1200),
                        "Автоовердрафт фиксируется на использованном (overdraftLim=0)"
                },
                {
                        BigDecimal.valueOf(1000), BigDecimal.valueOf(1700),
                        BigDecimal.valueOf(500), BigDecimal.valueOf(500), BigDecimal.ZERO,
                        ClientsOptionsStatusbalancebanned.No,
                        // Случай 2, но в этом месте мы узнаём о том, что клиент "забанен" в Балансе
                        // косвенно -- из-за того, что overdraftLim = 0, при том что autoOverdraftLim > 0
                        BigDecimal.valueOf(1500),
                        "Не отправляем в БК значения, превышающие Sum+AutoOverdraftLim (overdraftLim=0)"
                },
                {
                        BigDecimal.valueOf(1000), BigDecimal.valueOf(800),
                        BigDecimal.valueOf(0), BigDecimal.valueOf(500), BigDecimal.ZERO,
                        ClientsOptionsStatusbalancebanned.No,
                        // Случай 3, но в этом месте мы узнаём о том, что клиент "забанен" в Балансе
                        // косвенно -- из-за того, что overdraftLim = 0, при том что autoOverdraftLim > 0
                        BigDecimal.valueOf(1000),
                        "Деньги ещё есть, не ограничиваем клиента (overdraftLim=0)"
                },
                {
                        BigDecimal.valueOf(1000), BigDecimal.ZERO,
                        BigDecimal.valueOf(2000), BigDecimal.valueOf(1500), BigDecimal.valueOf(150),
                        ClientsOptionsStatusbalancebanned.No,
                        // Если у клиента есть неоплаченный счёт, который он выставил себе в рамках
                        // обычного овердрафта (не АВТОовердрафта), то лимит автоовердрафта уменьшается
                        // на сумму этого счёта (пока клиент не оплатит того, что уже задолжал).
                        BigDecimal.valueOf(2350),
                        "Учитываем неоплаченный счёт (clients_options.debt > 0)"
                }
        };
        return Arrays.asList(data);
    }

    /**
     * Создаём одного нового клиента для прогона тестов и дожидаемся обработки NotifyClient и NofityOrder2.
     * Добавляем ему кошелёк и кампанию.
     */
    @BeforeClass
    public static void beforeClass() {
        // Создаем сервисируемого клиента
        CreateNewSubclientResponse createClientResponse = api.userSteps.clientSteps()
                .createServicedClient("transport-wallet", LOGIN_MNGR, currency.toString());

        String login = createClientResponse.getLogin();
        clientID = createClientResponse.getClientID();

        // Инициализируем jooqDbSteps нужным номером шарда
        shard = api.userSteps.clientFakeSteps().getUserShard(login);
        jooqDbSteps = api.userSteps.getDirectJooqDbSteps();
        jooqDbSteps.useShard(shard);

        cid = api.userSteps.campaignSteps().addDefaultTextCampaign(login);

        // У менеджерского клиента ОС уже подключен
        // Надо только выдать НДС
        api.userSteps.clientFakeSteps().setVATRate(login, 0);
        walletCid = jooqDbSteps.campaignsSteps().getCampaignById(cid).getWalletCid();

        // Ожидаем создания записи в таблице clients_options,
        // чтобы избежать гонок по изменению поля clients_options.overdraft_lim
        ConditionFactories.NOTIFY_CLIENT.with().timeout(Duration.FIVE_MINUTES)
                .until(() -> {
                    logger.info("Wait clients_options record being created");
                    return jooqDbSteps.clientsOptionsSteps().getClientOptions(clientID) != null;
                });

        // Подождём, пока в ручку NotifyOrder2 придёт нотификация о созданном кошельке,
        // чтобы избежать гонок по изменению поля campaigns.sum
        ConditionFactories.NOTIFY_ORDER2.until(() -> {
            logger.info("Wait balance_tid > 0");
            return jooqDbSteps.campaignsSteps().getCampaignById(walletCid).getBalanceTid() > 0;
        });
    }

    @Before
    public void before() {
        // Проставляем поля в clients_options
        jooqDbSteps.clientsOptionsSteps().setOverdraftLimit(clientID, overdraftLimit);
        jooqDbSteps.clientsOptionsSteps().setAutoOverdraftLimit(clientID, autoOverdraftLimit);
        jooqDbSteps.clientsOptionsSteps().setStatusBalanceBanned(clientID, statusBalanceBanned);
        jooqDbSteps.clientsOptionsSteps().setDebt(clientID, clientDebt);

        // Установим sum на кошельке
        api.userSteps.campaignFakeSteps().setCampaignSum(walletCid, walletSum.floatValue());
        api.userSteps.campaignFakeSteps().setStatusBsSynced(walletCid, "No");

        // Установим sum_spent на кампании
        api.userSteps.campaignFakeSteps().setSumSpent(cid, campSumSpent.floatValue());

        // Отправляем кампанию-кошелек
        RunBsTransportScriptResponse resp = api.userSteps.getDarkSideSteps()
                .getTransportSteps().sendSyncedCampaign(shard, walletCid);
        wallet = api.userSteps.getDarkSideSteps().getTransportSteps().getClientDataRequestCampaign(resp, walletCid);
    }

    @Test
    @Title("Отправка в БК кампании-кошелька с автоовердрафтом")
    public void testTransportOfWalletSumCurWithAutoOverdraftLimit() {
        assertThat("Отправленный в БК SUMCur соответствует ожидаемому",
                Math.abs(expectedSumCur.doubleValue() - wallet.getSumCur()),
                lessThan(EPSILON));
    }
}
