package ru.yandex.autotests.directintapi.bstransport.main.campaign.parameters.geocoef;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.common.collect.ImmutableList;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.model.RegionIDValues;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.darkside.Logins;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.darkside.model.BidMultiplierConditionBuilder;
import ru.yandex.autotests.directapi.darkside.model.RunBsTransportScriptResponse;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.Campaign;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.Context;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.ExpressionCoefItem;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.ExpressionCoefs;
import ru.yandex.autotests.directapi.darkside.model.multipliers.DirectAtom;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directintapi.bstransport.FeatureNames;
import ru.yandex.autotests.directintapi.bstransport.StoriesNames;
import ru.yandex.autotests.directintapi.bstransport.TransportHelpSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.allure.annotations.Title;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.Matchers.nullValue;
import static ru.yandex.autotests.directapi.darkside.model.BidMultiplierUtils.geoBsAtomRegion;
import static ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.Campaign.REGION_COEF;
import static ru.yandex.autotests.directintapi.bstransport.TransportTestUtils.checkBsUnifiedMultipliers;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;
import static ru.yandex.autotests.irt.testutils.beandiffer2.BeanDifferMatcher.beanDiffer;

@Aqua.Test
@Tag(TagDictionary.RELEASE)
@Tag(TagDictionary.TRUNK)
@Title("Транспорт: проверка отправки параметра RegionCoef на уровне кампании")
@Stories(StoriesNames.CAMPAIGN_PARAMS_REGIONCOEF)
@Features(FeatureNames.CAMPAIGNS)
@Issue("https://st.yandex-team.ru/DIRECT-62869")
public class BsTransportOfOrderRegionCoefTest {

    private static final String LOGIN = Logins.LOGIN_TRANSPORT;
    private static final RegionIDValues FIRST_REGION = RegionIDValues.MOSCOW_AND_MOSCOW_REGION;
    private static final Short FIRST_COEF = 50;
    private static final RegionIDValues SECOND_REGION = RegionIDValues.MOSCOW;
    private static final Short SECOND_COEF = 1200;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trashman = new Trashman(api);

    private static TransportHelpSteps helpSteps = new TransportHelpSteps(api);

    private static int shard;

    private Long cid;
    private Long pid;
    private RunBsTransportScriptResponse resp;

    @BeforeClass
    public static void beforeClass() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
        shard = api.userSteps.clientFakeSteps().getUserShard(LOGIN);
    }

    @Before
    @Step("Подготовка тестовых данных")
    public void before() {
        cid = api.userSteps.campaignSteps().addDefaultTextCampaign();
        pid = api.userSteps.adGroupsSteps().addDefaultGroup(cid);
        Long bid = api.userSteps.adsSteps().addDefaultTextAd(pid);
        Long keywordId = api.userSteps.keywordsSteps().addDefaultKeyword(pid);
        api.userSteps.campaignFakeSteps().makeNewCampaignReadyForSendingToBS(cid);
        api.userSteps.groupFakeSteps().makeGroupFullyModerated(pid);
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(bid);
        api.userSteps.phrasesFakeSteps().makeKeywordModerated(keywordId);
    }

    @Test
    @Title("Отправка в БК корректировки RegionCoef на уровне кампании, одна группа без корректировок")
    public void testTransportOfOrderRegionCoefOneGroupNoCoef() {
        resp = api.userSteps.getDarkSideSteps().getTransportSteps().sendNewCampaign(shard, cid);
        Map<String, Map> orderMap = helpSteps.getOrderMap(resp, cid);
        Campaign campaign = api.userSteps.getDarkSideSteps().getTransportSteps()
                .getClientDataRequestCampaign(resp, 0, cid);
        assumeThat("в БК отправлен заказ (кампания)", orderMap, notNullValue());
        assertThat("в БК отправлена корректировка RegionCoef со значением undef на уровне кампании",
                orderMap.get(REGION_COEF), nullValue());
        assertThat("в БК отправлена корректировка ExpressionCoefs.Geo со значением undef на уровне кампании",
                campaign.getExpressionCoefs().getGeo(), nullValue());
    }

    @Test
    @Title("Отправка в БК корректировки RegionCoef на уровне кампании, одна группа c одной корректировкой")
    public void testTransportOfOrderRegionCoefOneGroupSingleCoef() {
        Map<Long, Short> multipliers = new HashMap<>();
        multipliers.put(FIRST_REGION.getId().longValue(), FIRST_COEF);

        api.userSteps.getDirectJooqDbSteps().useShard(shard).multipliersSteps()
                .saveGeoMultiplierValues(cid, multipliers);
        resp = api.userSteps.getDarkSideSteps().getTransportSteps().sendNewCampaign(shard, cid);

        Map<String, Integer> expected = new HashMap<>();
        expected.put(String.valueOf(FIRST_REGION.getId()), FIRST_COEF.intValue());
        Map<String, Map> orderMap = helpSteps.getOrderMap(resp, cid);
        assumeThat("в БК отправлен заказ (кампания)", orderMap, notNullValue());
        assertThat("отправленная в БК корректировка RegionCoef на уровне кампании соответствует ожидаемой",
                orderMap.get(REGION_COEF)
                , beanDiffer(expected));

        Campaign campaign = api.userSteps.getDarkSideSteps().getTransportSteps()
                .getClientDataRequestCampaign(resp, 0, cid);

        // проверяем передачу в унифицированном формате
        ExpressionCoefs expressionCoefs = campaign.getExpressionCoefs();
        assumeThat("Отправлен ExpressionCoefs", expressionCoefs, notNullValue());

        List<ExpressionCoefItem> geoCoefs = expressionCoefs.getGeo();
        assumeThat("Отправлен ExpressionCoefs.Geo", geoCoefs, notNullValue());

        List<List<DirectAtom>> expectedExpression = BidMultiplierConditionBuilder.init()
                .and(geoBsAtomRegion(Integer.toString(FIRST_REGION.getId())))
                .build();

        checkBsUnifiedMultipliers(geoCoefs,
                ImmutableList.of(FIRST_COEF.intValue()),
                ImmutableList.of(expectedExpression));
    }

    @Test
    @Title("Отправка в БК корректировки RegionCoef на уровне кампании, одна группа с двойной коректировкой")
    public void testTransportOfOrderRegionCoefOneGroupMutlyCoef() {
        Map<Long, Short> multipliers = new HashMap<>();
        multipliers.put(FIRST_REGION.getId().longValue(), FIRST_COEF);
        multipliers.put(SECOND_REGION.getId().longValue(), SECOND_COEF);

        api.userSteps.getDirectJooqDbSteps().useShard(shard).multipliersSteps()
                .saveGeoMultiplierValues(cid, multipliers);
        resp = api.userSteps.getDarkSideSteps().getTransportSteps().sendNewCampaign(shard, cid);

        Map<String, Integer> expected = new HashMap<>();
        expected.put(String.valueOf(FIRST_REGION.getId()), FIRST_COEF.intValue());
        expected.put(String.valueOf(SECOND_REGION.getId()), SECOND_COEF.intValue());
        Map<String, Map> orderMap = helpSteps.getOrderMap(resp, cid);
        assumeThat("в БК отправлен заказ (кампания)", orderMap, notNullValue());
        assertThat("отправленная в БК корректировка RegionCoef на уровне кампании соответствует ожидаемой",
                orderMap.get(REGION_COEF)
                , beanDiffer(expected));

        Campaign campaign = api.userSteps.getDarkSideSteps().getTransportSteps()
                .getClientDataRequestCampaign(resp, 0, cid);

        // проверяем передачу в унифицированном формате
        ExpressionCoefs expressionCoefs = campaign.getExpressionCoefs();
        assumeThat("Отправлен ExpressionCoefs", expressionCoefs, notNullValue());

        List<ExpressionCoefItem> geoCoefs = expressionCoefs.getGeo();
        assumeThat("Отправлен ExpressionCoefs.Geo", geoCoefs, notNullValue());

        List<List<DirectAtom>> expectedExpressionFirst = BidMultiplierConditionBuilder.init()
                .and(geoBsAtomRegion(Integer.toString(FIRST_REGION.getId())))
                .build();
        List<List<DirectAtom>> expectedExpressionSecond = BidMultiplierConditionBuilder.init()
                .and(geoBsAtomRegion(Integer.toString(SECOND_REGION.getId())))
                .build();

        checkBsUnifiedMultipliers(geoCoefs,
                ImmutableList.of(FIRST_COEF.intValue(), SECOND_COEF.intValue()),
                ImmutableList.of(expectedExpressionFirst, expectedExpressionSecond));
    }

    @Test
    @Title("Отправка в БК корректировки RegionCoef на уровне кампании, одна группа c одной отключенной корректировкой")
    public void testTransportOfOrderRegionCoefOneGroupDisabledSingleCoef() {
        Map<Long, Short> multipliers = new HashMap<>();
        multipliers.put(FIRST_REGION.getId().longValue(), FIRST_COEF);

        api.userSteps.getDirectJooqDbSteps().useShard(shard).multipliersSteps()
                .saveGeoMultiplierValues(cid, false, multipliers);
        resp = api.userSteps.getDarkSideSteps().getTransportSteps().sendNewCampaign(shard, cid);

        Map<String, Map> orderMap = helpSteps.getOrderMap(resp, cid);
        Campaign campaign = api.userSteps.getDarkSideSteps().getTransportSteps()
                .getClientDataRequestCampaign(resp, 0, cid);
        assumeThat("в БК отправлен заказ (кампания)", orderMap, notNullValue());
        assertThat("в БК отправлена корректировка RegionCoef со значением undef на уровне кампании",
                orderMap.get(REGION_COEF), nullValue());
        assertThat("в БК отправлена корректировка ExpressionCoefs.Geo со значением undef на уровне кампании",
                campaign.getExpressionCoefs().getGeo(), nullValue());
    }

    @Test
    @Title("Отправка в БК корректировки RegionCoef на уровне кампании, две группы c одной корректировкой")
    public void testTransportOfOrderRegionCoefTwoGroupsSingleCoef() {
        Map<Long, Short> multipliers = new HashMap<>();
        multipliers.put(FIRST_REGION.getId().longValue(), FIRST_COEF);
        Long pidAnother = api.userSteps.adGroupsSteps().addDefaultGroup(cid);
        Long bidAnother = api.userSteps.adsSteps().addDefaultTextAd(pidAnother);
        Long keywordIdAnother = api.userSteps.keywordsSteps().addDefaultKeyword(pidAnother);
        api.userSteps.campaignFakeSteps().makeNewCampaignReadyForSendingToBS(cid);
        api.userSteps.groupFakeSteps().makeGroupFullyModerated(pidAnother);
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(bidAnother);
        api.userSteps.phrasesFakeSteps().makeKeywordModerated(keywordIdAnother);

        api.userSteps.getDirectJooqDbSteps().useShard(shard).multipliersSteps()
                .saveGeoMultiplierValues(cid, multipliers);
        resp = api.userSteps.getDarkSideSteps().getTransportSteps().sendNewCampaign(shard, cid);

        Map<String, Integer> expected = new HashMap<>();
        expected.put(String.valueOf(FIRST_REGION.getId()), FIRST_COEF.intValue());
        Map<String, Map> orderMap = helpSteps.getOrderMap(resp, cid);
        Campaign campaign = api.userSteps.getDarkSideSteps().getTransportSteps()
                .getClientDataRequestCampaign(resp, cid);
        assumeThat("в БК отправлен заказ (кампания)", orderMap, notNullValue());
        assertThat("отправленная в БК корректировка RegionCoef на уровне кампании соответствует ожидаемой",
                orderMap.get(REGION_COEF)
                , beanDiffer(expected));

        // проверяем передачу в унифицированном формате
        ExpressionCoefs expressionCoefs = campaign.getExpressionCoefs();
        assumeThat("Отправлен ExpressionCoefs", expressionCoefs, notNullValue());

        List<ExpressionCoefItem> geoCoefs = expressionCoefs.getGeo();
        assumeThat("Отправлен ExpressionCoefs.Geo", geoCoefs, notNullValue());

        List<List<DirectAtom>> expectedExpression = BidMultiplierConditionBuilder.init()
                .and(geoBsAtomRegion(Integer.toString(FIRST_REGION.getId())))
                .build();

        checkBsUnifiedMultipliers(geoCoefs,
                ImmutableList.of(FIRST_COEF.intValue()),
                ImmutableList.of(expectedExpression));
    }

    @Test
    @Title("Неотправка в БК корректировки RegionCoef на уровне группы")
    public void testNotSendingRegionCoefOnContext() {
        Map<Long, Short> multipliers = new HashMap<>();
        multipliers.put(FIRST_REGION.getId().longValue(), FIRST_COEF);

        api.userSteps.getDirectJooqDbSteps().useShard(shard).multipliersSteps()
                .saveGeoMultiplierValues(cid, false, multipliers);
        resp = api.userSteps.getDarkSideSteps().getTransportSteps().sendNewCampaign(shard, cid);

        Map<String, Map> contextMap = helpSteps.getContextMap(resp, cid, pid);
        Campaign campaign = api.userSteps.getDarkSideSteps().getTransportSteps()
                .getClientDataRequestCampaign(resp, cid);
        Context context = campaign != null ? campaign.getContext(pid) : null;
        assumeThat("в БК отправлен контекст (группа)", contextMap, notNullValue());
        assertThat("в БК не отправлена корректировка RegionCoef на уровне группы",
                contextMap.get(REGION_COEF), nullValue());
        assertThat("в БК не отправлена корректировка ExpressionCoefs.Geo на уровне группы",
                context.getExpressionCoefs().getGeo(), nullValue());
    }
}
