package ru.yandex.autotests.directintapi.bstransport.main.campaign.parameters.mobile;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import com.yandex.direct.api.v5.campaigns.MobileAppCampaignSearchStrategyTypeEnum;
import org.hamcrest.Matcher;
import org.hamcrest.Matchers;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.steps.JooqCampaignsOpts;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.darkside.Logins;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.darkside.model.RunBsTransportScriptResponse;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.Campaign;
import ru.yandex.autotests.directapi.darkside.model.multipliers.BsAtom;
import ru.yandex.autotests.directapi.model.api5.campaigns.CampaignAddItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.MobileAppCampaignAddItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.MobileAppCampaignNetworkStrategyAddMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.MobileAppCampaignSearchStrategyAddMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.MobileAppCampaignStrategyAddMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directintapi.bstransport.FeatureNames;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Title;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static ru.yandex.autotests.directapi.darkside.model.internalads.AdditionalTargetingParameter.OS_FAMILIES;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.beandiffer2.BeanDifferMatcher.beanDiffer;

@Aqua.Test
@Tag(TagDictionary.TRUNK)
@Title("Проверка TargetingExpression в транспорте кампаний с типом mobile_content")
@Features({FeatureNames.CAMPAIGNS, FeatureNames.NOT_FOR_FULL_EXPORT})
@RunWith(Parameterized.class)
public class BsTransportOfMobileContentCampaignTargetingTest {

    private static final String login = Logins.LOGIN_TRANSPORT;
    private static final Currency currency = Currency.RUB;
    private static final List<List<BsAtom>> EXPECTED_TARGETING_FOR_SKADNETWORK = singletonList(
            singletonList(
                    new BsAtom(OS_FAMILIES.getBsValue(), OS_FAMILIES.getTargetingOperation(), "3:14000:")
            )
    );
    private static final List<List<BsAtom>> EXPECTED_ANTITARGETING_EXPRESSION = singletonList(
            singletonList(
                    new BsAtom(OS_FAMILIES.getBsValue(), OS_FAMILIES.getFilteringOperation(), "3:14005:")
            )
    );

    private static int shard;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(login);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @BeforeClass
    public static void beforeClass() {
        shard = api.userSteps.clientFakeSteps().getUserShard(login);
    }

    @Parameterized.Parameters(name = "test = {0}")
    public static Collection strategies() {
        Object[][] data = new Object[][]{
                {
                        "Стратегия AVERAGE_CPI отправляется с таргетингом до версии 14.5",
                        new CampaignAddItemMap()
                                .defaultCampaignAddItem()
                                .withMobileAppCampaign(new MobileAppCampaignAddItemMap().withBiddingStrategy(
                                new MobileAppCampaignStrategyAddMap()
                                        .withSearch(
                                                new MobileAppCampaignSearchStrategyAddMap()
                                                        .defaultAverageCpi(currency)
                                        ).withNetwork(
                                        new MobileAppCampaignNetworkStrategyAddMap()
                                                .defaultServingOff()
                                )
                                )
                        ),
                        emptyList(),
                        beanDiffer(EXPECTED_ANTITARGETING_EXPRESSION)
                },
                {
                        "Стратегия WB_MAXIMUM_APP_INSTALLS отправляется с таргетингом до версии 14.5",
                        new CampaignAddItemMap()
                                .defaultCampaignAddItem()
                                .withMobileAppCampaign(new MobileAppCampaignAddItemMap().withBiddingStrategy(
                                new MobileAppCampaignStrategyAddMap()
                                        .withSearch(
                                                new MobileAppCampaignSearchStrategyAddMap()
                                                        .defaultWbMaximumAppInstalls(currency)
                                        ).withNetwork(
                                        new MobileAppCampaignNetworkStrategyAddMap()
                                                .defaultServingOff()
                                )
                                )
                        ),
                        emptyList(),
                        beanDiffer(EXPECTED_ANTITARGETING_EXPRESSION)
                },
                {
                        "Стратегия PAY_FOR_INSTALL отправляется с таргетингом до версии 14.5",
                        new CampaignAddItemMap()
                                .defaultCampaignAddItem()
                                .withMobileAppCampaign(new MobileAppCampaignAddItemMap().withBiddingStrategy(
                                        new MobileAppCampaignStrategyAddMap()
                                                .withSearch(
                                                        new MobileAppCampaignSearchStrategyAddMap()
                                                                .withBiddingStrategyType(
                                                                        MobileAppCampaignSearchStrategyTypeEnum.PAY_FOR_INSTALL
                                                                ).defaultAverageCpi(currency)
                                                ).withNetwork(new MobileAppCampaignNetworkStrategyAddMap().defaultServingOff())
                                )
                        ),
                        emptyList(),
                        beanDiffer(EXPECTED_ANTITARGETING_EXPRESSION)
                },
                {
                        "Не конверсионная стратегия без показов на ios >= 14.5 и без SkAdNetwork отправляется с таргетингом до версии 14.5",
                        new CampaignAddItemMap().defaultCampaignAddItem().withDefaultMobileAppCampaign(),
                        emptyList(),
                        beanDiffer(EXPECTED_ANTITARGETING_EXPRESSION)
                },
                {
                        "Не конверсионная стратегия без показов на ios >= 14.5 и с опцией SkAdNetwork отправляется с таргетингом на версии 14.0+",
                        new CampaignAddItemMap().defaultCampaignAddItem().withDefaultMobileAppCampaign(),
                        singletonList(JooqCampaignsOpts.IS_SKADNETWORK_ENABLED),
                        beanDiffer(EXPECTED_TARGETING_FOR_SKADNETWORK)
                },
                {
                        "Не конверсионная стратегия с показами на ios >= 14.5 и с опцией SkAdNetwork отправляется с таргетингом на версии 14.0+",
                        new CampaignAddItemMap().defaultCampaignAddItem().withDefaultMobileAppCampaign(),
                        Arrays.asList(JooqCampaignsOpts.IS_NEW_IOS_VERSION_ENABLED, JooqCampaignsOpts.IS_SKADNETWORK_ENABLED),
                        beanDiffer(EXPECTED_TARGETING_FOR_SKADNETWORK)
                },
                {
                        "Не конверсионная стратегия с показами на ios >= 14.5 и без опции SkAdNetwork отправляется без TargetingExpression",
                        new CampaignAddItemMap().defaultCampaignAddItem().withDefaultMobileAppCampaign(),
                        singletonList(JooqCampaignsOpts.IS_NEW_IOS_VERSION_ENABLED),
                        Matchers.nullValue()
                },
        };
        return Arrays.asList(data);
    }

    @Parameterized.Parameter(value = 0)
    public String description;
    @Parameterized.Parameter(value = 1)
    public CampaignAddItemMap campaignAddItemMap;
    @Parameterized.Parameter(value = 2)
    public List<JooqCampaignsOpts> campaignsOpts;
    @Parameterized.Parameter(value = 3)
    public Matcher expectedTargetingExpression;

    @Test
    public void shouldSendTargetExpression() {
        Long cid = api.userSteps.campaignSteps().addCampaign(campaignAddItemMap);
        api.userSteps.campaignFakeSteps().makeNewCampaignReadyForSendingToBS(cid);
        api.userSteps.getDirectJooqDbSteps().useShard(shard).campaignsSteps().setOpts(cid, campaignsOpts);

        RunBsTransportScriptResponse resp =
                api.userSteps.getDarkSideSteps().getTransportSteps().sendNewCampaign(shard, cid);
        Campaign campaign =
                api.userSteps.getDarkSideSteps().getTransportSteps().getClientDataRequestCampaign(resp, cid);

        assertThat("отправляется правильное значение TargetingExpression",
                campaign.getTargetingExpression(), expectedTargetingExpression);
    }
}
