package ru.yandex.autotests.directintapi.bstransport.main.campaign.parameters.s2stracking;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableMap;
import org.apache.commons.lang3.RandomStringUtils;
import org.hamcrest.Matcher;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.CampaignsType;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.MobileAppsStoreType;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.MobileContentOsType;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.MobileAppsRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.direct.db.steps.JooqCampaignsOpts;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.darkside.model.RunBsTransportScriptResponse;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.Campaign;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignNetworkStrategyAddMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignSearchStrategyAddMap;
import ru.yandex.autotests.directapi.model.campaigns.MetrikaGoals;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directintapi.bstransport.FeatureNames;
import ru.yandex.autotests.directintapi.bstransport.MobileContentCleanupRule;
import ru.yandex.autotests.irt.testutils.RandomUtils;
import ru.yandex.autotests.irt.testutils.json.JsonUtils;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Title;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static java.util.Arrays.asList;
import static java.util.Collections.singletonList;
import static org.apache.commons.lang3.RandomStringUtils.randomNumeric;
import static org.hamcrest.Matchers.allOf;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasEntry;
import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.Matchers.nullValue;
import static org.hamcrest.Matchers.startsWith;
import static ru.yandex.autotests.direct.db.steps.MobileAppSteps.getDefaultMobileAppsRecord;
import static ru.yandex.autotests.directapi.darkside.model.MobileContentUtils.getDefaultMobileContent;
import static ru.yandex.autotests.directapi.model.api5.campaigns.StrategyPayForConversionCrrAddMap.MEANINGFUL_GOAL_OPTIMIZATION_GOAL_ID;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

@Aqua.Test
@Tag(TagDictionary.RELEASE)
@Title("Транспорт: отправка s2s трекерных ссылок")
@Features(FeatureNames.CAMPAIGNS)
@RunWith(Parameterized.class)
public class BsTransportS2sTrackingTest {
    private static final String LOGIN = "at-campaigns-client";
    private static final Long goalId = MetrikaGoals.getGoalsForLogin(LOGIN, 1).get(0);

    private static final String IOS_STORE_CONTENT_ID = "id" + randomNumeric(10);
    private static final String ANDROID_STORE_CONTENT_ID =
            "ru.yandex.autotests." + RandomStringUtils.randomAlphabetic(2) + RandomUtils.getString(8);

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);
    private static final DirectJooqDbSteps jooqDbSteps = api.userSteps.getDirectJooqDbSteps().useShardForLogin(LOGIN);
    private static final Long clientId = jooqDbSteps.shardingSteps().getClientIdByLogin(LOGIN);


    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();
    @ClassRule
    public static MobileContentCleanupRule mobileCleanupRule = new MobileContentCleanupRule(api);


    private static int shard;
    private static Long iosGoalId;
    private static Long androidGoalId;

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameter
    public String description;

    @Parameterized.Parameter(1)
    public CampaignsType campaignsType;

    @Parameterized.Parameter(2)
    public Boolean s2sTrackingEnabled;

    @Parameterized.Parameter(3)
    public Supplier<TextCampaignSearchStrategyAddMap> strategySupplier;

    @Parameterized.Parameter(4)
    public Supplier<List<Long>> meaningfulGoalsSupplier;

    @SuppressWarnings("rawtypes")
    @Parameterized.Parameter(5)
    public Function<Long, Matcher> matcherCreator;

    @SuppressWarnings("rawtypes")
    @Parameterized.Parameters(name = "{0} s2sTrackingEnabled={2}")
    public static Collection testData() {
        Object[][] data = new Object[][]{
                {"Текстовая, стратегия без цели (не конверсионная)", CampaignsType.text, true,
                        (Supplier<TextCampaignSearchStrategyAddMap>) () -> new TextCampaignSearchStrategyAddMap()
                                .defaultAverageCpc(Currency.RUB),
                        (Supplier<List<Long>>) Collections::emptyList,
                        (Function<Long, Matcher>) (cid) -> nullValue()
                },
                {"ДО, стратегия без цели (не конверсионная)", CampaignsType.dynamic, true,
                        (Supplier<TextCampaignSearchStrategyAddMap>) () -> new TextCampaignSearchStrategyAddMap()
                                .defaultAverageCpc(Currency.RUB),
                        (Supplier<List<Long>>) Collections::emptyList,
                        (Function<Long, Matcher>) (cid) -> nullValue()
                },
                {"Смарт, стратегия без цели (не конверсионная)", CampaignsType.performance, true,
                        (Supplier<TextCampaignSearchStrategyAddMap>) () -> new TextCampaignSearchStrategyAddMap()
                                .defaultAverageCpc(Currency.RUB),
                        (Supplier<List<Long>>) Collections::emptyList,
                        (Function<Long, Matcher>) (cid) -> nullValue()
                },

                {"Текстовая, конверсионная стратегия с обычной целью", CampaignsType.text, true,
                        (Supplier<TextCampaignSearchStrategyAddMap>) () -> new TextCampaignSearchStrategyAddMap()
                                .defaultAverageCpa(Currency.RUB, goalId),
                        (Supplier<List<Long>>) Collections::emptyList,
                        (Function<Long, Matcher>) (cid) -> nullValue()
                },
                {"ДО, конверсионная стратегия с обычной целью", CampaignsType.dynamic, true,
                        (Supplier<TextCampaignSearchStrategyAddMap>) () -> new TextCampaignSearchStrategyAddMap()
                                .defaultAverageCpa(Currency.RUB, goalId),
                        (Supplier<List<Long>>) Collections::emptyList,
                        (Function<Long, Matcher>) (cid) -> nullValue()
                },
                {"Смарт, конверсионная стратегия с обычной целью", CampaignsType.performance, true,
                        (Supplier<TextCampaignSearchStrategyAddMap>) () -> new TextCampaignSearchStrategyAddMap()
                                .defaultAverageCpa(Currency.RUB, goalId),
                        (Supplier<List<Long>>) Collections::emptyList,
                        (Function<Long, Matcher>) (cid) -> nullValue()
                },

                {"Текстовая, конверсионная стратегия с обычной целью в ключевых целях", CampaignsType.text, true,
                        (Supplier<TextCampaignSearchStrategyAddMap>) () -> new TextCampaignSearchStrategyAddMap()
                                .defaultAverageCpa(Currency.RUB, MEANINGFUL_GOAL_OPTIMIZATION_GOAL_ID),
                        (Supplier<List<Long>>) () -> singletonList(goalId),
                        (Function<Long, Matcher>) (cid) -> nullValue()
                },
                {"ДО, конверсионная стратегия с обычной целью в ключевых целях", CampaignsType.dynamic, true,
                        (Supplier<TextCampaignSearchStrategyAddMap>) () -> new TextCampaignSearchStrategyAddMap()
                                .defaultAverageCpa(Currency.RUB, MEANINGFUL_GOAL_OPTIMIZATION_GOAL_ID),
                        (Supplier<List<Long>>) () -> singletonList(goalId),
                        (Function<Long, Matcher>) (cid) -> nullValue()
                },
                {"Смарт, конверсионная стратегия с обычной целью в ключевых целях", CampaignsType.performance, true,
                        (Supplier<TextCampaignSearchStrategyAddMap>) () -> new TextCampaignSearchStrategyAddMap()
                                .defaultAverageCpa(Currency.RUB, MEANINGFUL_GOAL_OPTIMIZATION_GOAL_ID),
                        (Supplier<List<Long>>) () -> singletonList(goalId),
                        (Function<Long, Matcher>) (cid) -> nullValue()
                },

                {"Текстовая, конверсионная стратегия с мобильной целью", CampaignsType.text, true,
                        (Supplier<TextCampaignSearchStrategyAddMap>) () -> new TextCampaignSearchStrategyAddMap()
                                .defaultAverageCpa(Currency.RUB, iosGoalId),
                        (Supplier<List<Long>>) Collections::emptyList,
                        (Function<Long, Matcher>) (cid) -> hasEntry(equalTo("IOS"),
                                startsWith(createIosTrackingUrlPrefix(cid)))
                },
                {"ДО, конверсионная стратегия с мобильной целью", CampaignsType.dynamic, true,
                        (Supplier<TextCampaignSearchStrategyAddMap>) () -> new TextCampaignSearchStrategyAddMap()
                                .defaultAverageCpa(Currency.RUB, iosGoalId),
                        (Supplier<List<Long>>) Collections::emptyList,
                        (Function<Long, Matcher>) (cid) -> hasEntry(equalTo("IOS"),
                                startsWith(createIosTrackingUrlPrefix(cid)))
                },
                {"Смарт, конверсионная стратегия с мобильной целью", CampaignsType.performance, true,
                        (Supplier<TextCampaignSearchStrategyAddMap>) () -> new TextCampaignSearchStrategyAddMap()
                                .defaultAverageCpa(Currency.RUB, androidGoalId),
                        (Supplier<List<Long>>) Collections::emptyList,
                        (Function<Long, Matcher>) (cid) -> hasEntry(equalTo("Android"),
                                startsWith(createAndroidTrackingUrlPrefix(cid)))
                },

                {"Текстовая, конверсионная стратегия с мобильной целью в ключевых целях", CampaignsType.text, true,
                        (Supplier<TextCampaignSearchStrategyAddMap>) () -> new TextCampaignSearchStrategyAddMap()
                                .defaultAverageCpa(Currency.RUB, MEANINGFUL_GOAL_OPTIMIZATION_GOAL_ID),
                        (Supplier<List<Long>>) () -> asList(iosGoalId, androidGoalId),
                        (Function<Long, Matcher>) (cid) -> allOf(
                                hasEntry(equalTo("Android"), startsWith(createAndroidTrackingUrlPrefix(cid))),
                                hasEntry(equalTo("IOS"), startsWith(createIosTrackingUrlPrefix(cid)))
                        )
                },
                {"ДО, конверсионная стратегия с мобильной целью в ключевых целях", CampaignsType.dynamic, true,
                        (Supplier<TextCampaignSearchStrategyAddMap>) () -> new TextCampaignSearchStrategyAddMap()
                                .defaultAverageCpa(Currency.RUB, MEANINGFUL_GOAL_OPTIMIZATION_GOAL_ID),
                        (Supplier<List<Long>>) () -> asList(iosGoalId, androidGoalId),
                        (Function<Long, Matcher>) (cid) -> allOf(
                                hasEntry(equalTo("Android"), startsWith(createAndroidTrackingUrlPrefix(cid))),
                                hasEntry(equalTo("IOS"), startsWith(createIosTrackingUrlPrefix(cid)))
                        )
                },
                {"Смарт, конверсионная стратегия с мобильной целью в ключевых целях", CampaignsType.performance, true,
                        (Supplier<TextCampaignSearchStrategyAddMap>) () -> new TextCampaignSearchStrategyAddMap()
                                .defaultAverageCpa(Currency.RUB, MEANINGFUL_GOAL_OPTIMIZATION_GOAL_ID),
                        (Supplier<List<Long>>) () -> asList(iosGoalId, androidGoalId),
                        (Function<Long, Matcher>) (cid) -> allOf(
                                hasEntry(equalTo("Android"), startsWith(createAndroidTrackingUrlPrefix(cid))),
                                hasEntry(equalTo("IOS"), startsWith(createIosTrackingUrlPrefix(cid)))
                        )
                },
        };
        return asList(data);
    }

    private Long cid;

    @BeforeClass
    public static void beforeClass() {
        jooqDbSteps.featuresSteps().ensureClientHasFeature(clientId, "in_app_mobile_targeting");
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
        shard = api.userSteps.clientFakeSteps().getUserShard(LOGIN);

        Long iosMobileContentId = jooqDbSteps.mobileContentSteps().saveMobileContent(
                getDefaultMobileContent(clientId.toString())
                        .setStoreContentId(IOS_STORE_CONTENT_ID)
                        .setOsType(MobileContentOsType.iOS)
        );
        MobileAppsRecord iosMobileApp = getDefaultMobileAppsRecord(clientId, iosMobileContentId)
                .setStoreType(MobileAppsStoreType.AppleAppStore);
        Long iosMobileAppId = jooqDbSteps.mobileAppsSteps().createMobileApp(iosMobileApp);
        iosGoalId = jooqDbSteps.mobileAppsSteps().addMobileAppGoal(iosMobileAppId);

        Long androidMobileContentId = jooqDbSteps.mobileContentSteps().saveMobileContent(
                getDefaultMobileContent(clientId.toString())
                        .setStoreContentId(ANDROID_STORE_CONTENT_ID)
                        .setOsType(MobileContentOsType.Android)
        );
        MobileAppsRecord androidMobileApp = getDefaultMobileAppsRecord(clientId, androidMobileContentId)
                .setStoreType(MobileAppsStoreType.GooglePlayStore);
        Long androidMobileAppId = jooqDbSteps.mobileAppsSteps().createMobileApp(androidMobileApp);
        androidGoalId = jooqDbSteps.mobileAppsSteps().addMobileAppGoal(androidMobileAppId);

        mobileCleanupRule.setShard(shard);
        mobileCleanupRule.setMobileContentIds(asList(iosMobileContentId, androidMobileContentId));
        mobileCleanupRule.setMobileAppIdsToDeleteAfterTest(clientId, asList(iosMobileAppId, androidMobileAppId));
    }

    @Before
    public void before() {
        cid = api.userSteps.campaignSteps().addDefaultTextCampaignWithStrategies(
                strategySupplier.get(), new TextCampaignNetworkStrategyAddMap().defaultNetworkDefault());
        jooqDbSteps.campaignsSteps().setType(cid, campaignsType);
        jooqDbSteps.campaignsSteps().setOpts(cid, singletonList(JooqCampaignsOpts.S2S_TRACKING_ENABLED));
        List<Map<String, ?>> meaningfulGoals = meaningfulGoalsSupplier.get().stream()
                .map(goal_id -> ImmutableMap.of("goal_id", goal_id, "value", 33))
                .collect(Collectors.toList());
        jooqDbSteps.campaignsSteps().setCampaignMeaningfulGoals(cid, JsonUtils.toString(meaningfulGoals));

        api.userSteps.campaignFakeSteps().makeNewCampaignReadyForSendingToBS(cid);
    }

    @Test
    public void testS2sTrackingUrlTransport() {
        RunBsTransportScriptResponse resp =
                api.userSteps.getDarkSideSteps().getTransportSteps().sendNewCampaign(shard, cid);
        Campaign campaign =
                api.userSteps.getDarkSideSteps().getTransportSteps().getClientDataRequestCampaign(resp, cid);
        assumeThat("кампания отправляется в БК", campaign, notNullValue());
        //noinspection unchecked
        assertThat("значение S2STrackingURLs отправляется верным",
                campaign.getS2sTrackingUrls(), matcherCreator.apply(cid));
    }

    private static String createIosTrackingUrlPrefix(Long cid) {
        return "https://app.appsflyer.com/" + IOS_STORE_CONTENT_ID + "?is_retargeting" +
                "=true&idfa={IDFA_UC}&af_c_id=" + cid + "&clickid={TRACKID" +
                "}&pid=yandexdirect_int&af_ip={CLIENTIP}&af_lang={DEVICE_LANG}&af_ua" +
                "={USER_AGENT}&redirect=false";
    }

    private static String createAndroidTrackingUrlPrefix(Long cid) {
        return "https://app.appsflyer.com/" + ANDROID_STORE_CONTENT_ID +
                "?is_retargeting=true&advertising_id={GOOGLE_AID_LC}&oaid={OAID_LC}&af_c_id=" + cid + "&clickid" +
                "={TRACKID}&pid=yandexdirect_int&af_ip={CLIENTIP}&af_lang={DEVICE_LANG}&af_ua={USER_AGENT}&redirect" +
                "=false&af_ref=YandexDirectInt_{TRACKID}";
    }
}
