package ru.yandex.autotests.directintapi.bstransport.main.cpmbanner.transmit;

import java.util.List;

import javax.annotation.Nullable;

import com.google.common.collect.ImmutableList;
import org.hamcrest.Matchers;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersPerformanceStatusmoderate;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.HierarchicalMultipliersType;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.PerfCreativesStatusmoderate;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.ExpressionMultiplierValuesRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.darkside.Logins;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.displaycanvas.ModerationInfo;
import ru.yandex.autotests.directapi.darkside.exceptions.DarkSideException;
import ru.yandex.autotests.directapi.darkside.model.BidMultiplierConditionBuilder;
import ru.yandex.autotests.directapi.darkside.model.BidMultiplierUtils;
import ru.yandex.autotests.directapi.darkside.model.RunBsTransportScriptResponse;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.Campaign;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.Context;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.ExpressionCoefItem;
import ru.yandex.autotests.directapi.darkside.model.multipliers.DirectAtom;
import ru.yandex.autotests.directapi.darkside.model.multipliers.ExpressionParameter;
import ru.yandex.autotests.directapi.darkside.model.multipliers.Operation;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directintapi.bstransport.FeatureNames;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.allure.annotations.Title;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static java.util.Collections.singletonList;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.Matchers.notNullValue;
import static org.junit.Assert.fail;
import static ru.yandex.autotests.directapi.darkside.model.BidMultiplierUtils.directAtom;
import static ru.yandex.autotests.directapi.darkside.model.BidMultiplierUtils.serializeToDirectFormat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;
import static ru.yandex.autotests.irt.testutils.beandiffer2.BeanDifferMatcher.beanDiffer;


@Aqua.Test
@Tag(TagDictionary.RELEASE)
@Tag(TagDictionary.TRUNK)
@Description("Проверяем отправку в БК универсальной корректировки")
@Features(FeatureNames.CPM_BANNER)
@Issue("https://st.yandex-team.ru/DIRECT-101878")
public class SendingExpressionBidModifierTest {

    private static final String LOGIN = Logins.LOGIN_TRANSPORT;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private Long cid;
    private Long pid;
    private Long bid;
    private DirectJooqDbSteps jooqDbSteps;

    private static int shard;

    static private Integer MULTIPLIER_PCT1 = 200;
    static private Integer MULTIPLIER_PCT2 = 1200;

    @BeforeClass
    public static void configTestClass() {
        shard = api.userSteps.getDarkSideSteps().getClientFakeSteps().getUserShard(LOGIN);
    }

    @Before
    @Step("Подготовка тестовых данных")
    public void init() {
        shard = api.userSteps.clientFakeSteps().getUserShard(LOGIN);
        jooqDbSteps = api.userSteps.getDirectJooqDbSteps().useShard(shard);

        Long clientId = Long.valueOf(api.userSteps.clientFakeSteps().getClientData(LOGIN).getClientID());
        Long creativeId = jooqDbSteps.perfCreativesSteps().saveDefaultCanvasCreativesForClient(clientId);

        cid = api.userSteps.campaignSteps().addDefaultCpmBannerCampaign(LOGIN);
        //создаем группу cpm-banner с ключевыми словами
        pid = api.userSteps.adGroupsSteps().addDefaultCpmBannerKeywordsAdGroup(cid);
        List<Long> keywordIds = api.userSteps.keywordsSteps().addDefaultKeywords(pid, 1);
        bid = api.userSteps.adsSteps().addDefaultCpmBannerAdBuilderAd(pid, creativeId);

        api.userSteps.campaignFakeSteps().makeNewCampaignReadyForSendingToBS(cid);
        api.userSteps.groupFakeSteps().makeGroupFullyModerated(pid);
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(bid);
        api.userSteps.phrasesFakeSteps().makeKeywordModerated(keywordIds.get(0));

        jooqDbSteps.bannersPerformanceSteps()
                .setCreativeStatusModerate(cid, pid, bid, BannersPerformanceStatusmoderate.Yes);
        jooqDbSteps.perfCreativesSteps().setModerateInfo(creativeId,
                new ModerationInfo().generateTestData().toString());
        jooqDbSteps.perfCreativesSteps().setStatusModerate(creativeId, PerfCreativesStatusmoderate.Yes);
    }

    @Test
    @Title("Корректировка универсальная для кампании. Одна корректировка")
    public void checkCampaignExpressionMultiplier_SingleCoef() {
        List<List<DirectAtom>> initialExpression = createDefaultExpressMultiplierCondition();

        ExpressionMultiplierValuesRecord expressMultiplier = new ExpressionMultiplierValuesRecord()
                .setConditionJson(serializeToDirectFormat(initialExpression))
                .setMultiplierPct(MULTIPLIER_PCT1.shortValue());

        saveExpressMultipliers(singletonList(expressMultiplier), cid, null);

        Campaign campaign = sendMultiplierToBsAndGetResultCampaign();

        assumeThat("в БК отправлена кампания", campaign, notNullValue());
        assumeThat("Отправлен ExpessionCoefs",
                campaign.getExpressionCoefs(), notNullValue());
        List<ExpressionCoefItem> coefs = campaign.getExpressionCoefs().getTraffic();
        List<List<DirectAtom>> expectedExpression = BidMultiplierUtils.convertExpressionAtoms(initialExpression);

        checkBsExpressMultipliers(coefs,
                ImmutableList.of(MULTIPLIER_PCT1),
                ImmutableList.of(expectedExpression));
    }

    @Test
    @Title("Корректировка универсальная для кампании. Несколько корректировок")
    public void checkCampaignExpressMultiplier_SeveralCoef() {
        List<List<DirectAtom>> initialExpression1 = BidMultiplierConditionBuilder.init()
                .and(directAtom(ExpressionParameter.PREC_STRENGTH, Operation.eq, "50"),
                        directAtom(ExpressionParameter.PREC_STRENGTH, Operation.eq, "75"))
                .and(directAtom(ExpressionParameter.TEMP, Operation.le, "5"))
                .build();

        ExpressionMultiplierValuesRecord multiplier1 = new ExpressionMultiplierValuesRecord()
                .setConditionJson(serializeToDirectFormat(initialExpression1))
                .setMultiplierPct(MULTIPLIER_PCT1.shortValue());

        List<List<DirectAtom>> initialExpression2 = BidMultiplierConditionBuilder.init()
                .and(directAtom(ExpressionParameter.CLOUDNESS, Operation.eq, "50"))
                .build();

        ExpressionMultiplierValuesRecord multiplier2 = new ExpressionMultiplierValuesRecord()
                .setConditionJson(serializeToDirectFormat(initialExpression2))
                .setMultiplierPct(MULTIPLIER_PCT2.shortValue());

        saveExpressMultipliers(ImmutableList.of(multiplier1, multiplier2), cid, null);

        Campaign campaign = sendMultiplierToBsAndGetResultCampaign();

        assumeThat("в БК отправлена кампания", campaign, notNullValue());
        assumeThat("Отправлен ExpessionCoefs", campaign.getExpressionCoefs(), notNullValue());
        List<ExpressionCoefItem> coefs = campaign.getExpressionCoefs().getTraffic();

        List<List<DirectAtom>> expectedExpression1 = BidMultiplierUtils.convertExpressionAtoms(initialExpression1);
        List<List<DirectAtom>> expectedExpression2 = BidMultiplierUtils.convertExpressionAtoms(initialExpression2);

        checkBsExpressMultipliers(coefs,
                ImmutableList.of(MULTIPLIER_PCT1, MULTIPLIER_PCT2),
                ImmutableList.of(expectedExpression1, expectedExpression2));
    }

    @Test
    @Title("Корректировка универсальная для группы. Одна корректировка")
    public void checkAdGroupExpressMultiplier_SingleCoef() {
        List<List<DirectAtom>> initialExpression1 = BidMultiplierConditionBuilder.init()
                .and(directAtom(ExpressionParameter.TEMP, Operation.gt, "1"),
                        directAtom(ExpressionParameter.TEMP, Operation.lt, "20"))
                .and(directAtom(ExpressionParameter.CLOUDNESS, Operation.eq, "25"))
                .build();

        ExpressionMultiplierValuesRecord multiplier1 = new ExpressionMultiplierValuesRecord()
                .setConditionJson(serializeToDirectFormat(initialExpression1))
                .setMultiplierPct(MULTIPLIER_PCT1.shortValue());

        saveExpressMultipliers(singletonList(multiplier1), cid, pid);

        Campaign campaign = sendMultiplierToBsAndGetResultCampaign();
        assumeThat("в БК отправлена кампания", campaign, notNullValue());

        Context context = campaign != null ? campaign.getContext(pid) : null;
        assumeThat("в БК отправлен контекст", context, notNullValue());

        assumeThat("Отправлен ExpessionCoefs", context.getExpressionCoefs(), notNullValue());
        List<ExpressionCoefItem> coefs = context.getExpressionCoefs().getTraffic();

        List<List<DirectAtom>> expectedExpression1 = BidMultiplierUtils.convertExpressionAtoms(initialExpression1);

        checkBsExpressMultipliers(coefs,
                singletonList(MULTIPLIER_PCT1),
                singletonList(expectedExpression1));
    }

    @Test
    @Title("Корректировка универсальная для группы. Несколько корректировок")
    public void checkAdGroupExpressMultiplier_SeveralCoefs() {

        List<List<DirectAtom>> initialExpression1 = BidMultiplierConditionBuilder.init()
                .and(directAtom(ExpressionParameter.TEMP, Operation.gt, "1"),
                        directAtom(ExpressionParameter.TEMP, Operation.lt, "20"))
                .and(directAtom(ExpressionParameter.CLOUDNESS, Operation.eq, "25"))
                .build();

        ExpressionMultiplierValuesRecord multiplier1 = new ExpressionMultiplierValuesRecord()
                .setConditionJson(serializeToDirectFormat(initialExpression1))
                .setMultiplierPct(MULTIPLIER_PCT1.shortValue());

        List<List<DirectAtom>> initialExpression2 = BidMultiplierConditionBuilder.init()
                .and(directAtom(ExpressionParameter.TEMP, Operation.eq, "15"))
                .and(directAtom(ExpressionParameter.PREC_STRENGTH, Operation.eq, "75"))
                .build();

        ExpressionMultiplierValuesRecord multiplier2 = new ExpressionMultiplierValuesRecord()
                .setConditionJson(serializeToDirectFormat(initialExpression2))
                .setMultiplierPct(MULTIPLIER_PCT2.shortValue());

        saveExpressMultipliers(ImmutableList.of(multiplier1, multiplier2), cid, pid);

        Campaign campaign = sendMultiplierToBsAndGetResultCampaign();
        assumeThat("в БК отправлена кампания", campaign, notNullValue());

        Context context = campaign != null ? campaign.getContext(pid) : null;
        assumeThat("в БК отправлен контекст", context, notNullValue());
        assumeThat("Отправлен ExpessionCoefs", context.getExpressionCoefs(), notNullValue());
        List<ExpressionCoefItem> coefs = context.getExpressionCoefs().getTraffic();

        List<List<DirectAtom>> expectedExpression1 = BidMultiplierUtils.convertExpressionAtoms(initialExpression1);
        List<List<DirectAtom>> expectedExpression2 = BidMultiplierUtils.convertExpressionAtoms(initialExpression2);

        checkBsExpressMultipliers(coefs,
                ImmutableList.of(MULTIPLIER_PCT1, MULTIPLIER_PCT2),
                ImmutableList.of(expectedExpression1, expectedExpression2));
    }

    private void saveExpressMultipliers(List<ExpressionMultiplierValuesRecord> multipliers, Long cid,
                                        @Nullable Long pid) {
        api.userSteps.getDirectJooqDbSteps().useShard(shard).multipliersSteps()
                .saveExpressionMultiplierValues(cid, pid, multipliers,
                        HierarchicalMultipliersType.express_traffic_multiplier, true);
    }

    private Campaign sendMultiplierToBsAndGetResultCampaign() {
        api.userSteps.campaignFakeSteps().makeNewCampaignReadyForSendingToBS(cid);
        api.userSteps.groupFakeSteps().makeGroupFullyModerated(pid);

        RunBsTransportScriptResponse resp =
                api.userSteps.getDarkSideSteps().getTransportSteps().sendNewCampaign(shard, cid);

        return api.userSteps.getDarkSideSteps().getTransportSteps().getClientDataRequestCampaign(resp, 0, cid);
    }

    // Просто большой набор разных условий (безотносительно корректности формулы)
    private List<List<DirectAtom>> createDefaultExpressMultiplierCondition() {
        return BidMultiplierConditionBuilder.init()
                .and(directAtom(ExpressionParameter.PREC_STRENGTH, Operation.ge, "75"))
                .and(directAtom(ExpressionParameter.CLOUDNESS, Operation.ge, "50"))
                .and(directAtom(ExpressionParameter.TEMP, Operation.le, "5"))
                .and(directAtom(ExpressionParameter.TEMP, Operation.ge, "-5"))
                .and(directAtom(ExpressionParameter.TRAFFIC_JAM, Operation.eq, "1"))
                .and(directAtom(ExpressionParameter.TRAFFIC_JAM, Operation.ne, "2"))
                .build();
    }

    private void checkBsExpressMultipliers(List<ExpressionCoefItem> coefs, List<Integer> expectedCoefs,
                                           List<List<List<DirectAtom>>> expectedExpressions) {
        assumeThat("Количество ожидаемых значений совпадает с количеством проверяемых объектов",
                coefs.size(), equalTo(expectedCoefs.size()));
        assumeThat("Количество ожидаемых значений совпадает с количеством проверяемых объектов",
                coefs.size(), equalTo(expectedExpressions.size()));

        int exprectedCount = expectedCoefs.size();
        assumeThat("coefs содержит ожидаемое количество корректировок",
                coefs.size(), equalTo(exprectedCount));

        for (int i = 0; i < coefs.size(); i++) {
            assumeThat("Значение Coef соответствует ожидаемому значению",
                    coefs.get(i).getCoef(), equalTo(expectedCoefs.get(i)));

            assertThat("Выражение корректировки соответсвует ожиданиям",
                    coefs.get(i).getExpression(), beanDiffer(expectedExpressions.get(i)));
        }
    }
}
