package ru.yandex.autotests.directintapi.bstransport.main.cpmbanner.transmit;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.stream.Stream;

import org.junit.After;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.cmd.rules.CreativeBannerRule;
import ru.yandex.autotests.direct.cmd.rules.DirectCmdRule;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersBannerType;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersPerformanceStatusmoderate;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.PerfCreativesStatusmoderate;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.PhrasesAdgroupType;
import ru.yandex.autotests.direct.db.models.jooq.ppcdict.tables.records.LalSegmentsRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.direct.db.steps.LalSegmentsSteps;
import ru.yandex.autotests.direct.utils.campaigns.CampaignTypeEnum;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.direct.web.api.core.DirectRule;
import ru.yandex.autotests.direct.web.api.models.Condition;
import ru.yandex.autotests.direct.web.api.models.CryptaGoalWeb;
import ru.yandex.autotests.direct.web.api.models.MetrikaGoalWeb;
import ru.yandex.autotests.direct.web.api.models.RetargetingConditionActionResponse;
import ru.yandex.autotests.direct.web.api.models.RetargetingConditionWeb;
import ru.yandex.autotests.directapi.common.api45.RetargetingGoal;
import ru.yandex.autotests.directapi.darkside.Logins;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.displaycanvas.ModerationInfo;
import ru.yandex.autotests.directapi.darkside.model.CampaignsType;
import ru.yandex.autotests.directapi.darkside.model.RunBsTransportScriptResponse;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.Campaign;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.Context;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.GoalContext;
import ru.yandex.autotests.directapi.model.retargeting.RetargetingGoalType;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directintapi.bstransport.FeatureNames;
import ru.yandex.autotests.directintapi.bstransport.StoriesNames;
import ru.yandex.autotests.directintapi.bstransport.TransportHelpSteps;
import ru.yandex.autotests.irt.testutils.RandomUtils;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.greaterThan;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.core.IsNull.notNullValue;
import static org.junit.Assert.assertThat;
import static org.junit.Assume.assumeThat;
import static ru.yandex.autotests.directintapi.bstransport.TransportUtils.DEFAULT_METRIKA_GOAL_TIME;

@Aqua.Test
@Tag(TagDictionary.RELEASE)
@Tag(TagDictionary.TRUNK)
@Description("Проверяем транспорт сегментов крипты")
@Features({FeatureNames.RETARGETINGS, FeatureNames.CPM_BANNER})
@Issue("https://st.yandex-team.ru/DIRECT-72511")
@Stories(StoriesNames.RETARGETING_ACTION_CREATION)
@RunWith(Parameterized.class)
public class SendingOfCpmBannerRetargetingTest {

    private static final String LOGIN = Logins.LOGIN_TRANSPORT;
    // В первом этапе охватного продукта в интерфейсе будет всего 4е дохода.
    // Но в транспорте к БК нужно отдавать именно Средний доход как два значения.
    // https://wiki.yandex-team.ru/users/aliho/projects/direct/crypta/#diapazondljaobshhixpol/vozrast/doxod
    private static final long AVERAGE_INCOME_CRYPTA_GOAL_ID = 2499000010L;
    private static final long FIRST_INTEREST_CRYPTA_GOAL_ID = 2499001100L;

    @ClassRule
    public static DirectRule directClassRule = DirectRule.defaultClassRule();

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public DirectRule directRule = DirectRule.defaultRule().as(LOGIN);

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static DirectCmdRule defaultClassRule = DirectCmdRule.defaultClassRule();

    public CreativeBannerRule bannersRule = new CreativeBannerRule(CampaignTypeEnum.TEXT).withUlogin(LOGIN);

    @Rule
    public DirectCmdRule cmdRule = DirectCmdRule.defaultRule().withRules(bannersRule);

    private static TransportHelpSteps helpSteps = new TransportHelpSteps(api);

    private static int shard;
    private Long pid;
    private Campaign campaign;
    private int avgIncomeRetId;
    private int singleInterestRetId;
    private int mixedRetId;
    private int emptyGoalsRetId;
    private Long avgIncomeRetCondId;
    private Long signleInterestRetCondId;
    private Long mixedRetCondId;
    private Long emptyGoalsRetCondId;
    private String expectedAvgIncomeExpression;
    private String expectedSingleInterestExpression;
    private String expectedMixedExpression;
    private String expectedEmptyGoalsExpression;

    @Parameterized.Parameter(0)
    public Condition.InterestTypeEnum interestType;

    @Parameterized.Parameters(name = "тип сегмента интересов - {0}")
    public static Collection testData() {
        Object[][] data = new Object[][]{
                {Condition.InterestTypeEnum.ALL},
                {Condition.InterestTypeEnum.SHORT_TERM},
                {Condition.InterestTypeEnum.LONG_TERM},
        };
        return Arrays.asList(data);
    }

    @BeforeClass
    public static void beforeClass() {
        shard = api.userSteps.clientFakeSteps().getUserShard(LOGIN);
    }

    @After
    @Step("Утилизация тестовых данных")
    public void after() {
        Stream.of(avgIncomeRetId, singleInterestRetId, mixedRetId).filter(Objects::nonNull).forEach(
                retId -> api.userSteps.retargetingSteps().retargetingDelete(LOGIN, retId));
        Stream.of(avgIncomeRetCondId, signleInterestRetCondId, mixedRetCondId).filter(Objects::nonNull).forEach(
                retCondId -> directRule.webApiSteps().retargetingSteps()
                        .deleteRetargetingCondition(singletonList(retCondId), LOGIN));
    }

    @Before
    @Step("Подготовка тестовых данных")
    public void init() {
        this.avgIncomeRetCondId = createSingleCryptaSegmentCond(LOGIN, AVERAGE_INCOME_CRYPTA_GOAL_ID);
        assumeThat("добавился ретаргетинг на средний доход", this.avgIncomeRetCondId, is(notNullValue()));

        this.signleInterestRetCondId = createSingleCryptaSegmentCond(LOGIN, FIRST_INTEREST_CRYPTA_GOAL_ID);
        assumeThat("добавился ретаргетинг на сегмент интересов", this.signleInterestRetCondId, is(notNullValue()));

        this.mixedRetCondId = createMixedRetCondition(LOGIN);
        assumeThat("добавился смешанный ретаргетинг", this.mixedRetCondId, is(notNullValue()));

        RetargetingConditionActionResponse emptyGoalsCondition =
                directRule.webApiSteps().retargetingSteps()
                        .createRetargetingCondition(retCondReq(emptyList()), LOGIN);
        this.emptyGoalsRetCondId = emptyGoalsCondition.getId();
        assumeThat("добавился ретаргетинг с пустым списком сегментов", this.emptyGoalsRetCondId, is(notNullValue()));

        shard = api.userSteps.clientFakeSteps().getUserShard(LOGIN);
        Long cid = bannersRule.getCampaignId();
        Long creativeId = bannersRule.getCreativeId();
        Long bid = bannersRule.getBannerId();
        pid = bannersRule.getGroupId();
        Long keywordId = bannersRule.getCurrentGroup().getPhrases().get(0).getId();

        api.userSteps.campaignFakeSteps().makeNewCampaignReadyForSendingToBS(cid);
        api.userSteps.groupFakeSteps().makeGroupFullyModerated(pid);
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(bid);
        api.userSteps.phrasesFakeSteps().makeKeywordModerated(keywordId);

        avgIncomeRetId = api.userSteps.retargetingSteps().addRetargetingToBanner(LOGIN, bid, avgIncomeRetCondId.intValue());
        singleInterestRetId = api.userSteps.retargetingSteps().addRetargetingToBanner(LOGIN, bid, signleInterestRetCondId.intValue());
        mixedRetId = api.userSteps.retargetingSteps().addRetargetingToBanner(LOGIN, bid, mixedRetCondId.intValue());
        emptyGoalsRetId = api.userSteps.retargetingSteps().addRetargetingToBanner(LOGIN, bid, emptyGoalsRetCondId.intValue());

        helpSteps.useDirectWebApiSteps(directRule.webApiSteps());
        expectedAvgIncomeExpression = helpSteps.getRetConditionsBSStyle(avgIncomeRetCondId);
        expectedSingleInterestExpression = helpSteps.getRetConditionsBSStyle(signleInterestRetCondId);
        expectedMixedExpression = helpSteps.getRetConditionsBSStyle(mixedRetCondId);
        expectedEmptyGoalsExpression = helpSteps.getRetConditionsBSStyle(emptyGoalsRetCondId);

        DirectJooqDbSteps jooqDbSteps = api.userSteps.getDirectJooqDbSteps().useShard(shard);
        jooqDbSteps.bannersPerformanceSteps()
                .setCreativeStatusModerate(cid, pid, bid, BannersPerformanceStatusmoderate.Yes);
        jooqDbSteps.perfCreativesSteps().setModerateInfo(creativeId,
                new ModerationInfo().generateTestData().toString());
        jooqDbSteps.perfCreativesSteps().setStatusModerate(creativeId, PerfCreativesStatusmoderate.Yes);
        api.userSteps.campaignFakeSteps().setType(cid, CampaignsType.CPM_BANNER);
        jooqDbSteps.adGroupsSteps().setType(pid, PhrasesAdgroupType.cpm_banner);
        jooqDbSteps.bannersSteps().setBannersBannerType(bid, BannersBannerType.cpm_banner);
        RunBsTransportScriptResponse resp
                = api.userSteps.getDarkSideSteps().getTransportSteps().sendNewCampaign(shard, cid);
        campaign =
                api.userSteps.getDarkSideSteps().getTransportSteps().getClientDataRequestCampaign(resp, 0, cid);
    }

    @Step("Подготовка правила ретаргетинга на цель метрики и на сегменты крипты (goalIDs = ["
            + AVERAGE_INCOME_CRYPTA_GOAL_ID + ", " + FIRST_INTEREST_CRYPTA_GOAL_ID + "]) для пользователя {0}")
    public Long createMixedRetCondition(String login) {
        List<RetargetingGoal> segmentGoals
                = api.userSteps.retargetingSteps().getRetargetingGoals(RetargetingGoalType.SEGMENT, login);
        assumeThat("кол-во целей типа сегмент > 0", segmentGoals.size(), greaterThan(0));

        final RetargetingGoal goal = segmentGoals.get(RandomUtils.getRandomInteger(0, segmentGoals.size() - 1));

        long metrikaGoalID = goal.getGoalID();
        final Condition metrikaCond = getMetrikaCond(metrikaGoalID);

        final LalSegmentsRecord lalSegment;

        LalSegmentsSteps lalSegmentsSteps = api.userSteps.getDirectJooqDbSteps().lalSegmentsSteps();

        LalSegmentsRecord foundLalSegment = lalSegmentsSteps.findLalSegmentByParent(metrikaGoalID);
        if (foundLalSegment == null) {
            lalSegmentsSteps.addOrUpdateLalSegment(new LalSegmentsRecord()
                    .setParentGoalId(metrikaGoalID)
                    .setIsActive(1));
            lalSegment = lalSegmentsSteps.findLalSegmentByParent(metrikaGoalID);
        } else {
            lalSegment = foundLalSegment;
        }

        final Condition metrikaCondWithUnion = getMetrikaCondWithUnion(metrikaGoalID, lalSegment.getLalSegmentId());

        RetargetingConditionActionResponse mixedResult = directRule.webApiSteps().retargetingSteps()
                .createRetargetingCondition(
                        retCondReq(Arrays.asList(
                                getSingleCryptaSegmentCond(AVERAGE_INCOME_CRYPTA_GOAL_ID),
                                getSingleCryptaSegmentCond(FIRST_INTEREST_CRYPTA_GOAL_ID),
                                metrikaCond,
                                metrikaCondWithUnion)),
                        login);
        return mixedResult.getId();
    }

    @Step("Подготовка правила ретаргетинга на сегмент крипты goalID = {1} для пользователя {0}")
    public Long createSingleCryptaSegmentCond(String login, long cryptaGoalId) {
        final RetargetingConditionWeb retargetingConditionWeb =
                retCondReq(singletonList(getSingleCryptaSegmentCond(cryptaGoalId)));
        RetargetingConditionActionResponse avgIncomeResult =
                directRule.webApiSteps().retargetingSteps()
                        .createRetargetingCondition(retargetingConditionWeb, login);
        return avgIncomeResult.getId();
    }

    public Condition getMetrikaCond(long goalID) {
        return new Condition()
                .withGoals(singletonList(
                        new MetrikaGoalWeb()
                                .withTime(DEFAULT_METRIKA_GOAL_TIME)
                                .withId(goalID)
                ))
                .withType(Condition.TypeEnum.ALL);
    }

    public Condition getMetrikaCondWithUnion(long goalId, long goalIdWithUnion) {
        return new Condition()
                .withGoals(Arrays.asList(
                        new MetrikaGoalWeb()
                                .withTime(DEFAULT_METRIKA_GOAL_TIME)
                                .withId(goalId),
                        new MetrikaGoalWeb()
                                .withTime(DEFAULT_METRIKA_GOAL_TIME)
                                .withId(goalIdWithUnion)
                                .withUnionWithId(goalId))
                )
                .withType(Condition.TypeEnum.ALL);
    }

    public Condition getSingleCryptaSegmentCond(long cryptaGoalId) {
        return new Condition()
                .withInterestType(interestType)
                .withGoals(singletonList(new CryptaGoalWeb()
                        .withId(cryptaGoalId)
                ))
                .withType(Condition.TypeEnum.OR); // only OR type is allowed for crypta conditions
    }

    public RetargetingConditionWeb retCondReq(List<Condition> conditions) {
        return new RetargetingConditionWeb()
                .withName("testName" + System.currentTimeMillis())
                .withDescription("testDescription")
                .withType(RetargetingConditionWeb.TypeEnum.INTERESTS)
                .withConditions(conditions);
    }

    @Test
    public void averageIncomeSegmentExpressionShouldIncludeTwoBBKeywords() {
        Context context = campaign.getContext(pid);
        assumeThat("в запросе есть группа", context, notNullValue());

        final GoalContext goalContext = context.getGoalContext((long) avgIncomeRetId);
        assumeThat("в запросе есть ретаргетинг", goalContext, notNullValue());

        assertThat("передан корректный GoalContext->{Expression}", goalContext.getExpression(),
                equalTo(expectedAvgIncomeExpression));
    }

    @Test
    public void singleInterestSegmentExpressionShouldTakeInAccountBothShortAndLongTermKeywords() {
        Context context = campaign.getContext(pid);
        assumeThat("в запросе есть группа", context, notNullValue());

        final GoalContext goalContext = context.getGoalContext((long) singleInterestRetId);
        assumeThat("в запросе есть ретаргетинг", goalContext, notNullValue());

        assertThat("передан корректный GoalContext->{Expression}", goalContext.getExpression(),
                equalTo(expectedSingleInterestExpression));
    }

    @Test
    public void metrikaGoalsShouldGoFirstInMixedCryptaAndMetrikaExpression() {
        Context context = campaign.getContext(pid);
        assumeThat("в запросе есть группа", context, notNullValue());

        final GoalContext mixedGoalContext = context.getGoalContext((long) mixedRetId);
        assumeThat("в запросе есть ретаргетинг", mixedGoalContext, notNullValue());

        assertThat("передан корректный GoalContext->{Expression}", mixedGoalContext.getExpression(),
                equalTo(expectedMixedExpression));
    }

    @Test
    public void specialKeywordShouldBeSendOnEmptyGoalsCryptaExpression() {
        Context context = campaign.getContext(pid);
        assumeThat("в запросе есть группа", context, notNullValue());

        final GoalContext emptyGoalContext = context.getGoalContext((long) emptyGoalsRetId);
        assumeThat("в запросе есть ретаргетинг", emptyGoalContext, notNullValue());

        assertThat("передан корректный GoalContext->{Expression}", emptyGoalContext.getExpression(),
                equalTo(expectedEmptyGoalsExpression));
    }
}
