package ru.yandex.autotests.directintapi.bstransport.main.cpmbanner.transmit;

import java.util.List;

import javax.annotation.Nullable;

import com.google.common.collect.ImmutableList;
import org.hamcrest.Matchers;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannersPerformanceStatusmoderate;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.PerfCreativesStatusmoderate;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.WeatherMultiplierValuesRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.darkside.Logins;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.displaycanvas.ModerationInfo;
import ru.yandex.autotests.directapi.darkside.exceptions.DarkSideException;
import ru.yandex.autotests.directapi.darkside.model.BidMultiplierConditionBuilder;
import ru.yandex.autotests.directapi.darkside.model.BidMultiplierUtils;
import ru.yandex.autotests.directapi.darkside.model.RunBsTransportScriptResponse;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.Campaign;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.Context;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.WeatherCoef;
import ru.yandex.autotests.directapi.darkside.model.multipliers.DirectAtom;
import ru.yandex.autotests.directapi.darkside.model.multipliers.ExpressionParameter;
import ru.yandex.autotests.directapi.darkside.model.multipliers.Operation;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directintapi.bstransport.FeatureNames;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.allure.annotations.Title;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static java.util.Collections.singletonList;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.Matchers.notNullValue;
import static org.junit.Assert.fail;
import static ru.yandex.autotests.directapi.darkside.model.BidMultiplierUtils.directAtom;
import static ru.yandex.autotests.directapi.darkside.model.BidMultiplierUtils.serializeToDirectFormat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;
import static ru.yandex.autotests.irt.testutils.beandiffer2.BeanDifferMatcher.beanDiffer;


@Aqua.Test
@Tag(TagDictionary.RELEASE)
@Tag(TagDictionary.TRUNK)
@Description("Проверяем отправку в БК корректировки по погодным условиям")
@Features(FeatureNames.CPM_BANNER)
@Issue("https://st.yandex-team.ru/DIRECT-92304")
public class SendingWeatherBidModifierTest {

    private static final String LOGIN = Logins.LOGIN_TRANSPORT;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private Long cid;
    private Long pid;
    private Long bid;
    private DirectJooqDbSteps jooqDbSteps;

    private static int shard;

    static private Integer MULTIPLIER_PCT1 = 200;
    static private Integer MULTIPLIER_PCT2 = 1200;

    @BeforeClass
    public static void configTestClass() {
        shard = api.userSteps.getDarkSideSteps().getClientFakeSteps().getUserShard(LOGIN);
    }

    @Before
    @Step("Подготовка тестовых данных")
    public void init() {
        shard = api.userSteps.clientFakeSteps().getUserShard(LOGIN);
        jooqDbSteps = api.userSteps.getDirectJooqDbSteps().useShard(shard);

        Long clientId = Long.valueOf(api.userSteps.clientFakeSteps().getClientData(LOGIN).getClientID());
        Long creativeId = jooqDbSteps.perfCreativesSteps().saveDefaultCanvasCreativesForClient(clientId);

        cid = api.userSteps.campaignSteps().addDefaultCpmBannerCampaign(LOGIN);
        //создаем группу cpm-banner с ключевыми словами
        pid = api.userSteps.adGroupsSteps().addDefaultCpmBannerKeywordsAdGroup(cid);
        List<Long> keywordIds = api.userSteps.keywordsSteps().addDefaultKeywords(pid, 1);
        bid = api.userSteps.adsSteps().addDefaultCpmBannerAdBuilderAd(pid, creativeId);

        api.userSteps.campaignFakeSteps().makeNewCampaignReadyForSendingToBS(cid);
        api.userSteps.groupFakeSteps().makeGroupFullyModerated(pid);
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(bid);
        api.userSteps.phrasesFakeSteps().makeKeywordModerated(keywordIds.get(0));

        jooqDbSteps.bannersPerformanceSteps()
                .setCreativeStatusModerate(cid, pid, bid, BannersPerformanceStatusmoderate.Yes);
        jooqDbSteps.perfCreativesSteps().setModerateInfo(creativeId,
                new ModerationInfo().generateTestData().toString());
        jooqDbSteps.perfCreativesSteps().setStatusModerate(creativeId, PerfCreativesStatusmoderate.Yes);
    }

    @Test
    @Title("Корректировка на погоду для кампании. Одна корректировка")
    public void checkCampaignWeatherMultiplier_SingleCoef() {
        List<List<DirectAtom>> initialExpression = createDefaultWeatherMultiplierCondition();

        WeatherMultiplierValuesRecord weatherMultiplier = new WeatherMultiplierValuesRecord()
                .setConditionJson(serializeToDirectFormat(initialExpression))
                .setMultiplierPct(MULTIPLIER_PCT1.shortValue());

        saveWeatherMultipliers(singletonList(weatherMultiplier), cid, null);

        Campaign campaign = sendMultiplierToBsAndGetResultCampaign();

        assumeThat("в БК отправлена кампания", campaign, notNullValue());
        assumeThat("Отправлен ExpessionCoefs", campaign.getExpressionCoefs(), notNullValue());
        List<WeatherCoef> weatherCoefs = campaign.getExpressionCoefs().getWeather();
        List<List<DirectAtom>> expectedExpression = BidMultiplierUtils.convertExpressionAtoms(initialExpression);

        checkBsWeatherMultipliers(weatherCoefs,
                ImmutableList.of(MULTIPLIER_PCT1),
                ImmutableList.of(expectedExpression));
    }

    @Test
    @Title("Корректировка на погоду для кампании. Несколько корректировок")
    public void checkCampaignWeatherMultiplier_SeveralCoef() {
        List<List<DirectAtom>> initialExpression1 = BidMultiplierConditionBuilder.init()
                .and(directAtom(ExpressionParameter.PREC_STRENGTH, Operation.eq, "75"),
                        directAtom(ExpressionParameter.PREC_STRENGTH, Operation.eq, "50"))
                .and(directAtom(ExpressionParameter.TEMP, Operation.le, "5"))
                .build();

        WeatherMultiplierValuesRecord weatherMultiplier1 = new WeatherMultiplierValuesRecord()
                .setConditionJson(serializeToDirectFormat(initialExpression1))
                .setMultiplierPct(MULTIPLIER_PCT1.shortValue());

        List<List<DirectAtom>> initialExpression2 = BidMultiplierConditionBuilder.init()
                .and(directAtom(ExpressionParameter.CLOUDNESS, Operation.eq, "50"))
                .build();

        WeatherMultiplierValuesRecord weatherMultiplier2 = new WeatherMultiplierValuesRecord()
                .setConditionJson(serializeToDirectFormat(initialExpression2))
                .setMultiplierPct(MULTIPLIER_PCT2.shortValue());

        saveWeatherMultipliers(ImmutableList.of(weatherMultiplier1, weatherMultiplier2), cid, null);

        Campaign campaign = sendMultiplierToBsAndGetResultCampaign();

        assumeThat("в БК отправлена кампания", campaign, notNullValue());
        assumeThat("Отправлен ExpessionCoefs", campaign.getExpressionCoefs(), notNullValue());
        List<WeatherCoef> weatherCoefs = campaign.getExpressionCoefs().getWeather();

        List<List<DirectAtom>> expectedExpression1 = BidMultiplierUtils.convertExpressionAtoms(initialExpression1);
        List<List<DirectAtom>> expectedExpression2 = BidMultiplierUtils.convertExpressionAtoms(initialExpression2);

        checkBsWeatherMultipliers(weatherCoefs,
                ImmutableList.of(MULTIPLIER_PCT1, MULTIPLIER_PCT2),
                ImmutableList.of(expectedExpression1, expectedExpression2));
    }

    @Test
    @Title("Корректировка на погоду для группы. Одна корректировка")
    public void checkAdGroupWeatherMultiplier_SingleCoef() {
        List<List<DirectAtom>> initialExpression1 = BidMultiplierConditionBuilder.init()
                .and(directAtom(ExpressionParameter.TEMP, Operation.ge, "1"),
                        directAtom(ExpressionParameter.TEMP, Operation.le, "20"))
                .and(directAtom(ExpressionParameter.CLOUDNESS, Operation.eq, "25"))
                .build();

        WeatherMultiplierValuesRecord weatherMultiplier1 = new WeatherMultiplierValuesRecord()
                .setConditionJson(serializeToDirectFormat(initialExpression1))
                .setMultiplierPct(MULTIPLIER_PCT1.shortValue());

        saveWeatherMultipliers(singletonList(weatherMultiplier1), cid, pid);

        Campaign campaign = sendMultiplierToBsAndGetResultCampaign();
        assumeThat("в БК отправлена кампания", campaign, notNullValue());

        Context context = campaign != null ? campaign.getContext(pid) : null;
        assumeThat("в БК отправлен контекст", context, notNullValue());

        assumeThat("Отправлен ExpessionCoefs", context.getExpressionCoefs(), notNullValue());
        List<WeatherCoef> weatherCoefs = context.getExpressionCoefs().getWeather();

        List<List<DirectAtom>> expectedExpression1 = BidMultiplierUtils.convertExpressionAtoms(initialExpression1);

        checkBsWeatherMultipliers(weatherCoefs,
                singletonList(MULTIPLIER_PCT1),
                singletonList(expectedExpression1));
    }

    @Test
    @Title("Корректировка на погоду для группы. Несколько корректировок")
    public void checkAdGroupWeatherMultiplier_SeveralCoefs() {

        List<List<DirectAtom>> initialExpression1 = BidMultiplierConditionBuilder.init()
                .and(directAtom(ExpressionParameter.TEMP, Operation.ge, "1"),
                        directAtom(ExpressionParameter.TEMP, Operation.le, "20"))
                .and(directAtom(ExpressionParameter.CLOUDNESS, Operation.eq, "25"))
                .build();

        WeatherMultiplierValuesRecord weatherMultiplier1 = new WeatherMultiplierValuesRecord()
                .setConditionJson(serializeToDirectFormat(initialExpression1))
                .setMultiplierPct(MULTIPLIER_PCT1.shortValue());

        List<List<DirectAtom>> initialExpression2 = BidMultiplierConditionBuilder.init()
                .and(directAtom(ExpressionParameter.PREC_STRENGTH, Operation.eq, "75"))
                .build();

        WeatherMultiplierValuesRecord weatherMultiplier2 = new WeatherMultiplierValuesRecord()
                .setConditionJson(serializeToDirectFormat(initialExpression2))
                .setMultiplierPct(MULTIPLIER_PCT2.shortValue());

        saveWeatherMultipliers(ImmutableList.of(weatherMultiplier1, weatherMultiplier2), cid, pid);

        Campaign campaign = sendMultiplierToBsAndGetResultCampaign();
        assumeThat("в БК отправлена кампания", campaign, notNullValue());

        Context context = campaign != null ? campaign.getContext(pid) : null;
        assumeThat("в БК отправлен контекст", context, notNullValue());
        assumeThat("Отправлен ExpessionCoefs", context.getExpressionCoefs(), notNullValue());
        List<WeatherCoef> weatherCoefs = context.getExpressionCoefs().getWeather();

        List<List<DirectAtom>> expectedExpression1 = BidMultiplierUtils.convertExpressionAtoms(initialExpression1);
        List<List<DirectAtom>> expectedExpression2 = BidMultiplierUtils.convertExpressionAtoms(initialExpression2);

        checkBsWeatherMultipliers(weatherCoefs,
                ImmutableList.of(MULTIPLIER_PCT1, MULTIPLIER_PCT2),
                ImmutableList.of(expectedExpression1, expectedExpression2));
    }

    private void saveWeatherMultipliers(List<WeatherMultiplierValuesRecord> weatherMultipliers, Long cid,
                                        @Nullable Long pid) {
        api.userSteps.getDirectJooqDbSteps().useShard(shard).multipliersSteps()
                .saveWeatherMultiplierValues(cid, pid, weatherMultipliers, true);
    }

    private Campaign sendMultiplierToBsAndGetResultCampaign() {
        api.userSteps.campaignFakeSteps().makeNewCampaignReadyForSendingToBS(cid);
        api.userSteps.groupFakeSteps().makeGroupFullyModerated(pid);

        RunBsTransportScriptResponse resp =
                api.userSteps.getDarkSideSteps().getTransportSteps().sendNewCampaign(shard, cid);

        return api.userSteps.getDarkSideSteps().getTransportSteps().getClientDataRequestCampaign(resp, 0, cid);
    }

    public List<List<DirectAtom>> createDefaultWeatherMultiplierCondition() {
        return BidMultiplierConditionBuilder.init()
                .and(directAtom(ExpressionParameter.PREC_STRENGTH, Operation.ge, "75"))
                .and(directAtom(ExpressionParameter.CLOUDNESS, Operation.ge, "50"))
                .and(directAtom(ExpressionParameter.TEMP, Operation.le, "5"))
                .and(directAtom(ExpressionParameter.TEMP, Operation.ge, "-5"))
                .build();
    }

    private void checkBsWeatherMultipliers(List<WeatherCoef> weatherCoefs, List<Integer> expectedCoefs,
                                           List<List<List<DirectAtom>>> expectedExpressions) {
        assumeThat("Количество ожидаемых значений совпадает с количеством проверяемых объектов",
                weatherCoefs.size(), equalTo(expectedCoefs.size()));
        assumeThat("Количество ожидаемых значений совпадает с количеством проверяемых объектов",
                weatherCoefs.size(), equalTo(expectedExpressions.size()));

        int exprectedCount = expectedCoefs.size();
        assumeThat("WeatherCoefs содержит ожидаемое количество корректировок",
                weatherCoefs.size(), equalTo(exprectedCount));

        for (int i = 0; i < weatherCoefs.size(); i++) {
            assumeThat("Значение Coef соответствует ожидаемому значению",
                    weatherCoefs.get(i).getCoef(), equalTo(expectedCoefs.get(i)));

            assertThat("Выражение корректировки соответсвует ожиданиям",
                    weatherCoefs.get(i).getExpression(), beanDiffer(expectedExpressions.get(i)));
        }

    }


}


