package ru.yandex.autotests.directintapi.bstransport.main.group.parameters.socdemcoef;

import java.util.Arrays;
import java.util.List;
import java.util.Map;

import com.google.common.collect.ImmutableList;
import com.yandex.direct.api.v5.general.AgeRangeEnum;
import com.yandex.direct.api.v5.general.GenderEnum;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.darkside.Logins;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.darkside.model.BidMultiplierConditionBuilder;
import ru.yandex.autotests.directapi.darkside.model.RunBsTransportScriptResponse;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.Campaign;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.Context;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.ExpressionCoefItem;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.ExpressionCoefs;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.SocdemCoef;
import ru.yandex.autotests.directapi.darkside.model.multipliers.DirectAtom;
import ru.yandex.autotests.directapi.darkside.model.multipliers.SocdemValueAge;
import ru.yandex.autotests.directapi.darkside.model.multipliers.SocdemValueGender;
import ru.yandex.autotests.directapi.enums.AgeGroup;
import ru.yandex.autotests.directapi.enums.Gender;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.BidModifierAddMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.DemographicsAdjustmentMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directintapi.bstransport.FeatureNames;
import ru.yandex.autotests.directintapi.bstransport.StoriesNames;
import ru.yandex.autotests.directintapi.bstransport.TransportHelpSteps;
import ru.yandex.autotests.irt.testutils.beandiffer.BeanDifferMatcher;
import ru.yandex.autotests.irt.testutils.beandiffer.matchvariation.DefaultMatchVariation;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.allure.annotations.Title;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.notNullValue;
import static ru.yandex.autotests.directapi.darkside.model.BidMultiplierUtils.socdemBsAtomAge;
import static ru.yandex.autotests.directapi.darkside.model.BidMultiplierUtils.socdemBsAtomGender;
import static ru.yandex.autotests.directintapi.bstransport.TransportTestUtils.checkBsUnifiedMultipliers;
import static ru.yandex.autotests.directintapi.bstransport.TransportTestUtils.getDefaultSocdemCoef;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

@Aqua.Test
@Tag(TagDictionary.RELEASE)
@Title("Транспорт: проверка отправки параметра SocdemCoef на уровне группы")
@Stories(StoriesNames.GROUP_PARAMS_SOCDEMCOEF)
@Features(FeatureNames.GROUPS)
@Issue("https://st.yandex-team.ru/DIRECT-42533")
public class BsTransportOfGroupSocdemCoefTest {

    private static final String LOGIN = Logins.LOGIN_TRANSPORT;
    private static final int COEF = 50;
    private static final int ANOTHER_COEF = 250;
    private static final AgeRangeEnum AGE_GROUP = AgeRangeEnum.AGE_0_17;
    private static final GenderEnum GENDER = GenderEnum.GENDER_FEMALE;
    private static final AgeRangeEnum ANOTHER_AGE_GROUP = AgeRangeEnum.AGE_25_34;
    private static final GenderEnum ANOTHER_GENDER = GenderEnum.GENDER_MALE;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trashman = new Trashman(api);

    private static TransportHelpSteps helpSteps = new TransportHelpSteps(api);

    private static int shard;

    private Long cid;
    private Long pid;


    @BeforeClass
    public static void beforeClass() {
        shard = api.userSteps.clientFakeSteps().getUserShard(LOGIN);
    }

    @Before
    @Step("Подготовка тестовых данных")
    public void before() {
        cid = api.userSteps.campaignSteps().addDefaultTextCampaign();
        pid = api.userSteps.adGroupsSteps().addDefaultGroup(cid);
        Long bid = api.userSteps.adsSteps().addDefaultTextAd(pid);
        Long keywordId = api.userSteps.keywordsSteps().addDefaultKeyword(pid);
        api.userSteps.campaignFakeSteps().makeNewCampaignReadyForSendingToBS(cid);
        api.userSteps.groupFakeSteps().makeGroupFullyModerated(pid);
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(bid);
        api.userSteps.phrasesFakeSteps().makeKeywordModerated(keywordId);
    }

    @Test
    @Title("Отправка в БК корректировки SocdemCoef на уровне группы, 1 простой коэффециент")
    public void testTransportOfGroupOneSocdemCoef() {
        api.userSteps.bidModifiersSteps().bidModifiersAdd(
                new BidModifierAddMap()
                        .withAdGroupId(pid)
                        .withDemographicsAdjustment(new DemographicsAdjustmentMap()
                                .withAge(AGE_GROUP)
                                .withGender(GENDER)
                                .withBidModifier(COEF)
                        )
        );

        RunBsTransportScriptResponse resp = api.userSteps.getDarkSideSteps().getTransportSteps().sendNewCampaign(shard, cid);

        Map<String, Map> expected = getDefaultSocdemCoef();
        expected.get(Gender.FEMALE.toString()).put(AgeGroup.CHILD.toString(), COEF);
        Map<String, Map> contextMap = helpSteps.getContextMap(resp, cid, pid);
        assumeThat("в БК отправлен контекст (группа)", contextMap, notNullValue());
        assertThat("отправленная в БК корректировка SocdemCoef на уровне группы соответствует ожидаемой",
                contextMap.get(Context.SOCDEM_COEF)
                , BeanDifferMatcher.beanDiffer(expected).withVariation(
                        new DefaultMatchVariation()
                                .setIgnoreNullFieldsOfExpectedBean(false)
                )
        );

        // проверяем передачу в унифицированном формате
        Campaign campaign = api.userSteps.getDarkSideSteps().getTransportSteps().getClientDataRequestCampaign(resp, 0, cid);
        assumeThat("в БК отправлена кампания", campaign, notNullValue());

        ExpressionCoefs expressionCoefs = campaign.getContext(pid).getExpressionCoefs();
        assumeThat("Отправлен ExpessionCoefs", expressionCoefs, notNullValue());

        List<List<DirectAtom>> expectedExpression = BidMultiplierConditionBuilder.init()
                .and(socdemBsAtomAge(SocdemValueAge.AGE_0_17))
                .and(socdemBsAtomGender(SocdemValueGender.GENDER_FEMALE))
                .build();

        List<ExpressionCoefItem> socdemCoefs = expressionCoefs.getSocdem();

        checkBsUnifiedMultipliers(socdemCoefs,
                ImmutableList.of(COEF),
                ImmutableList.of(expectedExpression));
    }

    @Test
    @Title("Отправка в БК корректировки SocdemCoef на уровне группы, 2 простых коэффециента")
    public void testTransportOfGroupTwoSocdemCoef() {
        api.userSteps.bidModifiersSteps().bidModifiersAdd(
                new BidModifierAddMap()
                        .withAdGroupId(pid)
                        .withDemographicsAdjustment(new DemographicsAdjustmentMap()
                                        .withAge(AGE_GROUP)
                                        .withGender(GENDER)
                                        .withBidModifier(COEF)
                                , new DemographicsAdjustmentMap()
                                        .withAge(ANOTHER_AGE_GROUP)
                                        .withGender(ANOTHER_GENDER)
                                        .withBidModifier(ANOTHER_COEF)
                        )
        );
        RunBsTransportScriptResponse resp = api.userSteps.getDarkSideSteps().getTransportSteps().sendNewCampaign(shard, cid);

        Map<String, Map> expected = getDefaultSocdemCoef();
        expected.get(Gender.MALE.toString()).put(AgeGroup.YOUNGADULT.toString(), ANOTHER_COEF);
        expected.get(Gender.FEMALE.toString()).put(AgeGroup.CHILD.toString(), COEF);
        Map<String, Map> contextMap = helpSteps.getContextMap(resp, cid, pid);
        assumeThat("в БК отправлен контекст (группа)", contextMap, notNullValue());
        assertThat("отправленная в БК корректировка SocdemCoef на уровне группы соответствует ожидаемой",
                contextMap.get(Context.SOCDEM_COEF)
                , BeanDifferMatcher.beanDiffer(expected).withVariation(
                        new DefaultMatchVariation()
                                .setIgnoreNullFieldsOfExpectedBean(false)
                )
        );

        // проверяем передачу в унифицированном формате
        Campaign campaign = api.userSteps.getDarkSideSteps().getTransportSteps().getClientDataRequestCampaign(resp, 0, cid);
        assumeThat("в БК отправлена кампания", campaign, notNullValue());

        ExpressionCoefs expressionCoefs = campaign.getContext(pid).getExpressionCoefs();
        assumeThat("Отправлен ExpessionCoefs", expressionCoefs, notNullValue());

        List<List<DirectAtom>> expectedExpression1 = BidMultiplierConditionBuilder.init()
                .and(socdemBsAtomAge(SocdemValueAge.AGE_0_17))
                .and(socdemBsAtomGender(SocdemValueGender.GENDER_FEMALE))
                .build();
        List<List<DirectAtom>> expectedExpression2 = BidMultiplierConditionBuilder.init()
                .and(socdemBsAtomAge(SocdemValueAge.AGE_25_34))
                .and(socdemBsAtomGender(SocdemValueGender.GENDER_MALE))
                .build();

        List<ExpressionCoefItem> socdemCoefs = expressionCoefs.getSocdem();

        checkBsUnifiedMultipliers(socdemCoefs,
                ImmutableList.of(COEF, ANOTHER_COEF),
                ImmutableList.of(expectedExpression1, expectedExpression2));
    }

    @Test
    @Title("Отправка в БК корректировки SocdemCoef на уровне группы, 1 коэффециент на возрастную группу")
    public void testTransportOfGroupAgeSocdemCoef() {
        api.userSteps.bidModifiersSteps().bidModifiersAdd(
                new BidModifierAddMap()
                        .withAdGroupId(pid)
                        .withDemographicsAdjustment(new DemographicsAdjustmentMap()
                                .withAge(AGE_GROUP)
                                .withBidModifier(COEF)
                        )
        );

        RunBsTransportScriptResponse resp = api.userSteps.getDarkSideSteps().getTransportSteps().sendNewCampaign(shard, cid);

        Map<String, Map> expected = getDefaultSocdemCoef();
        expected.get(Gender.MALE.toString()).put(AgeGroup.CHILD.toString(), COEF);
        expected.get(Gender.FEMALE.toString()).put(AgeGroup.CHILD.toString(), COEF);
        expected.get(SocdemCoef.UNKNOWN).put(AgeGroup.CHILD.toString(), COEF);
        Map<String, Map> contextMap = helpSteps.getContextMap(resp, cid, pid);
        assumeThat("в БК отправлен контекст (группа)", contextMap, notNullValue());
        assertThat("отправленная в БК корректировка SocdemCoef на уровне группы соответствует ожидаемой",
                contextMap.get(Context.SOCDEM_COEF)
                , BeanDifferMatcher.beanDiffer(expected).withVariation(
                        new DefaultMatchVariation()
                                .setIgnoreNullFieldsOfExpectedBean(false)
                )
        );

        // проверяем передачу в унифицированном формате
        Campaign campaign = api.userSteps.getDarkSideSteps().getTransportSteps().getClientDataRequestCampaign(resp, 0, cid);
        assumeThat("в БК отправлена кампания", campaign, notNullValue());

        ExpressionCoefs expressionCoefs = campaign.getContext(pid).getExpressionCoefs();
        assumeThat("Отправлен ExpessionCoefs", expressionCoefs, notNullValue());

        List<List<DirectAtom>> expectedExpression = BidMultiplierConditionBuilder.init()
                .and(socdemBsAtomAge(SocdemValueAge.AGE_0_17))
                .build();

        List<ExpressionCoefItem> socdemCoefs = expressionCoefs.getSocdem();

        checkBsUnifiedMultipliers(socdemCoefs,
                ImmutableList.of(COEF),
                ImmutableList.of(expectedExpression));
    }


    @Test
    @Title("Отправка в БК корректировки SocdemCoef на уровне группы, 1 коэффециент на пол")
    public void testTransportOfGroupGenderSocdemCoef() {
        api.userSteps.bidModifiersSteps().bidModifiersAdd(
                new BidModifierAddMap()
                        .withAdGroupId(pid)
                        .withDemographicsAdjustment(new DemographicsAdjustmentMap()
                                .withGender(GENDER)
                                .withBidModifier(COEF)
                        )
        );

        RunBsTransportScriptResponse resp = api.userSteps.getDarkSideSteps().getTransportSteps().sendNewCampaign(shard, cid);

        Map<String, Map> expected = getDefaultSocdemCoef();
        Arrays.asList(AgeGroup.values()).forEach(
                ageGroup -> expected.get(Gender.FEMALE.toString()).put(ageGroup.toString(), COEF));
        expected.get(Gender.FEMALE.toString()).put(SocdemCoef.UNKNOWN, COEF);

        Map<String, Map> contextMap = helpSteps.getContextMap(resp, cid, pid);
        assumeThat("в БК отправлен контекст (группа)", contextMap, notNullValue());
        assertThat("отправленная в БК корректировка SocdemCoef на уровне группы соответствует ожидаемой",
                contextMap.get(Context.SOCDEM_COEF)
                , BeanDifferMatcher.beanDiffer(expected).withVariation(
                        new DefaultMatchVariation()
                                .setIgnoreNullFieldsOfExpectedBean(false)
                )
        );

        // проверяем передачу в унифицированном формате
        Campaign campaign = api.userSteps.getDarkSideSteps().getTransportSteps().getClientDataRequestCampaign(resp, 0, cid);
        assumeThat("в БК отправлена кампания", campaign, notNullValue());

        ExpressionCoefs expressionCoefs = campaign.getContext(pid).getExpressionCoefs();
        assumeThat("Отправлен ExpessionCoefs", expressionCoefs, notNullValue());

        List<List<DirectAtom>> expectedExpression = BidMultiplierConditionBuilder.init()
                .and(socdemBsAtomGender(SocdemValueGender.GENDER_FEMALE))
                .build();

        List<ExpressionCoefItem> socdemCoefs = expressionCoefs.getSocdem();

        checkBsUnifiedMultipliers(socdemCoefs,
                ImmutableList.of(COEF),
                ImmutableList.of(expectedExpression));
    }
}
