package ru.yandex.autotests.directintapi.bstransport.main.internalads;

import java.math.BigDecimal;
import java.sql.Timestamp;
import java.time.Instant;
import java.util.List;

import org.jetbrains.annotations.NotNull;
import org.junit.After;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.cmd.rules.CampaignRule;
import ru.yandex.autotests.direct.cmd.rules.DirectCmdRule;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.RetargetingConditionsRetargetingConditionsType;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BidsRetargetingRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.RetargetingConditionsRecord;
import ru.yandex.autotests.direct.httpclient.TestEnvironment;
import ru.yandex.autotests.direct.utils.campaigns.CampaignTypeEnum;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.Campaign;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.Context;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.GoalContext;
import ru.yandex.autotests.directapi.darkside.model.multipliers.BsAtom;
import ru.yandex.autotests.directintapi.bstransport.FeatureNames;
import ru.yandex.autotests.directintapi.bstransport.TransportHelpSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;

import static java.util.Collections.singletonList;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.notNullValue;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;
import static ru.yandex.autotests.irt.testutils.beandiffer2.BeanDifferMatcher.beanDiffer;

@Aqua.Test
@Tag(TagDictionary.RELEASE)
@Tag(TagDictionary.TRUNK)
@Description("Проверяем отправку условий ретаргетинга внутренней рекламы")
@Features({FeatureNames.INTERNAL_ADS, FeatureNames.RETARGETINGS})
@Issue("https://st.yandex-team.ru/DIRECT-117150")
public class InternalAdGroupRetargetingConditionTest extends InternalAdsBaseTest {
    private static final String INTERNAL_AD_RET_COND_JSON =
            "[{\"goals\":[{\"goal_type\":\"interests\",\"goal_id\":2499001148,\"time\":0}," +
                    "{\"goal_type\":\"internal\",\"goal_id\":2499980549,\"time\":0}],\"type\":\"not\"," +
                    "\"interest_type\":\"all\",\"section_id\":null},{\"goals\":[{\"goal_type\":\"interests\"," +
                    "\"goal_id\":2499001146,\"time\":0},{\"goal_type\":\"internal\",\"goal_id\":2499980542," +
                    "\"time\":0}],\"type\":\"or\",\"interest_type\":\"all\",\"section_id\":null}," +
                    "{\"goals\":[{\"goal_type\":\"interests\",\"goal_id\":2499001146,\"time\":0}," +
                    "{\"goal_type\":\"internal\",\"goal_id\":2499980547,\"time\":0}],\"type\":\"all\"," +
                    "\"interest_type\":\"all\",\"section_id\":null},{\"goals\":[{\"goal_type\":\"social_demo\"," +
                    "\"goal_id\":2499000010,\"time\":0}],\"type\":\"or\",\"interest_type\":null,\"section_id\":null}]";
    public static final String EXPECTED_GOAL_CONTEXT_EXPRESSION = "~(12:0@601|12:0@602)&~1367:0@549&" +
            "(223:0@601|223:0@602|13066246:0@557)&(223:0@601|223:0@602)&1257:0@549&(1:0@618)";

    @ClassRule
    public static DirectCmdRule defaultClassRule = DirectCmdRule.defaultClassRule();

    public CampaignRule campaignRule = new CampaignRule()
            .withMediaType(CampaignTypeEnum.INTERNAL_FREE)
            .withUlogin(LOGIN);

    @Rule
    public DirectCmdRule cmdRule = DirectCmdRule.defaultRule().withRules(campaignRule);

    private Long clientId;
    private Long retCondId;
    private Long retId;
    private Campaign campaignDataReq;
    private Context context;

    @BeforeClass
    public static void beforeClass() {
        shard = api.userSteps.getDarkSideSteps().getClientFakeSteps().getUserShard(LOGIN);
        dbSteps = TestEnvironment.newDbSteps().useShard(shard);
        transportHelpSteps = new TransportHelpSteps(api).useDbJooqSteps(dbSteps);
    }

    @Before
    public void before() {
        cid = campaignRule.getCampaignId();
        api.userSteps.campaignFakeSteps().makeCampaignInternalReadyForSendingToBS(cid);

        clientId =
                Long.valueOf(api.userSteps.getDarkSideSteps().getClientFakeSteps().getClientData(LOGIN).getClientID());

        pid = createInternalAdgroupsWithAdditionalTargetings(clientId, cid, null);
        bid = createInternalBanner(clientId, cid, pid);
        retCondId = addRetargetingCondition(clientId);
        retId = addRetargeting(cid, pid, retCondId);
        campaignDataReq = sendNewCampaign();
        assertThat("в БК отправлена кампания", campaignDataReq, notNullValue());

        context = campaignDataReq.getContext(pid);
        assumeThat("в БК отправлена группа", context, notNullValue());
    }

    @After
    @Step("Утилизация тестовых данных")
    public void after() {
        dbSteps.bidsRetargetingSteps().deleteRetargeting(retId);
        dbSteps.retargetingConditionSteps().deleteRetargeingCondition(retCondId);
    }

    @Test
    public void checkRetargetingWasSent() {
        GoalContext goalContext = context.getGoalContext(retId);
        assertThat("в запросе есть ретаргетинг", goalContext, notNullValue());
    }

    @Test
    public void checkGoalContextExpression() {
        GoalContext goalContext = context.getGoalContext(retId);
        assumeThat("в запросе есть ретаргетинг", goalContext, notNullValue());
        assumeThat("в TargetingExpression указан таргетинг на goalContext",
                context.getTargetingExpression(), beanDiffer(createExpectedTargetingExpression()));

        assertThat("Expression в GoalContext ожидаемый",
                goalContext.getExpression(), equalTo(EXPECTED_GOAL_CONTEXT_EXPRESSION));
    }

    private Long addRetargetingCondition(Long clientId) {
        RetargetingConditionsRecord record = new RetargetingConditionsRecord()
                .setClientid(clientId)
                .setRetargetingConditionsType(RetargetingConditionsRetargetingConditionsType.interests)
                .setConditionName("Аудитория для внутренней рекламы")
                .setConditionDesc(null)
                .setConditionJson(INTERNAL_AD_RET_COND_JSON)
                .setIsDeleted(0);
        return dbSteps.retargetingConditionSteps().addRetargetingCondition(record, this.clientId);
    }

    private Long addRetargeting(Long cid, Long pid, Long retCondId) {
        BidsRetargetingRecord record = new BidsRetargetingRecord()
                .setRetCondId(retCondId)
                .setCid(cid)
                .setPid(pid)
                .setBid(0L)
                .setModtime(Timestamp.from(Instant.now()))
                .setPriceContext(BigDecimal.ONE)
                .setAutobudgetpriority(3)
                .setIsSuspended(0);

        return dbSteps.bidsRetargetingSteps().addBidsRetargeting(record);
    }

    @NotNull
    private List<List<BsAtom>> createExpectedTargetingExpression() {
        return singletonList(
                singletonList(new BsAtom("goal-context-id", "match goal context", retCondId.toString())));
    }

}
