package ru.yandex.autotests.directintapi.bstransport.main.offerretargeting;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BidsBaseBidType;
import ru.yandex.autotests.direct.db.steps.BidsBaseSteps;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.darkside.Logins;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.darkside.model.RunBsTransportScriptResponse;
import ru.yandex.autotests.directapi.darkside.model.Status;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.Campaign;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.Context;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.OfferRetargeting;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directintapi.bstransport.FeatureNames;
import ru.yandex.autotests.irt.testutils.beandiffer2.comparestrategy.defaultcomparestrategy.DefaultCompareStrategies;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.nullValue;
import static ru.yandex.autotests.directintapi.bstransport.TransportTestUtils.getMapValues;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;
import static ru.yandex.autotests.irt.testutils.beandiffer2.BeanDifferMatcher.beanDiffer;

@Aqua.Test
@Tag(TagDictionary.TRUNK)
@Tag(TagDictionary.RELEASE)
@Description("Транспорт: проверка отправки в БК офферных ретаргетингов в различных состояниях")
@Features({FeatureNames.OFFER_RETARGETING, FeatureNames.NOT_FOR_FULL_EXPORT})
@Issue("https://st.yandex-team.ru/DIRECT-158191")
public class OfferRetargetingTest {

    private static final String LOGIN = Logins.LOGIN_TRANSPORT;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trashman = new Trashman(api);

    private static int shard;
    private static BidsBaseSteps bidsBaseSteps;

    private Long cid;
    private Long pid;
    private Long bid;

    @BeforeClass
    public static void beforeClass() {
        shard = api.userSteps.clientFakeSteps().getUserShard(LOGIN);
        bidsBaseSteps = api.userSteps.getDirectJooqDbSteps().useShard(shard).bidsBaseSteps();
    }

    @Before
    public void before() {
        cid = api.userSteps.campaignSteps().addDefaultTextCampaign();
        pid = api.userSteps.adGroupsSteps().addDefaultGroup(cid);
        bid = api.userSteps.adsSteps().addDefaultTextAd(pid);
    }

    @Test
    public void oneSuspendedOfferRetargeting() {
        bidsBaseSteps.saveDefaultOfferRetargeting(cid, pid, Currency.RUB, BidsBaseSteps.OPTS_SUSPENDED);
        api.userSteps.campaignFakeSteps().makeNewCampaignReadyForSendingToBS(cid);
        api.userSteps.groupFakeSteps().makeGroupFullyModerated(pid);
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(bid);

        RunBsTransportScriptResponse resp =
                api.userSteps.getDarkSideSteps().getTransportSteps().sendNewCampaign(shard, cid);

        Campaign campaign =
                api.userSteps.getDarkSideSteps().getTransportSteps().getClientDataRequestCampaign(resp, 0, cid);
        Context context = campaign != null ? campaign.getContext(pid) : null;
        assertThat("контекст не отправился", context, nullValue());
    }

    @Test
    public void oneDeletedOfferRetargeting() {
        bidsBaseSteps.saveDefaultOfferRetargeting(cid, pid, Currency.RUB, BidsBaseSteps.OPTS_DELETED);
        api.userSteps.campaignFakeSteps().makeNewCampaignReadyForSendingToBS(cid);
        api.userSteps.groupFakeSteps().makeGroupFullyModerated(pid);
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(bid);

        RunBsTransportScriptResponse resp = api.userSteps.getDarkSideSteps().getTransportSteps()
                .sendNewCampaign(shard, cid);
        Campaign campaign = api.userSteps.getDarkSideSteps().getTransportSteps()
                .getClientDataRequestCampaign(resp, 0, cid);
        Context context = campaign != null ? campaign.getContext(pid) : null;
        assertThat("контекст не отправился", context, nullValue());
    }

    @Test
    public void oneOfferRetargetingAfterSuspended() {
        Long offerRetargetingId = bidsBaseSteps.saveDefaultOfferRetargeting(cid, pid, Currency.RUB);
        api.userSteps.campaignFakeSteps().makeNewCampaignReadyForSendingToBS(cid);
        api.userSteps.groupFakeSteps().makeGroupFullyModerated(pid);
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(bid);

        api.userSteps.getDarkSideSteps().getTransportSteps().sendNewCampaign(shard, cid);

        bidsBaseSteps.setBidsBaseOpts(offerRetargetingId, BidsBaseSteps.OPTS_SUSPENDED);
        api.userSteps.groupFakeSteps().setGroupFakeStatusBsSynced(pid, Status.NO);

        RunBsTransportScriptResponse resp =
                api.userSteps.getDarkSideSteps().getTransportSteps().sendSyncedCampaign(shard, cid);

        Campaign campaign =
                api.userSteps.getDarkSideSteps().getTransportSteps().getClientDataRequestCampaign(resp, 0, cid);
        Context context = campaign != null ? campaign.getContext(pid) : null;
        List<OfferRetargeting> offerRetargetings = getMapValues(context != null ? context.getOfferRetargetings() : null);
        assertThat("остановленный офферный ретаргетинг не отправился", offerRetargetings, hasSize(0));
    }

    @Test
    public void oneOfferRetargetingAfterDeleted() {
        Long offerRetargetingId = bidsBaseSteps.saveDefaultOfferRetargeting(cid, pid, Currency.RUB);
        api.userSteps.campaignFakeSteps().makeNewCampaignReadyForSendingToBS(cid);
        api.userSteps.groupFakeSteps().makeGroupFullyModerated(pid);
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(bid);

        api.userSteps.getDarkSideSteps().getTransportSteps().sendNewCampaign(shard, cid);

        bidsBaseSteps.setBidsBaseOpts(offerRetargetingId, BidsBaseSteps.OPTS_DELETED);
        api.userSteps.groupFakeSteps().setGroupFakeStatusBsSynced(pid, Status.NO);

        RunBsTransportScriptResponse resp =
                api.userSteps.getDarkSideSteps().getTransportSteps().sendSyncedCampaign(shard, cid);

        Campaign campaign =
                api.userSteps.getDarkSideSteps().getTransportSteps().getClientDataRequestCampaign(resp, 0, cid);
        Context context = campaign != null ? campaign.getContext(pid) : null;
        List<OfferRetargeting> offerRetargetings = getMapValues(context != null ? context.getOfferRetargetings() : null);
        assertThat("удаленный офферный ретаргетинг не отправился", offerRetargetings, hasSize(0));
    }

    @Test
    public void oneSuspendedOfferRetargetingAndPhrase() {
        Long bidsId = api.userSteps.keywordsSteps().addDefaultKeyword(pid);
        bidsBaseSteps.saveDefaultOfferRetargeting(cid, pid, Currency.RUB, BidsBaseSteps.OPTS_SUSPENDED);
        api.userSteps.campaignFakeSteps().makeNewCampaignReadyForSendingToBS(cid);
        api.userSteps.groupFakeSteps().makeGroupFullyModerated(pid);
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(bid);
        api.userSteps.phrasesFakeSteps().makeKeywordModerated(bidsId);

        RunBsTransportScriptResponse resp =
                api.userSteps.getDarkSideSteps().getTransportSteps().sendNewCampaign(shard, cid);

        Campaign campaign =
                api.userSteps.getDarkSideSteps().getTransportSteps().getClientDataRequestCampaign(resp, 0, cid);
        Context context = campaign != null ? campaign.getContext(pid) : null;
        List<OfferRetargeting> offerRetargetings = getMapValues(context != null ? context.getOfferRetargetings() : null);
        assertThat("остановленный офферный ретаргетинг не отправился", offerRetargetings, hasSize(0));
    }

    @Test
    public void oneDeletedOfferRetargetingAndPhrase() {
        Long bidsId = api.userSteps.keywordsSteps().addDefaultKeyword(pid);
        bidsBaseSteps.saveDefaultOfferRetargeting(cid, pid, Currency.RUB, BidsBaseSteps.OPTS_DELETED);
        api.userSteps.campaignFakeSteps().makeNewCampaignReadyForSendingToBS(cid);
        api.userSteps.groupFakeSteps().makeGroupFullyModerated(pid);
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(bid);
        api.userSteps.phrasesFakeSteps().makeKeywordModerated(bidsId);

        RunBsTransportScriptResponse resp =
                api.userSteps.getDarkSideSteps().getTransportSteps().sendNewCampaign(shard, cid);

        Campaign campaign =
                api.userSteps.getDarkSideSteps().getTransportSteps().getClientDataRequestCampaign(resp, 0, cid);
        Context context = campaign != null ? campaign.getContext(pid) : null;
        List<OfferRetargeting> offerRetargetings = getMapValues(context != null ? context.getOfferRetargetings() : null);
        assertThat("удаленный офферный ретаргетинг не отправился", offerRetargetings, hasSize(0));
    }

    @Test
    public void twoActiveAndSuspendedAndDeletedOfferRetargeting() {
        Long offerRetargetingId = bidsBaseSteps.saveDefaultOfferRetargeting(cid, pid, Currency.RUB);
        Long offerRetargetingIdElse = bidsBaseSteps.saveDefaultOfferRetargeting(cid, pid, Currency.RUB);
        bidsBaseSteps.saveDefaultOfferRetargeting(cid, pid, Currency.RUB, BidsBaseSteps.OPTS_SUSPENDED);
        bidsBaseSteps.saveDefaultOfferRetargeting(cid, pid, Currency.RUB, BidsBaseSteps.OPTS_DELETED);
        api.userSteps.campaignFakeSteps().makeNewCampaignReadyForSendingToBS(cid);
        api.userSteps.groupFakeSteps().makeGroupFullyModerated(pid);
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(bid);

        RunBsTransportScriptResponse resp =
                api.userSteps.getDarkSideSteps().getTransportSteps().sendNewCampaign(shard, cid);

        Campaign campaign =
                api.userSteps.getDarkSideSteps().getTransportSteps().getClientDataRequestCampaign(resp, 0, cid);
        Context context = campaign != null ? campaign.getContext(pid) : null;
        List<OfferRetargeting> offerRetargetings = getMapValues(context != null ? context.getOfferRetargetings() : null);
        assumeThat("отправилось два офферных ретаргетинга", offerRetargetings, hasSize(2));
        Map<String, OfferRetargeting> offerRetargetingMap = context.getOfferRetargetings();
        Map<String, OfferRetargeting> expectedOfferRetargetingMap = new HashMap<>();
        BidsBaseBidType expectedBidType = BidsBaseBidType.offer_retargeting;

        expectedOfferRetargetingMap.put(offerRetargetingId.toString(), new OfferRetargeting()
                .withEid(offerRetargetingId.toString())
                .withType(expectedBidType.toString())
                .withUpdateInfo(1));
        expectedOfferRetargetingMap.put(offerRetargetingIdElse.toString(), new OfferRetargeting()
                .withEid(offerRetargetingIdElse.toString())
                .withType(expectedBidType.toString())
                .withUpdateInfo(1));
        assertThat("отправился правильный офферный ретаргетинг", offerRetargetingMap,
                beanDiffer(expectedOfferRetargetingMap).useCompareStrategy(
                        DefaultCompareStrategies.onlyExpectedFields()));
    }

    @Test
    public void oneActiveSynchronizedOfferRetargetingTest() {
        Long offerRetargetingId = bidsBaseSteps.saveDefaultOfferRetargeting(cid, pid, Currency.RUB);
        api.userSteps.campaignFakeSteps().makeNewCampaignReadyForSendingToBS(cid);
        api.userSteps.groupFakeSteps().makeGroupFullyModerated(pid);
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(bid);

        RunBsTransportScriptResponse resp =
                api.userSteps.getDarkSideSteps().getTransportSteps().sendNewCampaign(shard, cid);

        Campaign campaign =
                api.userSteps.getDarkSideSteps().getTransportSteps().getClientDataRequestCampaign(resp, 0, cid);
        Context context = campaign != null ? campaign.getContext(pid) : null;
        List<OfferRetargeting> offerRetargetings = getMapValues(context != null ? context.getOfferRetargetings() : null);
        assumeThat("отправился один офферный ретаргетинг", offerRetargetings, hasSize(1));
        BidsBaseBidType expectedBidType = BidsBaseBidType.offer_retargeting;
        OfferRetargeting offerRetargeting = context.getOfferRetargeting(offerRetargetingId);
        OfferRetargeting expectedOfferRetargeting = new OfferRetargeting()
                .withEid(offerRetargetingId.toString())
                .withType(expectedBidType.toString())
                .withUpdateInfo(1)
                .withPrice(MoneyCurrency.get(Currency.RUB).getMinPrice().doubleValue())
                .withPriceContext(MoneyCurrency.get(Currency.RUB).getMinPrice().doubleValue())
                .withCurrencyIsoCode(Currency.RUB.getIsoCode())
                .withParams(new HashMap<>());
        assertThat("отправился правильный офферный ретаргетинг", offerRetargeting,
                beanDiffer(expectedOfferRetargeting).useCompareStrategy(DefaultCompareStrategies.onlyExpectedFields()));
    }
}
