package ru.yandex.autotests.directintapi.bstransport.main.relevancematch;

import java.util.Optional;

import org.jetbrains.annotations.Nullable;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BidsBaseBidType;
import ru.yandex.autotests.direct.db.steps.BidsBaseSteps;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.darkside.Logins;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.darkside.model.RunBsTransportScriptResponse;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.Campaign;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.RelevanceMatch;
import ru.yandex.autotests.directapi.enums.AdGroupType;
import ru.yandex.autotests.directapi.enums.BannerType;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directintapi.bstransport.FeatureNames;
import ru.yandex.autotests.irt.testutils.beandiffer2.BeanDifferMatcher;
import ru.yandex.autotests.irt.testutils.beandiffer2.comparestrategy.defaultcomparestrategy.DefaultCompareStrategies;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static ru.yandex.autotests.direct.utils.features.FeatureNameEnum.CONTEXT_RELEVANCE_MATCH;
import static ru.yandex.autotests.direct.utils.features.FeatureNameEnum.CONTEXT_RELEVANCE_MATCH_INTERFACE_ONLY;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.beandiffer2.BeanDifferMatcher.beanDiffer;

@Aqua.Test
@Tag(TagDictionary.TRUNK)
@Tag(TagDictionary.RELEASE)
@Description("Транспорт: проверка отправки в БК бесфразных таргетингов для new РМП кампаний")
@Features({FeatureNames.RELEVANCE_MATCH, FeatureNames.NOT_FOR_FULL_EXPORT})
public class RelevanceMatchNewRmpTest {
    private static final String LOGIN = Logins.LOGIN_TRANSPORT_NEW_RMP;

    private static final Money MIN_PRICE = MoneyCurrency.get(Currency.RUB).getMinPrice();
    private static final Money PRICE = MoneyCurrency.get(Currency.RUB).getMinPrice().multiply(5L);
    private static final Money PRICE_CONTEXT = MoneyCurrency.get(Currency.RUB).getMinPrice().multiply(4L);

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    private static int shard;
    private static BidsBaseSteps bidsBaseSteps;

    private Long mobileCampaignId;
    private Long mobileAdGroupId;
    private Long mobileAdId;

    private Long textCampaignId;
    private Long textAdGroupId;
    private Long textAdId;

    @BeforeClass
    public static void beforeClass() {
        shard = api.userSteps.clientFakeSteps().getUserShard(LOGIN);
        long contextRelevanceMatchFeatureId = api.userSteps.getDirectJooqDbSteps().featuresSteps()
                .getFeatureId(CONTEXT_RELEVANCE_MATCH.getValue());
        long contextRelevanceMatchInterfaceOnlyFeatureId = api.userSteps.getDirectJooqDbSteps().featuresSteps()
                .getFeatureId(CONTEXT_RELEVANCE_MATCH_INTERFACE_ONLY.getValue());
        long clientId = Long.valueOf(User.get(LOGIN).getClientID());
        bidsBaseSteps = api.userSteps.getDirectJooqDbSteps().useShard(shard).bidsBaseSteps();

        api.userSteps.getDirectJooqDbSteps().useShard(shard).clientsSteps()
                .setClientFeature(clientId, contextRelevanceMatchFeatureId, false);
        api.userSteps.getDirectJooqDbSteps().useShard(shard).clientsSteps()
                .setClientFeature(clientId, contextRelevanceMatchInterfaceOnlyFeatureId, false);
    }

    @Before
    public void before() {
        mobileCampaignId = api.userSteps.campaignSteps().addDefaultMobileAppCampaign();
        api.userSteps.campaignFakeSteps().makeNewCampaignReadyForSendingToBS(mobileCampaignId);
        mobileAdGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(mobileCampaignId, AdGroupType.MOBILE_CONTENT);
        mobileAdId = api.userSteps.adsSteps().addDefaultAd(mobileAdGroupId, BannerType.MOBILE_CONTENT);

        textCampaignId = api.userSteps.campaignSteps().addDefaultTextCampaign();
        api.userSteps.campaignFakeSteps().makeNewCampaignReadyForSendingToBS(textCampaignId);
        textAdGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(textCampaignId, AdGroupType.BASE);
        textAdId = api.userSteps.adsSteps().addDefaultAd(textAdGroupId, BannerType.TEXT);
    }

    @Test
    public void checkRelevanceMatchWithNetStop() {
        Long relevanceMatchId = bidsBaseSteps.saveDefaultRelevanceMatchWithPriceAndOpts(
                mobileCampaignId, mobileAdGroupId, Currency.RUB,
                PRICE.bigDecimalValue(), PRICE_CONTEXT.bigDecimalValue(), BidsBaseSteps.OPTS_NET_STOP);

        RelevanceMatch relevanceMatch = getSentRelevanceMatch(
                mobileCampaignId, mobileAdGroupId, mobileAdId, relevanceMatchId);

        assertThat("отправился правильный бесфразный таргетинг", relevanceMatch,
                beanDifferExpectedFieldsOnly(
                        new RelevanceMatch()
                                .withEid(relevanceMatchId.toString())
                                .withUpdateInfo(1)
                                .withType(BidsBaseBidType.relevance_match.toString())
                                .withPrice(PRICE.doubleValue())
                                .withPriceContext(PRICE_CONTEXT.doubleValue())
                                .withCurrencyIsoCode(Currency.RUB.getIsoCode())
                                .withRelevanceMatchType(RelevanceMatch.TYPE_BOTH)
                ));
    }

    @Test
    public void checkRelevanceMatchWithSearchStop() {
        Long relevanceMatchId = bidsBaseSteps.saveDefaultRelevanceMatchWithPriceAndOpts(
                mobileCampaignId, mobileAdGroupId, Currency.RUB,
                PRICE.bigDecimalValue(), PRICE_CONTEXT.bigDecimalValue(), BidsBaseSteps.OPTS_SEARCH_STOP);

        RelevanceMatch relevanceMatch = getSentRelevanceMatch(
                mobileCampaignId, mobileAdGroupId, mobileAdId, relevanceMatchId);

        assertThat("отправился правильный бесфразный таргетинг", relevanceMatch,
                beanDifferExpectedFieldsOnly(
                        new RelevanceMatch()
                                .withEid(relevanceMatchId.toString())
                                .withUpdateInfo(1)
                                .withType(BidsBaseBidType.relevance_match.toString())
                                .withPrice(PRICE.doubleValue())
                                .withPriceContext(PRICE_CONTEXT.doubleValue())
                                .withCurrencyIsoCode(Currency.RUB.getIsoCode())
                                .withRelevanceMatchType(RelevanceMatch.TYPE_BOTH)
                ));
    }

    @Test
    public void checkRelevanceMatchWithBothScopes() {
        Long relevanceMatchId = bidsBaseSteps.saveDefaultRelevanceMatchWithPriceAndOpts(
                mobileCampaignId, mobileAdGroupId, Currency.RUB,
                PRICE.bigDecimalValue(), PRICE_CONTEXT.bigDecimalValue(), BidsBaseSteps.OPTS_DEFAULT);

        RelevanceMatch relevanceMatch = getSentRelevanceMatch(
                mobileCampaignId, mobileAdGroupId, mobileAdId, relevanceMatchId);

        assertThat("отправился правильный бесфразный таргетинг", relevanceMatch,
                beanDifferExpectedFieldsOnly(
                        new RelevanceMatch()
                                .withEid(relevanceMatchId.toString())
                                .withUpdateInfo(1)
                                .withType(BidsBaseBidType.relevance_match.toString())
                                .withPrice(PRICE.doubleValue())
                                .withPriceContext(PRICE_CONTEXT.doubleValue())
                                .withCurrencyIsoCode(Currency.RUB.getIsoCode())
                                .withRelevanceMatchType(RelevanceMatch.TYPE_BOTH)
                ));
    }

    @Test
    public void checkRelevanceMatchWithSearchStopInTgo() {
        Long relevanceMatchId = bidsBaseSteps.saveDefaultRelevanceMatchWithPriceAndOpts(
                textCampaignId, textAdGroupId, Currency.RUB,
                PRICE.bigDecimalValue(), PRICE_CONTEXT.bigDecimalValue(), BidsBaseSteps.OPTS_SEARCH_STOP);

        RelevanceMatch relevanceMatch = getSentRelevanceMatch(
                textCampaignId, textAdGroupId, textAdId, relevanceMatchId);

        assertThat("отправился правильный бесфразный таргетинг", relevanceMatch,
                beanDifferExpectedFieldsOnly(
                        new RelevanceMatch()
                                .withEid(relevanceMatchId.toString())
                                .withUpdateInfo(1)
                                .withType(BidsBaseBidType.relevance_match.toString())
                                .withPrice(PRICE.doubleValue())
                                .withPriceContext(PRICE_CONTEXT.doubleValue())
                                .withCurrencyIsoCode(Currency.RUB.getIsoCode())
                                .withRelevanceMatchType(null)
                ));
    }

    private <T> BeanDifferMatcher<T> beanDifferExpectedFieldsOnly(T bean) {
        return beanDiffer(bean).useCompareStrategy(DefaultCompareStrategies.onlyExpectedFields());
    }

    @Nullable
    private RelevanceMatch getSentRelevanceMatch(Long campaignId, Long adGroupId, Long adId, Long relevanceMatchId) {
        api.userSteps.campaignFakeSteps().makeNewCampaignReadyForSendingToBS(campaignId);
        api.userSteps.groupFakeSteps().makeGroupFullyModerated(adGroupId);
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(adId);

        RunBsTransportScriptResponse resp = api.userSteps.getDarkSideSteps().getTransportSteps()
                .sendNewCampaign(shard, campaignId);
        Campaign campaign = api.userSteps.getDarkSideSteps().getTransportSteps()
                .getClientDataRequestCampaign(resp, campaignId);
        return Optional.ofNullable(campaign)
                .map(c -> c.getContext(adGroupId))
                .map(ctx -> ctx.getRelevanceMatch(relevanceMatchId))
                .orElse(null);
    }
}
