package ru.yandex.autotests.directintapi.bstransport.main.relevancematch;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BidsBaseBidType;
import ru.yandex.autotests.direct.db.steps.BidsBaseSteps;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.darkside.Logins;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.darkside.model.RunBsTransportScriptResponse;
import ru.yandex.autotests.directapi.darkside.model.Status;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.Campaign;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.Context;
import ru.yandex.autotests.directapi.darkside.model.bslogs.clientdata.RelevanceMatch;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directintapi.bstransport.FeatureNames;
import ru.yandex.autotests.irt.testutils.beandiffer2.comparestrategy.defaultcomparestrategy.DefaultCompareStrategies;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.nullValue;
import static ru.yandex.autotests.direct.db.steps.ClientsOptionsSteps.ClientFlagsEnum.FEATURE_CONTEXT_RELEVANCE_MATCH_ALLOWED;
import static ru.yandex.autotests.direct.db.steps.ClientsOptionsSteps.ClientFlagsEnum.FEATURE_CONTEXT_RELEVANCE_MATCH_INTERFACE_ONLY;
import static ru.yandex.autotests.direct.utils.features.FeatureNameEnum.CONTEXT_RELEVANCE_MATCH;
import static ru.yandex.autotests.direct.utils.features.FeatureNameEnum.CONTEXT_RELEVANCE_MATCH_INTERFACE_ONLY;
import static ru.yandex.autotests.directintapi.bstransport.TransportTestUtils.getMapValues;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;
import static ru.yandex.autotests.irt.testutils.beandiffer2.BeanDifferMatcher.beanDiffer;

/**
 * Created by pavryabov on 06.03.17.
 */
@Aqua.Test
@Tag(TagDictionary.TRUNK)
@Tag(TagDictionary.RELEASE)
@Description("Транспорт: проверка отправки в БК бесфразных таргетингов в различных состояниях")
@Features({FeatureNames.RELEVANCE_MATCH, FeatureNames.NOT_FOR_FULL_EXPORT})
@Issue("https://st.yandex-team.ru/DIRECT-63412")
@RunWith(Parameterized.class)
public class RelevanceMatchTest {

    private static final String LOGIN = Logins.LOGIN_TRANSPORT_RM;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trashman = new Trashman(api);

    private static int shard;
    private static BidsBaseSteps bidsBaseSteps;

    private Long cid;
    private Long pid;
    private Long bid;
    private static long featureErmId;
    private static long featureIrmId;
    private static long clientId;

    @Parameterized.Parameter(0)
    public boolean isFeatureIrmEnabled;

    @Parameterized.Parameters(name = "extendedRelevanceMatchInterfaceOnly = {0}")
    public static Collection<Object[]> data() {
        Object[][] data = new Object[][]{
                {false},
                {true}
        };
        return Arrays.asList(data);
    }


    @BeforeClass
    public static void beforeClass() {
        shard = api.userSteps.clientFakeSteps().getUserShard(LOGIN);
        bidsBaseSteps = api.userSteps.getDirectJooqDbSteps().useShard(shard).bidsBaseSteps();
        featureErmId = api.userSteps.getDirectJooqDbSteps().featuresSteps()
                .getFeatureId(CONTEXT_RELEVANCE_MATCH.getValue());
        featureIrmId = api.userSteps.getDirectJooqDbSteps().featuresSteps()
                .getFeatureId(CONTEXT_RELEVANCE_MATCH_INTERFACE_ONLY.getValue());
        clientId = Long.valueOf(User.get(LOGIN).getClientID());
    }

    @Before
    public void before() {
        cid = api.userSteps.campaignSteps().addDefaultTextCampaign();
        pid = api.userSteps.adGroupsSteps().addDefaultGroup(cid);
        bid = api.userSteps.adsSteps().addDefaultTextAd(pid);
        api.userSteps.getDirectJooqDbSteps().useShard(shard).clientsSteps()
                .setClientFeature(clientId, featureErmId, false);
        api.userSteps.getDirectJooqDbSteps().clientsOptionsSteps()
                .dropClientFlag(clientId, FEATURE_CONTEXT_RELEVANCE_MATCH_ALLOWED);
        api.userSteps.getDirectJooqDbSteps().useShard(shard).clientsSteps()
                .setClientFeature(clientId, featureIrmId, isFeatureIrmEnabled);
        if (isFeatureIrmEnabled) {
            api.userSteps.getDirectJooqDbSteps().clientsOptionsSteps()
                    .setClientFlag(clientId, FEATURE_CONTEXT_RELEVANCE_MATCH_INTERFACE_ONLY);
        } else {
            api.userSteps.getDirectJooqDbSteps().clientsOptionsSteps()
                    .dropClientFlag(clientId, FEATURE_CONTEXT_RELEVANCE_MATCH_INTERFACE_ONLY);
        }
    }


    @Test
    public void oneActiveRelevanceMatchWithNetStop() {
        oneActiveRelevanceMatchWithExtendedOpts(BidsBaseSteps.OPTS_NET_STOP, RelevanceMatch.TYPE_BOTH, true);
    }

    @Test
    public void oneActiveRelevanceMatchWithSearchStop() {
        oneActiveRelevanceMatchWithExtendedOpts(BidsBaseSteps.OPTS_SEARCH_STOP, RelevanceMatch.TYPE_BOTH, true);
    }

    @Test
    public void oneActiveRelevanceMatchWithBothScopes() {
        oneActiveRelevanceMatchWithExtendedOpts(BidsBaseSteps.OPTS_DEFAULT, RelevanceMatch.TYPE_BOTH, true);
    }

    @Test
    public void oneActiveRelevanceMatchWithNetStopWithoutExtendedRelevanceMatchFeature() {
        oneActiveRelevanceMatchWithExtendedOpts(BidsBaseSteps.OPTS_NET_STOP,
                isFeatureIrmEnabled ? RelevanceMatch.TYPE_BOTH : null,
                false);
    }

    private void oneActiveRelevanceMatchWithExtendedOpts(String opts,
            String expectedType,
            boolean isExtendedRelevanceMatchFeatureEnabled
    )
    {
        if (isExtendedRelevanceMatchFeatureEnabled) {
            api.userSteps.getDirectJooqDbSteps().useShard(shard).clientsSteps()
                    .setClientFeature(clientId, featureErmId, true);
            api.userSteps.getDirectJooqDbSteps().clientsOptionsSteps()
                    .setClientFlag(clientId, FEATURE_CONTEXT_RELEVANCE_MATCH_ALLOWED);
        }

        Long relevanceMatchId = bidsBaseSteps.saveDefaultRelevanceMatchWithOpts(cid, pid, Currency.RUB, opts);

        api.userSteps.campaignFakeSteps().makeNewCampaignReadyForSendingToBS(cid);
        api.userSteps.groupFakeSteps().makeGroupFullyModerated(pid);
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(bid);

        RunBsTransportScriptResponse resp = api.userSteps.getDarkSideSteps().getTransportSteps()
                .sendNewCampaign(shard, cid);
        Campaign campaign = api.userSteps.getDarkSideSteps().getTransportSteps()
                .getClientDataRequestCampaign(resp, 0, cid);
        Context context = campaign != null ? campaign.getContext(pid) : null;
        List<RelevanceMatch> relevanceMatches = getMapValues(context != null ? context.getRelevanceMatches() : null);
        assumeThat("отправился один бесфразный таргетинг", relevanceMatches, hasSize(1));
        BidsBaseBidType expectedBidType = BidsBaseBidType.relevance_match;
        RelevanceMatch relevanceMatch = context.getRelevanceMatch(relevanceMatchId);
        RelevanceMatch expectedRelevanceMatch = new RelevanceMatch()
                .withEid(relevanceMatchId.toString())
                .withType(expectedBidType.toString())
                .withUpdateInfo(1)
                .withRelevanceMatchType(expectedType);
        assertThat("отправился правильный бесфразный таргетинг", relevanceMatch,
                beanDiffer(expectedRelevanceMatch).useCompareStrategy(DefaultCompareStrategies.onlyExpectedFields()));
    }


    @Test
    public void oneActiveRelevanceMatchFullStructureTest() {
        Long relevanceMatchId = bidsBaseSteps.saveDefaultRelevanceMatch(cid, pid, Currency.RUB);
        api.userSteps.campaignFakeSteps().makeNewCampaignReadyForSendingToBS(cid);
        api.userSteps.groupFakeSteps().makeGroupFullyModerated(pid);
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(bid);

        RunBsTransportScriptResponse resp = api.userSteps.getDarkSideSteps().getTransportSteps()
                .sendNewCampaign(shard, cid);
        Campaign campaign = api.userSteps.getDarkSideSteps().getTransportSteps()
                .getClientDataRequestCampaign(resp, 0, cid);
        Context context = campaign != null ? campaign.getContext(pid) : null;
        List<RelevanceMatch> relevanceMatches = getMapValues(context != null ? context.getRelevanceMatches() : null);
        assumeThat("отправился один бесфразный таргетинг", relevanceMatches, hasSize(1));
        RelevanceMatch relevanceMatch = context.getRelevanceMatch(relevanceMatchId);
        BidsBaseBidType expectedBidType = BidsBaseBidType.relevance_match;
        RelevanceMatch expectedRelevanceMatch = new RelevanceMatch()
                .withEid(relevanceMatchId.toString())
                .withType(expectedBidType.toString())
                .withUpdateInfo(1)
                .withPrice(MoneyCurrency.get(Currency.RUB).getMinPrice().doubleValue())
                .withPriceContext(MoneyCurrency.get(Currency.RUB).getMinPrice().doubleValue())
                .withCurrencyIsoCode(Currency.RUB.getIsoCode())
                .withParams(new HashMap<>());
        assertThat("отправился правильный бесфразный таргетинг", relevanceMatch,
                beanDiffer(expectedRelevanceMatch).useCompareStrategy(DefaultCompareStrategies.onlyExpectedFields()));
    }

    @Test
    public void oneSuspendedRelevanceMatch() {
        bidsBaseSteps.saveDefaultRelevanceMatchWithOpts(cid, pid, Currency.RUB, BidsBaseSteps.OPTS_SUSPENDED);
        api.userSteps.campaignFakeSteps().makeNewCampaignReadyForSendingToBS(cid);
        api.userSteps.groupFakeSteps().makeGroupFullyModerated(pid);
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(bid);

        RunBsTransportScriptResponse resp =
                api.userSteps.getDarkSideSteps().getTransportSteps().sendNewCampaign(shard, cid);

        Campaign campaign =
                api.userSteps.getDarkSideSteps().getTransportSteps().getClientDataRequestCampaign(resp, 0, cid);
        Context context = campaign != null ? campaign.getContext(pid) : null;
        assertThat("контекст не отправился", context, nullValue());
    }

    @Test
    public void oneDeletedRelevanceMatch() {
        bidsBaseSteps.saveDefaultRelevanceMatchWithOpts(cid, pid, Currency.RUB, BidsBaseSteps.OPTS_DELETED);
        api.userSteps.campaignFakeSteps().makeNewCampaignReadyForSendingToBS(cid);
        api.userSteps.groupFakeSteps().makeGroupFullyModerated(pid);
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(bid);

        RunBsTransportScriptResponse resp = api.userSteps.getDarkSideSteps().getTransportSteps()
                .sendNewCampaign(shard, cid);
        Campaign campaign = api.userSteps.getDarkSideSteps().getTransportSteps()
                .getClientDataRequestCampaign(resp, 0, cid);
        Context context = campaign != null ? campaign.getContext(pid) : null;
        assertThat("контекст не отправился", context, nullValue());
    }

    @Test
    public void oneRelevanceMatchAfterSuspended() {
        Long relevanceMatchId = bidsBaseSteps.saveDefaultRelevanceMatch(cid, pid, Currency.RUB);
        api.userSteps.campaignFakeSteps().makeNewCampaignReadyForSendingToBS(cid);
        api.userSteps.groupFakeSteps().makeGroupFullyModerated(pid);
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(bid);

        api.userSteps.getDarkSideSteps().getTransportSteps().sendNewCampaign(shard, cid);

        bidsBaseSteps.setBidsBaseOpts(relevanceMatchId, BidsBaseSteps.OPTS_SUSPENDED);
        api.userSteps.groupFakeSteps().setGroupFakeStatusBsSynced(pid, Status.NO);

        RunBsTransportScriptResponse resp =
                api.userSteps.getDarkSideSteps().getTransportSteps().sendSyncedCampaign(shard, cid);

        Campaign campaign =
                api.userSteps.getDarkSideSteps().getTransportSteps().getClientDataRequestCampaign(resp, 0, cid);
        Context context = campaign != null ? campaign.getContext(pid) : null;
        List<RelevanceMatch> relevanceMatches = getMapValues(context != null ? context.getRelevanceMatches() : null);
        assertThat("остановленный бесфразный таргетинг не отправился", relevanceMatches, hasSize(0));
    }

    @Test
    public void oneRelevanceMatchAfterDeleted() {
        Long relevanceMatchId = bidsBaseSteps.saveDefaultRelevanceMatch(cid, pid, Currency.RUB);
        api.userSteps.campaignFakeSteps().makeNewCampaignReadyForSendingToBS(cid);
        api.userSteps.groupFakeSteps().makeGroupFullyModerated(pid);
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(bid);

        api.userSteps.getDarkSideSteps().getTransportSteps().sendNewCampaign(shard, cid);

        bidsBaseSteps.setBidsBaseOpts(relevanceMatchId, BidsBaseSteps.OPTS_DELETED);
        api.userSteps.groupFakeSteps().setGroupFakeStatusBsSynced(pid, Status.NO);

        RunBsTransportScriptResponse resp =
                api.userSteps.getDarkSideSteps().getTransportSteps().sendSyncedCampaign(shard, cid);

        Campaign campaign =
                api.userSteps.getDarkSideSteps().getTransportSteps().getClientDataRequestCampaign(resp, 0, cid);
        Context context = campaign != null ? campaign.getContext(pid) : null;
        List<RelevanceMatch> relevanceMatches = getMapValues(context != null ? context.getRelevanceMatches() : null);
        assertThat("удаленный бесфразный таргетинг не отправился", relevanceMatches, hasSize(0));
    }

    @Test
    public void oneSuspendedRelevanceMatchAndPhrase() {
        Long bidsId = api.userSteps.keywordsSteps().addDefaultKeyword(pid);
        bidsBaseSteps.saveDefaultRelevanceMatchWithOpts(cid, pid, Currency.RUB, BidsBaseSteps.OPTS_SUSPENDED);
        api.userSteps.campaignFakeSteps().makeNewCampaignReadyForSendingToBS(cid);
        api.userSteps.groupFakeSteps().makeGroupFullyModerated(pid);
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(bid);
        api.userSteps.phrasesFakeSteps().makeKeywordModerated(bidsId);

        RunBsTransportScriptResponse resp =
                api.userSteps.getDarkSideSteps().getTransportSteps().sendNewCampaign(shard, cid);

        Campaign campaign =
                api.userSteps.getDarkSideSteps().getTransportSteps().getClientDataRequestCampaign(resp, 0, cid);
        Context context = campaign != null ? campaign.getContext(pid) : null;
        List<RelevanceMatch> relevanceMatches = getMapValues(context != null ? context.getRelevanceMatches() : null);
        assertThat("остановленный бесфразный таргетинг не отправился", relevanceMatches, hasSize(0));
    }

    @Test
    public void oneDeletedRelevanceMatchAndPhrase() {
        Long bidsId = api.userSteps.keywordsSteps().addDefaultKeyword(pid);
        bidsBaseSteps.saveDefaultRelevanceMatchWithOpts(cid, pid, Currency.RUB, BidsBaseSteps.OPTS_DELETED);
        api.userSteps.campaignFakeSteps().makeNewCampaignReadyForSendingToBS(cid);
        api.userSteps.groupFakeSteps().makeGroupFullyModerated(pid);
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(bid);
        api.userSteps.phrasesFakeSteps().makeKeywordModerated(bidsId);

        RunBsTransportScriptResponse resp =
                api.userSteps.getDarkSideSteps().getTransportSteps().sendNewCampaign(shard, cid);

        Campaign campaign =
                api.userSteps.getDarkSideSteps().getTransportSteps().getClientDataRequestCampaign(resp, 0, cid);
        Context context = campaign != null ? campaign.getContext(pid) : null;
        List<RelevanceMatch> relevanceMatches = getMapValues(context != null ? context.getRelevanceMatches() : null);
        assertThat("удаленный бесфразный таргетинг не отправился", relevanceMatches, hasSize(0));
    }

    @Test
    public void twoActiveAndSuspendedAndDeletedRelevanceMatches() {
        Long relevanceMatchId = bidsBaseSteps.saveDefaultRelevanceMatch(cid, pid, Currency.RUB);
        Long relevanceMatchIdElse = bidsBaseSteps.saveDefaultRelevanceMatch(cid, pid, Currency.RUB);
        bidsBaseSteps.saveDefaultRelevanceMatchWithOpts(cid, pid, Currency.RUB, BidsBaseSteps.OPTS_SUSPENDED);
        bidsBaseSteps.saveDefaultRelevanceMatchWithOpts(cid, pid, Currency.RUB, BidsBaseSteps.OPTS_DELETED);
        api.userSteps.campaignFakeSteps().makeNewCampaignReadyForSendingToBS(cid);
        api.userSteps.groupFakeSteps().makeGroupFullyModerated(pid);
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(bid);

        RunBsTransportScriptResponse resp =
                api.userSteps.getDarkSideSteps().getTransportSteps().sendNewCampaign(shard, cid);

        Campaign campaign =
                api.userSteps.getDarkSideSteps().getTransportSteps().getClientDataRequestCampaign(resp, 0, cid);
        Context context = campaign != null ? campaign.getContext(pid) : null;
        List<RelevanceMatch> relevanceMatches = getMapValues(context != null ? context.getRelevanceMatches() : null);
        assumeThat("отправилось два бесфразных таргетинга", relevanceMatches, hasSize(2));
        Map<String, RelevanceMatch> relevanceMatchMap = context.getRelevanceMatches();
        Map<String, RelevanceMatch> expectedRelevanceMatchMap = new HashMap<>();
        BidsBaseBidType expectedBidType = BidsBaseBidType.relevance_match;

                expectedRelevanceMatchMap.put(relevanceMatchId.toString(), new RelevanceMatch()
                .withEid(relevanceMatchId.toString())
                .withType(expectedBidType.toString())
                .withUpdateInfo(1));
        expectedRelevanceMatchMap.put(relevanceMatchIdElse.toString(), new RelevanceMatch()
                .withEid(relevanceMatchIdElse.toString())
                .withType(expectedBidType.toString())
                .withUpdateInfo(1));
        assertThat("отправился правильный бесфразный таргетинг", relevanceMatchMap,
                beanDiffer(expectedRelevanceMatchMap).useCompareStrategy(
                        DefaultCompareStrategies.onlyExpectedFields()));
    }

    @Test
    public void oneActiveSynchronizedRelevanceMatchTest() {
        Long relevanceMatchId = bidsBaseSteps.saveDefaultRelevanceMatch(cid, pid, Currency.RUB);
        api.userSteps.campaignFakeSteps().makeNewCampaignReadyForSendingToBS(cid);
        api.userSteps.groupFakeSteps().makeGroupFullyModerated(pid);
        api.userSteps.bannersFakeSteps().makeBannerFullyModerated(bid);

        RunBsTransportScriptResponse resp =
                api.userSteps.getDarkSideSteps().getTransportSteps().sendNewCampaign(shard, cid);

        Campaign campaign =
                api.userSteps.getDarkSideSteps().getTransportSteps().getClientDataRequestCampaign(resp, 0, cid);
        Context context = campaign != null ? campaign.getContext(pid) : null;
        List<RelevanceMatch> relevanceMatches = getMapValues(context != null ? context.getRelevanceMatches() : null);
        assumeThat("отправился один бесфразный таргетинг", relevanceMatches, hasSize(1));
        BidsBaseBidType expectedBidType = BidsBaseBidType.relevance_match;
        RelevanceMatch relevanceMatch = context.getRelevanceMatch(relevanceMatchId);
        RelevanceMatch expectedRelevanceMatch = new RelevanceMatch()
                .withEid(relevanceMatchId.toString())
                .withType(expectedBidType.toString())
                .withUpdateInfo(1)
                .withPrice(MoneyCurrency.get(Currency.RUB).getMinPrice().doubleValue())
                .withPriceContext(MoneyCurrency.get(Currency.RUB).getMinPrice().doubleValue())
                .withCurrencyIsoCode(Currency.RUB.getIsoCode())
                .withParams(new HashMap<>());
        assertThat("отправился правильный бесфразный таргетинг", relevanceMatch,
                beanDiffer(expectedRelevanceMatch).useCompareStrategy(DefaultCompareStrategies.onlyExpectedFields()));
    }
}
