package ru.yandex.autotests.directintapi.bstransport.matchers;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import com.google.common.collect.ImmutableMap;
import one.util.streamex.EntryStream;
import org.hamcrest.Description;
import org.hamcrest.TypeSafeMatcher;

import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BsExportQueueRecord;

import static com.google.common.base.Preconditions.checkNotNull;

public class QueueRecordMatcher extends TypeSafeMatcher<BsExportQueueRecord> {
    private static final String CAMPS = "camps_num";
    private static final String CONTEXTS = "contexts_num";
    private static final String BANNERS = "banners_num";
    private static final String BIDS = "bids_num";
    private static final String PRICES = "prices_num";
    private static final String FULL_EXPORT = "is_full_export";

    private static final Map<String, Function<BsExportQueueRecord, Number>> mapping =
            ImmutableMap.<String, Function<BsExportQueueRecord, Number>>builder()
                    .put(CAMPS, BsExportQueueRecord::getCampsNum)
                    .put(CONTEXTS, BsExportQueueRecord::getContextsNum)
                    .put(BANNERS, BsExportQueueRecord::getBannersNum)
                    .put(BIDS, BsExportQueueRecord::getBidsNum)
                    .put(PRICES, BsExportQueueRecord::getPricesNum)
                    .put(FULL_EXPORT, BsExportQueueRecord::getIsFullExport)
                    .build();

    private Map<String, Boolean> expectations;

    public QueueRecordMatcher() {
        expectations = new HashMap<>();
    }

    private QueueRecordMatcher with(String what, Boolean state) {
        checkNotNull(state);
        expectations.put(what, state);
        return this;
    }

    public QueueRecordMatcher withCamps() {
        return with(CAMPS, true);
    }

    public QueueRecordMatcher withoutCamps() {
        return with(CAMPS, false);
    }

    public QueueRecordMatcher withContexts(Boolean state) {
        return with(CONTEXTS, state);
    }

    public QueueRecordMatcher withContexts() {
        return withContexts(true);
    }

    public QueueRecordMatcher withoutContexts() {
        return withContexts(false);
    }

    public QueueRecordMatcher withBanners(Boolean state) {
        return with(BANNERS, state);
    }

    public QueueRecordMatcher withBanners() {
        return withBanners(true);
    }

    public QueueRecordMatcher withoutBanners() {
        return withBanners(false);
    }

    public QueueRecordMatcher withBids() {
        return with(BIDS, true);
    }

    public QueueRecordMatcher withoutBids() {
        return with(BIDS, false);
    }

    public QueueRecordMatcher withPrices() {
        return with(PRICES, true);
    }

    public QueueRecordMatcher withoutPrices() {
        return with(PRICES, false);
    }

    public QueueRecordMatcher withFullExport() {
        return with(FULL_EXPORT, true);
    }

    public QueueRecordMatcher withoutFullExport() {
        return with(FULL_EXPORT, false);
    }

    @Override
    protected boolean matchesSafely(BsExportQueueRecord actual) {
        if (actual == null) {
            return false;
        }

        for (Map.Entry<String, Boolean> entry : expectations.entrySet()) {
            Number value = mapping.get(entry.getKey()).apply(actual);
            if (value == null) {
                return false;
            }
            if ((value.longValue() > 0) != entry.getValue()) {
                return false;
            }
        }

        return true;
    }

    @Override
    protected void describeMismatchSafely(BsExportQueueRecord item, Description mismatchDescription) {
        if (item == null) {
            mismatchDescription.appendText("was null");
            return;
        }

        List<String> values = EntryStream.of(expectations)
                .keys()
                .sorted()
                .mapToEntry(mapping::get)
                .mapValues(v -> v.apply(item))
                .mapKeyValue((k, v) -> k + " = " + v)
                .toList();

        mismatchDescription.appendValueList("record with: ", ", ", "", values);
    }

    @Override
    public void describeTo(Description description) {
        List<String> values = EntryStream.of(expectations)
                .sortedBy(Map.Entry::getKey)
                .mapValues(v -> v ? " > 0" : " = 0")
                .mapKeyValue((k, v) -> k + v)
                .toList();

        description.appendValueList("record with: ", ", ", "", values);
    }
}
