package ru.yandex.autotests.directintapi.bstransport.matchers.flags;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.hamcrest.Description;
import org.hamcrest.TypeSafeMatcher;

import java.util.*;
import java.util.regex.Pattern;

/**
 * Матчер поля Flags на уровне баннера. Значение поля в транспорте
 * зависит от значения поля в БД ppc:banners.flags, которое
 * содержит флаги модерации через запятую. Значение поля Flags
 * представляет собой перечисление тех же флагов модерации из БД
 * через запятую без сохранения порядка и с некоторыми исключениями:
 * 1. когда среди флагов модерации присутствует флаг "age:[число]",
 *    то в поле Flags вместо него подставляется флаг "plus18"
 *    (а в поле Age отправляется возрастная метка в виде "[число]+");
 * 2. когда среди флагов модерации присутствует флаг "baby_food:[число]",
 *    то в поле Flags вместо него подставляется флаг "baby_food_[число]".
 */
public class FlagsMatcher extends TypeSafeMatcher<String> {

    /**
     * @param dbFlags значение поля в БД ppc:banners.flags
     * @return матчер для поля Flags на уровне баннера
     */
    public static FlagsMatcher transportFlagsCompliesWithDb(String dbFlags) {
        return new FlagsMatcher(dbFlags);
    }

    private static final Pattern AGE_PATTERN = Pattern.compile("age:\\d+");
    private static final Pattern BABY_FOOD_PATTERN = Pattern.compile("baby_food:\\d+");

    private List<String> expectedTransportFlagsList;

    private FlagsMatcher(String dbFlags) {
        if (dbFlags == null) {
            return;
        }
        List<String> dbFlagsList = new ArrayList<>(Arrays.asList(dbFlags.split(",")));
        if (hasDuplicates(dbFlagsList)) {
            throw new IllegalArgumentException("в списке флагов найдены дубликаты: " + dbFlags);
        }
        expectedTransportFlagsList = dbToTransport(dbFlagsList);
    }

    @Override
    public boolean matchesSafely(String transportFlags) {
        if (CollectionUtils.isEmpty(expectedTransportFlagsList)) {
            return transportFlags.equals("");
        }
        List<String> transportFlagsList = new ArrayList<>(Arrays.asList(transportFlags.split(",")));
        Collections.sort(expectedTransportFlagsList);
        return !hasDuplicates(transportFlagsList) &&
                transportFlagsList.equals(expectedTransportFlagsList);
    }

    @Override
    public void describeTo(Description description) {
        if (CollectionUtils.isEmpty(expectedTransportFlagsList)) {
            description.appendText("\"\"");
        }
        String expectedTransportFlags = StringUtils.join(expectedTransportFlagsList, ",");
        description.appendText(expectedTransportFlags);
    }

    private List<String> dbToTransport(List<String> dbFlags) {
        List<String> transportFlags = new LinkedList<>();
        for (String dbFlag : dbFlags) {
            if (AGE_PATTERN.matcher(dbFlag).matches()) {
                continue;
            }
            if (BABY_FOOD_PATTERN.matcher(dbFlag).matches()) {
                transportFlags.add(dbFlag.replace(":", "_"));
                continue;
            }
            transportFlags.add(dbFlag);
        }
        return transportFlags;
    }

    private <E> boolean hasDuplicates(List<E> list) {
        for (int i = 0; i < list.size() - 1; i++) {
            E elem1 = list.get(i);
            for (int j = i + 1; j < list.size(); j++) {
                E elem2 = list.get(j);
                if (elem1.equals(elem2)) {
                    return true;
                }
            }
        }
        return false;
    }
}
