package ru.yandex.autotests.directintapi.bstransport.matchers.timetarget;

import com.yandex.direct.api.v5.campaigns.TimeTargetingAdd;
import com.yandex.direct.api.v5.general.YesNoEnum;
import ru.yandex.autotests.directapi.common.api45.TimeTargetInfo;
import ru.yandex.autotests.directapi.common.api45.TimeTargetItem;
import ru.yandex.autotests.directapi.model.common.Value;

import java.util.Arrays;

public class TimeTargetCoeffUtils {

    private static final String DONT_SHOW_ALL_DAY_COEFFS = "aaaaaaaaaaaaaaaaaaaaaaaa";
    private static final String SHOW_IN_HOLIDAYS_AS_IN_WORKDAYS = "zzzzzzzzzzzzzzzzzzzzzzzz";
    private static final String COEFF_100 = "k";

    /**
     * Преобразование коэффициентов ставок по часам в формате API
     * в формат транспорта в БК.
     * <p/>
     * Формат коэффициентов ставок транспорта в БК представляет из себя
     * массив символов размером (24 * 7 + 24); позиция каждого символа
     * с 0 до (24 * 7 - 1) - номер часа в неделе, а позиция каждого
     * символа с (24 * 7) до (24 * 8 - 1) - номер часа в праздничные дни,
     * а значение каждого символа указывает на размер коэффициента в тот час,
     * которому соответствует его позиция. Символ "a" - коэффициент равен 0,
     * или, другими словами, показы выключены. Символ "b" - 10%, "с" - 20%,
     * и так до 200%.
     * <p/>
     * Например, если первые 10 символов равны "aaaaakkkkk" - это значит,
     * что в понедельник с 0 до 5 часов показы выключены, а с 5 до 10 -
     * включены с коэффициентом 100%.
     *
     * @param timeTargetInfo временной таргетинг в формате API
     * @return коэффициенты ставок в формате транспорта в БК
     */
    public static String timeTargetToBidCoeff(TimeTargetInfo timeTargetInfo) {
        /*
             Если нет ни одного коэффициента отличного от 100,
             тогда в БК отправляется пустая строка
         */
        if (!checkForNotFullCoeffs(timeTargetInfo)) {
            return "";
        }

        /**
         * Выходное значение формируется в цикле по дням
         * с понедельника по воскресенье конкатенацией.
         *
         * Для каждого дня определяется массив часов,
         * в которые показы включены.
         *
         * Если в определенный день показы выключены, то
         * к результирующей строке добавляется 24 символа "a".
         */
        StringBuilder stringBuilder = new StringBuilder();
        for (int day = 1; day <= 7; day++) {
            // получаем массив часов, в которые показы включены
            int[] hours = findHoursForDay(timeTargetInfo, day);
            int[] coeffs = findCoeffsForDay(timeTargetInfo, day);
            // если в данный день показы выключены, добавляем нулевые коэффициенты
            if (hours == null) {
                stringBuilder.append(DONT_SHOW_ALL_DAY_COEFFS);
                continue;
            }
            // если показы включены, то добавляем к строке символы, соответствующие коэффициентам
            for (int hour = 0; hour <= 23; hour++) {
                int coeff = getCoeffForHour(hours, coeffs, hour);
                stringBuilder.append(BID_COEFF_CODES[coeff / 10]);
            }
        }

        // добавляем к строке коэффициенты для праздничных дней
        stringBuilder.append(getHolidaysCoeffs(timeTargetInfo));
        return stringBuilder.toString();
    }

    /**
     * @param timeTargetInfo временной таргетинг в формате API
     * @return строка коэффициентов ставок для праздничных дней (24 символа)
     */
    private static String getHolidaysCoeffs(TimeTargetInfo timeTargetInfo) {
        StringBuilder stringBuilder = new StringBuilder();
        if (timeTargetInfo.getShowOnHolidays().equals(Value.YES)) {
            if (timeTargetInfo.getHolidayShowFrom() == null ||
                    timeTargetInfo.getHolidayShowTo() == null) {
                stringBuilder.append(SHOW_IN_HOLIDAYS_AS_IN_WORKDAYS);
            } else {
                int from = timeTargetInfo.getHolidayShowFrom();
                int to = timeTargetInfo.getHolidayShowTo();
                for (int hour = 0; hour < from; hour++) {
                    stringBuilder.append(BID_COEFF_CODES[0]);
                }
                for (int hour = from; hour < to; hour++) {
                    stringBuilder.append(COEFF_100);
                }
                for (int hour = to; hour <= 23; hour++) {
                    stringBuilder.append(BID_COEFF_CODES[0]);
                }
            }
        } else {
            stringBuilder.append(DONT_SHOW_ALL_DAY_COEFFS);
        }
        return stringBuilder.toString();
    }

    /**
     * @param timeTargetInfo временной таргетинг в формате API
     * @param day            день, для которого необходимо вернуть список часов
     * @return список часов для определенного дня, в которые показы включены
     */
    private static int[] findHoursForDay(TimeTargetInfo timeTargetInfo, int day) {
        for (TimeTargetItem timeTargetItem : timeTargetInfo.getDaysHours()) {
            int[] days = timeTargetItem.getDays();
            for (int d : days) {
                if (d == day) {
                    return timeTargetItem.getHours();
                }
            }
        }
        return null;
    }

    /**
     * @param timeTargetInfo временной таргетинг в формате API
     * @param day            день, для которого необходимо вернуть список коэффициентов
     * @return список коэффициентов для часов определенного дня
     */
    private static int[] findCoeffsForDay(TimeTargetInfo timeTargetInfo, int day) {
        for (TimeTargetItem timeTargetItem : timeTargetInfo.getDaysHours()) {
            int[] days = timeTargetItem.getDays();
            for (int d : days) {
                if (d == day) {
                    return timeTargetItem.getBidCoefs();
                }
            }
        }
        return null;
    }

    /**
     * @param hours  список часов, в которые показы включены
     * @param coeffs список коэффициентов
     * @param hour   час, для которого требуется коэффициент
     * @return коэффициент для определенного часа; если часа нет в исходном массиве,
     * то есть показы в этот час выключены, то возвращается 0
     */
    private static int getCoeffForHour(int[] hours, int[] coeffs, int hour) {
        for (int i = 0; i < hours.length; i++) {
            if (hours[i] == hour) {
                return coeffs[i];
            }
        }
        return 0;
    }

    /**
     * @param timeTargetInfo временной таргетинг в формате API
     * @return true, если присутствует хотя бы один коэффициент, отличный от 100%
     * (в случае, когда все коэффициенты равны 100%, в БК отправляется пустой параметр)
     */
    private static boolean checkForNotFullCoeffs(TimeTargetInfo timeTargetInfo) {
        for (TimeTargetItem timeTargetItem : timeTargetInfo.getDaysHours()) {
            for (int coeff : timeTargetItem.getBidCoefs()) {
                if (coeff != 100) {
                    return true;
                }
            }
        }
        return false;
    }

    private static final char[] BID_COEFF_CODES = new char[]{
            'a', // = 0
            'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k',
            'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u'
    };

    /**
     * Преобразование коэффициентов ставок по часам в формате API5
     * в формат транспорта в БК.
     * <p/>
     * Принцип такой же, как и для API4. Отличается только формат входных данных
     * <p/>
     *
     * @param timeTargetingAdd временной таргетинг в формате API5
     * @return коэффициенты ставок в формате транспорта в БК
     */
    public static String timeTargetingAddToBidCoeff(TimeTargetingAdd timeTargetingAdd) {
        int[] numbers = getScheduleAsNumbers(timeTargetingAdd.getSchedule().getItems().toArray(new String[]{}));
        if (!checkForNotFullCoeffs(numbers)) {
            return "";
        }
        StringBuilder stringBuilder = new StringBuilder();
        for (int number : numbers) {
            stringBuilder.append(BID_COEFF_CODES[number / 10]);
        }
        stringBuilder.append(getHolidaysCoeffs(timeTargetingAdd));
        return stringBuilder.toString();
    }

    private static int[] getScheduleAsNumbers(String[] schedule) {
        int[] numbers = new int[24 * 7];
        Arrays.fill(numbers, 100);
        for (String dayAsString : schedule) {
            int[] dayAsNumbers = getStringAsNumbers(dayAsString);
            int numberOfDay = dayAsNumbers[0];
            int startNumber = (numberOfDay - 1) * 24;
            for (int i = startNumber; i < startNumber + 24; i++) {
                numbers[i] = dayAsNumbers[i - startNumber + 1];
            }
        }
        return numbers;
    }

    private static int[] getStringAsNumbers(String inputString) {
        String[] parts = inputString.split(",");
        int[] numbers = new int[parts.length];
        for (int i = 0; i < parts.length; i++) {
            numbers[i] = Integer.parseInt(parts[i]);
        }
        return numbers;
    }

    private static boolean checkForNotFullCoeffs(int[] numbers) {
        for (int number : numbers) {
            if (number != 100) {
                return true;
            }
        }
        return false;
    }

    private static String getHolidaysCoeffs(TimeTargetingAdd timeTargetingAdd) {
        StringBuilder stringBuilder = new StringBuilder();
        if (timeTargetingAdd.getHolidaysSchedule() == null) {
            stringBuilder.append(SHOW_IN_HOLIDAYS_AS_IN_WORKDAYS);
            return stringBuilder.toString();
        }
        if (timeTargetingAdd.getHolidaysSchedule().getSuspendOnHolidays().equals(YesNoEnum.NO)) {
            int from = timeTargetingAdd.getHolidaysSchedule().getStartHour();
            int to = timeTargetingAdd.getHolidaysSchedule().getEndHour();
            for (int hour = 0; hour < from; hour++) {
                stringBuilder.append(BID_COEFF_CODES[0]);
            }
            for (int hour = from; hour < to; hour++) {
                stringBuilder.append(BID_COEFF_CODES[timeTargetingAdd.getHolidaysSchedule().getBidPercent() / 10]);
            }
            for (int hour = to; hour <= 23; hour++) {
                stringBuilder.append(BID_COEFF_CODES[0]);
            }
        } else {
            stringBuilder.append(DONT_SHOW_ALL_DAY_COEFFS);
        }
        return stringBuilder.toString();
    }

}
