package ru.yandex.autotests.directapi.darkside;

import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.concurrent.ThreadLocalRandom;

import ru.yandex.autotests.direct.utils.PhraseTexts;

public class BannerSystemUtils {
    public static BigInteger getRandomYabsMd5HalfHash() {
        return new BigInteger(64, ThreadLocalRandom.current());
    }

    /**
     * Вычисление хеша по строке алгоритмом, используемым в БК для вычисления DataMD5(Phrase)
     *
     * @param str строка, по которой требуется вычислить md5-хэш
     * @return {@link BigInteger}, значения которого соответствуют {@code unsigned long}
     * @implNote портированная версия БКшной реализации на Perl
     * @see <a href=https://a.yandex-team.ru/arc/trunk/arcadia/yabs/basic_packages/yabs-base-modules/Yabs/Funcs.pm#L1408>Yabs::Funcs::md5int</a>
     */
    public static BigInteger getYabsMd5HalfHashUtf8(String str) {
        byte[] md5 = getMd5MessageDigest().digest(str.getBytes(StandardCharsets.UTF_8));
        ByteBuffer buffer = ByteBuffer.wrap(md5);

        // четыре 32-х битных числа (беззнаковые int), составляющие 128 бит md5
        long[] parts = new long[4];
        for (int i = 0; i < parts.length; i++) {
            parts[i] = Integer.toUnsignedLong(buffer.getInt(i * Integer.BYTES));
        }

        BigInteger high = BigInteger.valueOf(parts[1] ^ parts[3]);
        BigInteger low = BigInteger.valueOf(parts[0] ^ parts[2]);

        return high.shiftLeft(Integer.SIZE).or(low);
    }

    // Отдать известный PhraseID если он есть, иначе хешированный
    public static BigInteger getBsPhraseId(String str) {
        BigInteger md5 = getYabsMd5HalfHashUtf8(str);
        return PhraseTexts.PHRASES.getOrDefault(str, md5);
    }

    private static MessageDigest getMd5MessageDigest() {
        try {
            return MessageDigest.getInstance("MD5");
        } catch (NoSuchAlgorithmException e) {
            throw new IllegalStateException("can not retrieve java.security.MessageDigest for MD5 algorithm", e);
        }
    }
}
