package ru.yandex.autotests.directapi.darkside.clients;

import java.io.IOException;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;

import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.ws.rs.client.ClientBuilder;
import javax.ws.rs.client.ClientRequestContext;
import javax.ws.rs.client.ClientRequestFilter;
import javax.ws.rs.client.WebTarget;
import javax.ws.rs.core.MultivaluedMap;

import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.glassfish.jersey.apache.connector.ApacheClientProperties;
import org.glassfish.jersey.client.ClientProperties;
import org.glassfish.jersey.client.HttpUrlConnectorProvider;

import ru.yandex.autotests.direct.utils.clients.tvm.ServiceTicketProvider;
import ru.yandex.autotests.direct.utils.clients.tvm.ServiceTicketProviderProxy;
import ru.yandex.autotests.direct.utils.config.DirectTestRunProperties;
import ru.yandex.autotests.directapi.darkside.connection.DarksideConfig;
import ru.yandex.autotests.directapi.darkside.exceptions.DarkSideException;
import ru.yandex.autotests.irt.jersey.logging.ClientLoggingFilter;

import static ru.yandex.autotests.direct.utils.clients.tvm.ServiceTicketProvider.SERVICE_TICKET_HEADER;

public class IntapiJsonRestClient extends BaseIntapiClient {

    private Class<? extends ClientRequestFilter> tvmServiceTicketFilterClass;

    public IntapiJsonRestClient(DarksideConfig config) {
        super(config, "JSON_REST");
    }

    public IntapiJsonRestClient(DarksideConfig config,
                                Class<? extends ClientRequestFilter> tvmServiceTicketFilterClass) {
        super(config, "JSON_REST");
        this.tvmServiceTicketFilterClass = tvmServiceTicketFilterClass;
    }

    private void disableSslVerification(ClientBuilder clientBuilder) {
        try {
            // Create a trust manager that does not validate certificate chains
            TrustManager[] trustAllCerts = new TrustManager[]{new CertificatesUtils.IgnoreSSLTrustManager()};

            // Install the all-trusting trust manager
            SSLContext sc = SSLContext.getInstance("SSL");
            sc.init(null, trustAllCerts, new java.security.SecureRandom());
            clientBuilder
                    .sslContext(sc)
                    .hostnameVerifier(new CertificatesUtils.AllowAllHostnameVerifier());
        } catch (NoSuchAlgorithmException | KeyManagementException e) {
            throw new DarkSideException("Не удалось отключить SslVerification для IntapiJsonRestClient", e);
        }
    }

    public WebTarget initClient(String serviceName, String methodName) {
        return initClientInternal(serviceName)
                .register(new ClientLoggingFilter("Intapi JSON REST Client: " + serviceName + ":" + methodName))
                .path(methodName);
    }

    public WebTarget initClient(String serviceName) {
        return initClientInternal(serviceName)
                .register(new ClientLoggingFilter("Intapi JSON REST Client: " + serviceName));
    }

    private WebTarget initClientInternal(String serviceName) {
        ClientBuilder clientBuilder = ClientBuilder.newBuilder();
        disableSslVerification(clientBuilder);
        WebTarget result =  clientBuilder.build()
                .property(ApacheClientProperties.CONNECTION_MANAGER, new PoolingHttpClientConnectionManager())
                .property(ClientProperties.CONNECT_TIMEOUT, connectionTimeoutMillis)
                .property(ClientProperties.READ_TIMEOUT, connectionTimeoutMillis)
                .property(ClientProperties.READ_TIMEOUT, connectionTimeoutMillis)
                .property(HttpUrlConnectorProvider.SET_METHOD_WORKAROUND, true)
                .target(getConfig().getHost())
                .path(serviceName);
        if(tvmServiceTicketFilterClass == null) {
            result.register(TvmServiceTicketFilter.class);
        } else {
            result.register(tvmServiceTicketFilterClass);
        }
        return result;
    }

    private static class TvmServiceTicketFilter implements ClientRequestFilter {
        private ServiceTicketProvider serviceTicketProvider;

        public TvmServiceTicketFilter() {
            this.serviceTicketProvider = new ServiceTicketProviderProxy();
        }

        @Override
        public void filter(ClientRequestContext clientRequestContext) throws IOException {
            long tvmDstClientId = DirectTestRunProperties.getInstance().getDirectIntapiTvmClientId();
            MultivaluedMap<String, Object> headers = clientRequestContext.getHeaders();
            headers.putSingle(SERVICE_TICKET_HEADER, serviceTicketProvider.getServiceTicket(tvmDstClientId));
        }
    }
}
