package ru.yandex.autotests.directapi.darkside.clients;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.util.HashMap;
import java.util.Map;
import java.util.function.Supplier;

import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.type.TypeFactory;
import com.googlecode.jsonrpc4j.JsonRpcHttpClient;
import com.googlecode.jsonrpc4j.ReflectionUtil;
import org.apache.http.client.utils.URIBuilder;

import ru.yandex.autotests.direct.utils.clients.tvm.ServiceTicketProvider;
import ru.yandex.autotests.direct.utils.clients.tvm.ServiceTicketProviderProxy;
import ru.yandex.autotests.direct.utils.config.DirectTestRunProperties;
import ru.yandex.autotests.directapi.darkside.connection.DarksideConfig;
import ru.yandex.autotests.directapi.darkside.exceptions.DarkSideException;

import static ru.yandex.autotests.direct.utils.clients.tvm.ServiceTicketProvider.SERVICE_TICKET_HEADER;

/**
 * User: omaz
 * Date: 26.08.13
 * Json-rpc клиент
 */
public class IntapiJsonRpcClient extends BaseIntapiClient {
    private final ObjectMapper objectMapper;

    private ServiceTicketProvider serviceTicketProvider;

    private final Supplier<Long> tvmDstClientIdFactory;

    private static final Supplier<Long> directIntapiTvmClientIdProvider =
            () -> DirectTestRunProperties.getInstance().getDirectIntapiTvmClientId();

    public IntapiJsonRpcClient(DarksideConfig config, Supplier<Long> tvmDstClientIdFactory) {
        super(config, "JSON_RPC");
        this.objectMapper = new ObjectMapper()
                .configure(SerializationFeature.INDENT_OUTPUT, true)
                .setSerializationInclusion(JsonInclude.Include.NON_NULL);
        this.serviceTicketProvider = new ServiceTicketProviderProxy();
        this.tvmDstClientIdFactory = tvmDstClientIdFactory;
    }

    public IntapiJsonRpcClient(DarksideConfig config) {
        this(config, directIntapiTvmClientIdProvider);
    }

    private void disableSslVerification(JsonRpcHttpClient jsonRpcHttpClient) {
        try {
            // Create a trust manager that does not validate certificate chains
            TrustManager[] trustAllCerts = new TrustManager[]{new CertificatesUtils.IgnoreSSLTrustManager()};

            // Install the all-trusting trust manager
            SSLContext sc = SSLContext.getInstance("SSL");
            sc.init(null, trustAllCerts, new java.security.SecureRandom());
            jsonRpcHttpClient.setSslContext(sc);

            // Install the all-trusting host verifier
            jsonRpcHttpClient.setHostNameVerifier(new CertificatesUtils.AllowAllHostnameVerifier());
        } catch (NoSuchAlgorithmException | KeyManagementException e) {
            throw new DarkSideException("Не удалось отключить SslVerification для JsonRpcHttpClient", e);
        }
    }

    private JsonRpcHttpClient initClient(String serviceName) {
        JsonRpcHttpClient jsonRpcHttpClient;
        URIBuilder builder = new URIBuilder(getConfig().getHost());
        try {
            builder.setPath(builder.getPath() + "/" + serviceName);
            jsonRpcHttpClient = new JsonRpcHttpClient(objectMapper,
                    builder.build().toURL(),
                    new HashMap<String, String>());
            disableSslVerification(jsonRpcHttpClient);
            jsonRpcHttpClient.setReadTimeoutMillis(connectionTimeoutMillis);
            jsonRpcHttpClient.setConnectionTimeoutMillis(connectionTimeoutMillis);
        } catch (URISyntaxException | MalformedURLException e) {
            throw new DarkSideException("Ошибка в адресе Json-RPC клиента...", e);
        }
        return jsonRpcHttpClient;
    }

    public <T> T getServiceWithNamedParams(String serviceName, Class<T> tClass) {
        JsonRpcHttpClient jsonRpcHttpClient = initClient(serviceName);
        return createCustomClientProxy(tClass, jsonRpcHttpClient, true);
    }

    public <T> T getServiceWithOneComplexParam(String serviceName, Class<T> tClass) {
        JsonRpcHttpClient jsonRpcHttpClient = initClient(serviceName);
        return createCustomClientProxy(tClass, jsonRpcHttpClient, false);
    }

    public <T> T getServiceWithOneComplexParam(String serviceName, Class<T> tClass, boolean ignoreNullFields) {
        JsonRpcHttpClient jsonRpcHttpClient = initClient(serviceName);
        if (ignoreNullFields) {
            jsonRpcHttpClient.getObjectMapper().setSerializationInclusion(JsonInclude.Include.NON_NULL);
        }
        return createCustomClientProxy(tClass, jsonRpcHttpClient, false);
    }

    @SuppressWarnings("unchecked")
    // create and return the proxy
    private <T> T createCustomClientProxy(Class<T> proxyInterface,
                                          final JsonRpcHttpClient client,
                                          final boolean useNamedParams) {
        return (T) Proxy.newProxyInstance(getClass().getClassLoader(), new Class<?>[]{proxyInterface},
                new InvocationHandler() {
                    public Object invoke(Object proxy, Method method, Object[] args) throws Exception {
                        IntapiMessage message = new IntapiMessage();

                        long tvmDstClientId = tvmDstClientIdFactory.get();
                        //Получаем ответ в Object, логируем ответ а потом пытаемся распарсить.
                        Object response;
                        try {
                            message.setUrl(client.getServiceUrl().toString() + "?method=" + method.getName());
                            Object argument = useNamedParams ?
                                    ReflectionUtil.parseArguments(method, args, true) : args[0];
                            message.setRequest(objectMapper.writeValueAsString(argument));

                            Map<String, String> headers = new HashMap<>();
                            headers.put(SERVICE_TICKET_HEADER, serviceTicketProvider.getServiceTicket(tvmDstClientId));

                            response = client.invoke(method.getName(), argument, Object.class, headers);
                            message.setResponse(objectMapper.writeValueAsString(response));
                        } catch (Throwable e) {
                            message.setResponse(e.toString());
                            String errorText = String.format(
                                    "Ошибка при вызове метода %s" +
                                            "\nError: %s", method.getName(), e.getMessage()
                            );
                            throw new DarkSideException(errorText, e);
                        } finally {
                            attachIntapiMessage(message);
                        }

                        try {
                            JavaType returnJavaType =
                                    TypeFactory.defaultInstance().constructType(method.getGenericReturnType());
                            return objectMapper.convertValue(response, returnJavaType);
                        } catch (IllegalArgumentException e) {
                            String errorText =
                                    String.format("Не удалось распарсить ответ метода: %s" +
                                                    "\nОжидаемый тип ответа: %s" +
                                                    "\nResponse: %s",
                                            method.getName(), method.getReturnType(),
                                            objectMapper.writeValueAsString(response));
                            throw new DarkSideException(errorText, e);
                        }
                    }
                });
    }
}
