package ru.yandex.autotests.directapi.darkside.clients;

import ru.yandex.autotests.directapi.darkside.connection.DarksideConfig;
import ru.yandex.autotests.directapi.darkside.datacontainers.soap.BaseSoapRequest;
import ru.yandex.autotests.directapi.darkside.datacontainers.soap.BaseSoapResponse;
import ru.yandex.autotests.directapi.darkside.exceptions.DarkSideException;
import ru.yandex.autotests.irt.testutils.converter.XMLConverter;

import javax.xml.soap.*;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLStreamHandler;

/**
 * User: omaz, xy6er
 * Date: 21.08.13
 * Soap клиент
 */
public class IntapiSoapClient extends BaseIntapiClient {
    private SOAPConnection connection;
    private MessageFactory messageFactory;

    public IntapiSoapClient(DarksideConfig config) {
        super(config, "SOAP");
        try {
            CertificatesUtils.ignoreCertificates();
            SOAPConnectionFactory soapConnectionFactory = SOAPConnectionFactory.newInstance();
            this.connection = soapConnectionFactory.createConnection();
            this.messageFactory = MessageFactory.newInstance();
        } catch (SOAPException e) {
            throw new DarkSideException("Не удалось подключиться к SOAP серверу", e);
        }
    }

    private String soapMessageToString(SOAPMessage soapMessage) {
        try {
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            soapMessage.writeTo(out);
            return XMLConverter.prettyDocumentToString(new String(out.toByteArray()));
        } catch (IOException | SOAPException e) {
            throw new DarkSideException("Ошибка при выводе SOAP сообщения", e);
        }
    }

    public void sendRequest(BaseSoapRequest requestObject, BaseSoapResponse responseObject) {
        IntapiMessage message = new IntapiMessage();
        try {
            String url = getConfig().getHost().toString();
            message.setUrl(url);

            URL endpoint = new URL(null, url,
                    new URLStreamHandler() {
                        @Override
                        protected URLConnection openConnection(URL url) throws IOException {
                            URL target = new URL(url.toString());
                            URLConnection connection = target.openConnection();
                            // Connection settings
                            connection.setConnectTimeout(connectionTimeoutMillis);
                            connection.setReadTimeout(connectionTimeoutMillis);
                            return (connection);
                        }
                    });

            SOAPMessage request = messageFactory.createMessage();
            requestObject.generateSOAPMessage(request);
            message.setRequest(soapMessageToString(request));

            SOAPMessage response = connection.call(request, endpoint);
            message.setResponse(soapMessageToString(response));

            responseObject.unmarshallSOAPMessage(response);
        } catch (MalformedURLException e) {
            message.setResponse("Ошибка формирования запроса" + e.getMessage());
            throw new DarkSideException(message.getResponse(), e);
        } catch (SOAPException e) {
            message.setResponse("Ошибка при вызове SOAP-метода: " + e.getMessage());
            throw new DarkSideException(message.getResponse(), e);
        } catch (ClassCastException e) {
            message.setResponse(e.toString());
            throw new DarkSideException(
                    "Не удалось распарсить ответ SOAP-метода" +
                            "\nResponse: " + message.getResponse(), e
            );
        } finally {
            attachIntapiMessage(message);
        }
    }

    public void finalize() throws Throwable {
        super.finalize();
        try {
            connection.close();
        } catch (SOAPException e) {
            throw new DarkSideException("Не удалось закрыть соединение с SOAP сервером", e);
        }
    }

}
