package ru.yandex.autotests.directapi.darkside.steps;


import java.util.List;

import ru.yandex.autotests.direct.scriptrunner.service.clientdata.ClientDataParams;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.ScriptRunService;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.TestScriptRunResponse;
import ru.yandex.autotests.directapi.darkside.exceptions.DarkSideException;
import ru.yandex.autotests.directapi.darkside.model.QueryReportType;
import ru.yandex.autotests.directapi.darkside.model.ScriptParams;
import ru.yandex.autotests.directapi.darkside.model.Scripts;
import ru.yandex.autotests.directapi.darkside.steps.base.BaseJsonRpcSteps;
import ru.yandex.autotests.irt.testutils.beandiffer2.beanfield.BeanFieldPath;
import ru.yandex.autotests.irt.testutils.beandiffer2.comparestrategy.defaultcomparestrategy.DefaultCompareStrategies;
import ru.yandex.autotests.irt.testutils.json.JsonUtils;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.HazelcastAnnotations;

import static org.hamcrest.Matchers.startsWith;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.beandiffer2.BeanDifferMatcher.beanDiffer;
public class RunScriptSteps extends BaseJsonRpcSteps<ScriptRunService> {

    String SECRET = "b3uPkvVcVQGdGKpu";

    @Step("Запуск скрипта {0} с параметрами {1}")
    public String runScript(String scriptName, String cmdLine) {
        TestScriptRunResponse response = runScriptRaw(scriptName, cmdLine);
        if (!response.getCode().equals(0)) {
            throw new DarkSideException(
                    String.format(
                            "Скрипт %s завершился с ошибкой\n%s",
                            scriptName,
                            JsonUtils.toStringLow(response)));
        }
        return response.getStdout();
    }


    public String runScript(String scriptName) {
        return runScript(scriptName, "");
    }

    public String runScript(Scripts script) {
        return runScript(script.getScriptName());
    }

    public String runScript(Scripts script, ScriptParams params) {
        return runScript(script.getScriptName(), params.toString());
    }

    @Step("Запуск скрипта {0} с параметрами {1}")
    public TestScriptRunResponse runScriptRaw(String scriptName, String cmdLine) {
        return service().run_script(scriptName, cmdLine, SECRET);
    }

    public TestScriptRunResponse runScriptRaw(Scripts script, ScriptParams params) {
        return runScriptRaw(script.getScriptName(), params.toString());
    }

// metrica monitoring

    // Cкрипт получения данных от мониторинга доменов БК
    // https://st.yandex-team.ru/TESTIRT-9372

    @Step("Запуск скрипта bsCheckUrlAvailability.pl с эмуляцией отключения доменов: {0}")
    @HazelcastAnnotations.Lock("direct.bs_check_url_availability")
    public void runBsCheckUrlAvailabilityWithDeadDomains(Integer shard, String... domains) {
        runBsCheckUrlAvailabilityScript(new ScriptParams()
                .withShardId(shard)
                .withDebug()
                .withDeadDamains(domains)
        );
    }

    @Step("Запуск скрипта bsCheckUrlAvailability.pl с эмуляцией оживших доменов: {0}")
    @HazelcastAnnotations.Lock("direct.bs_check_url_availability")
    public void runBsCheckUrlAvailabilityWithAliveDomains(Integer shard, String... domains) {
        runBsCheckUrlAvailabilityScript(new ScriptParams()
                .withShardId(shard)
                .withDebug()
                .withAliveDamains(domains)
        );
    }

    @Step("Запуск скрипта bsCheckUrlAvailability.pl с эмуляцией оживших доменов {0} и умерших доменов {1}")
    @HazelcastAnnotations.Lock("direct.bs_check_url_availability")
    public void runBsCheckUrlAvailability(
            Integer shard, List<String> aliveDomains, List<String> deadDomains)
    {

        ScriptParams params = new ScriptParams().withDebug().withShardId(shard);
        if (aliveDomains != null) {
            params.withAliveDamains(aliveDomains.toArray(new String[]{}));
        }
        if (deadDomains != null) {
            params.withDeadDamains(
                    deadDomains.toArray(new String[]{}));
        }
        runBsCheckUrlAvailabilityScript(params);
    }

    public String runBsCheckUrlAvailabilityScript(ScriptParams scriptParams) {
        return runScript(Scripts.BS_CHECK_URL_AVAILABILITY, scriptParams);
    }

    public String runScriptWithDefaultParams(Scripts script) {
        return runScript(script, script.getDefaultParams());
    }

    @HazelcastAnnotations.Lock("direct.balanceGetClientNDSDiscountSchedule")
    public void runBalanceGetClientNDSDiscountSchedule(int shard, String login) {
        runScript(Scripts.BALANCE_GET_CLIENT_NDS_DISCOUNT_SHEDULE,
                new ScriptParams().withShardId(shard).withLogin(login));
    }

    public TestScriptRunResponse runBsExportMaster(ScriptParams params) {
        return this.service().run_script(Scripts.BS_EXPORT_MASTER.getScriptName(), params.toString(), SECRET);
    }

    @Deprecated
    public TestScriptRunResponse runBsClientData(ClientDataParams params) {
        return this.service().run_script(Scripts.BS_CLIENT_DATA.getScriptName(), params.toString(), SECRET);
    }

    public TestScriptRunResponse runBsClientData(ScriptParams params) {
        return this.service().run_script(Scripts.BS_CLIENT_DATA.getScriptName(), params.toString(), SECRET);
    }

    public void runBsExportMasterOnce() {
        runScript(Scripts.BS_EXPORT_MASTER,
                new ScriptParams()
                        .once());
    }

    public void runBsClientDataOnce() {
        runScript(Scripts.BS_CLIENT_DATA,
                new ScriptParams()
                        .once()
                        .withParId("std:1"));
    }


    public void runPpcRedirectCheck(int shardNumber) {
        runScript(Scripts.PPC_REDIRECT_CHECK,
                new ScriptParams()
                        .withShardId(shardNumber)

        );
    }

    /**
     * Хотя поле cid в скрипте опционально, здесь оно обязательное,
     * т.к. скрипт создает большую нагрузку, если кампания не специфицирована
     *
     * @param shard
     * @param cids
     */
    public void runPpcArchiveOldBanners(int shard, Long cids) {
        runScript(Scripts.PPC_ARCHIVE_OLD_BANNERS,
                new ScriptParams()
                        .withShardId(shard)
                        .withCids(cids));
    }

    public void runPpcCampAutoPrice() {
        runScript(Scripts.PPC_CAMP_AUTOPRICE,
                new ScriptParams()
                        .withParId("easy:1")
                        .withShardId(1)
        );
    }

    public void runPpcPushNotificationsQueue(final int shard) {
        new Thread(new Runnable() {
            @Override
            public void run() {
                log().info("Start ppcPushNotificationsQueue script in New Thread");
                try {
                    runScript(Scripts.PPC_PUSH_NOTIFICATIONS_QUEUE,
                            new ScriptParams()
                                    .withShardId(shard)
                    );
                } catch (DarkSideException e) {
                    if (e.getMessage().contains(": 140,")) {
                        log().info("Script ppcPushNotificationsQueue running in another thread\n");
                    } else if (e.getMessage().contains("TIMED OUT") || e.getMessage().contains("Read timed out")) {
                        log().info("Script ppcPushNotificationsQueue stopped: Read timed out\n");
                    } else {
                        throw e;
                    }
                }
            }
        }).start();
    }

    public void runAntispamCreateReport() {
        runScript(Scripts.ANTISPAM_CREATE_REPORT);
    }

    public void runPpcFetchAgencyCountryCurrency(String login) {
        runScript(
                Scripts.PPC_FETCH_AGENCY_COUNTRY_CURRENCY,
                new ScriptParams()
                        .withLogin(login)

        );
    }

    /**
     * Запуск скрипта обработки фоновой очереди архивации/разархивации/удаления/копирования кампаний
     * @param shard - шард
     * @param cid - идентификатор кампании
     * @param parId - какую очередь обрабатывать. Может принимать два значения:
     *              copy - копирования кампаний
     *              rest - архивации/разархивации/удаления
     */
    public void runPpcCampQueue(Integer shard, Integer cid, String parId) {
        ScriptParams scriptParams =
                new ScriptParams()
                        .once()
                        .withShardId(shard)
                        .withParId(parId)
                        .withCids(cid);
        runScript(
                Scripts.PPC_CAMP_QUEUE,
                scriptParams
        );
    }

    public void runFillCurrencyRates() {
        runScript(Scripts.FILL_CURRENCY_RATES);
    }

    public void runPpcCampAutoPrice(int shardID, int... cids) {
        runScript(
                Scripts.PPC_CAMP_AUTOPRICE,
                new ScriptParams()
                        .once()
                        .withShardId(shardID)
                        .withCids(cids));
    }

    public void runPpcForceCurrencyConvert(Integer shard, String login) {
        ScriptParams scriptParams =
                new ScriptParams()
                        .withShardId(shard)
                        .withLogin(login);
        runScript(
                Scripts.PPC_FORCE_CURRENCY_CONVERT,
                scriptParams
        );
    }

    public void runPpcForceCurrencyConvertNotify(Integer shard, String login) {
        ScriptParams scriptParams =
                new ScriptParams()
                        .withShardId(shard)
                        .withLogin(login);
        runScript(
                Scripts.PPC_FORCE_CURRENCY_CONVERT_NOTIFY,
                scriptParams
        );
    }

    public void runPpcPrepareDynamicAdgroupsForExport(int shardID, int... cids) {
        runScript(
                Scripts.PPC_PREPARE_DYNAMIC_ADGROUPS_FOR_EXPORT,
                new ScriptParams()
                        .withShardId(shardID)
                        .withCids(cids));
    }

    public void runPpcRetargetingCheckGoals(int shardID, String... clientIds) {
        runScript(
                Scripts.PPC_RETARGETING_CHECK_GOALS,
                new ScriptParams()
                        .withShardId(shardID)
                        .withClientIds(clientIds));
    }

    public void runPpcFetchClientMulticurrencyTeaserData(int shard, String login) {
        runScript(Scripts.PPC_FETCH_CLIENT_MULTICURRENCY_TEASER_DATA,
                new ScriptParams().withShardId(shard).withLogin(login));
    }

    public void runPpcUpdateStoreContent(String clientId, String url, Integer shard) {
        runScript(Scripts.PPC_UPDATE_STORE_CONTENT,
                new ScriptParams().withClientIds(clientId).withUrl(url).withShardId(shard));
    }

    public void runPpcSearchQueryStatus(int reportID, QueryReportType queryReportType) {
        runScript(Scripts.PPC_SEARCH_QUERY_STATUS, new ScriptParams().withReportId(reportID)
                .withReportType(queryReportType.value()));
    }

    public void runPpcFeedToBannerLand(int shardId, String clientId) {
        runScript(
                Scripts.PPC_FEED_TO_BANNER_LAND,
                new ScriptParams()
                        .withShardId(shardId)
                        .withClientIds(clientId)
                        .once());
    }

    public void runPpcFeedToBannerLand(int shardId, Long[] feedIds, String clientId, int maxFeedSize ) {
        runScript(
                Scripts.PPC_FEED_TO_BANNER_LAND,
                new ScriptParams()
                        .withShardId(shardId)
                        .withFeedIds(feedIds)
                        .withClientIds(clientId)
                        .withMaxFeedSize(maxFeedSize)
                        .once());
    }

    public void runPpcFeedToBannerLand(int shardId, Long[] feedIds, String[] clientIds) {
        runPpcFeedToBannerLand(shardId, feedIds, clientIds, null, true);
    }

    public void runPpcFeedToBannerLand(int shardId, Long[] feedIds, String[] clientIds, String bannerLandUrl) {
        runPpcFeedToBannerLand(shardId, feedIds, clientIds, bannerLandUrl, true);
    }

    public void runPpcFeedToBannerLand(int shardId, Long[] feedIds, String[] clientIds, String bannerLandUrl
            , boolean once)
    {
        ScriptParams scriptParams = new ScriptParams()
                .withShardId(shardId)
                .withFeedIds(feedIds)
                .withClientIds(clientIds)
                .withBannerLandUrl(bannerLandUrl);
        if (once) {
            scriptParams.once();
        }
        runScript(Scripts.PPC_FEED_TO_BANNER_LAND, scriptParams);
    }

    public void runPpcFeedsRejectedOffers(String url) {
        runScript(
                Scripts.PPC_FEEDS_REJECTED_OFFERS,
                new ScriptParams()
                        .withUrl(url));
    }

    public void runPpcFeedsRejectedOffers() {
        runScript(
                Scripts.PPC_FEEDS_REJECTED_OFFERS,
                new ScriptParams());
    }

    public void runPpcCopySubclientsForceCurrency(String login) {
        ScriptParams scriptParams =
                new ScriptParams()
                        .withLogin(login);
        runScript(
                Scripts.PPC_COPY_SUBCLIENTS_FORCE_CURRENCY,
                scriptParams
        );
    }

    @Step("Запуск скрипта update_perf_counters.pl")
    @HazelcastAnnotations.Lock("direct.update_perf_counters")
    public void runUpdatePerfCounters(Integer shardId, int... cids) {
        runScript(Scripts.UPDATE_PERF_COUNTERS,
                new ScriptParams()
                        .withShardId(shardId)
                        .withCids(cids));
    }

    @Step("Запуск скрипта ppcManageExperiments.pl")
    @HazelcastAnnotations.Lock("direct.ppcManageExperiments")
    public void runPpcManageExperiments(Integer shardId, Long experimentId) {
        runScript(Scripts.PPC_MANAGE_EXPERIMENTS,
                new ScriptParams()
                        .withShardId(shardId)
                        .withExperimentId(experimentId));
    }

    @HazelcastAnnotations.Lock("direct.ppcProcessAutoPayments")
    public void runPpcProcessAutoPayments(Integer shard, Long walletId, boolean waitNotification) {
        ScriptParams scriptParams =
                new ScriptParams()
                        .withShardId(shard)
                        .withWalletCid(walletId)
                        .withForceCreateCampBalance();
        if (!waitNotification) {
            scriptParams.withDontWaitNotification();
        }
        runScript(
                Scripts.PPC_PROCESS_AUTO_PAYMENTS,
                scriptParams
        );
    }

    public void runPpcProcessAutoPayments(Integer shard, Long walletId) {
        runPpcProcessAutoPayments(shard, walletId, false);
    }

    @Step("Запуск скрипта ppcProcessImageQueue.pl")
    public void runProcessImageQueue(ScriptParams scriptParams) {
        runScript(Scripts.PPC_PROCESS_IMAGE_QUEUE, scriptParams);
    }

    public void runProcessImageQueue(Integer shardId, String clientId, Integer beta) {
        runProcessImageQueue(new ScriptParams()
                .withUniq(beta)
                .once()
                .withShardId(shardId)
                .withClientIds(clientId));
    }

    @Step("Запуск скрипта ppcSetAutoResources.pl")
    public void runSetAutoResources(ScriptParams scriptParams) {
        runScript(Scripts.PPC_SET_AUTO_RESOURCES, scriptParams);
    }

    public void runSetAutoResources(Integer shardId, String clientId) {
        runSetAutoResources(new ScriptParams()
                .withShardId(shardId)
                .withClientIds(clientId));
    }

    @Step("Запуск скрипта ppcClearOldUnpaidCampaigns.pl")
    public void runPpcClearOldUnpaidCampaigns(Integer shardId, String login) {
        runScript(Scripts.PPC_CLEAR_OLD_UNPAID_CAMPAIGNS,
                new ScriptParams()
                        .withShardId(shardId)
                        .withLogin(login)
        );
    }

    @Step("Запуск скрипта ppcCampGetGoals.pl, date_from = {0}, date_to = {1}, bs-url = {2}")
    @HazelcastAnnotations.Lock("direct.ppcCampGetGoals.pl")
    public void runPpcCampGetGoals(String dateFrom, String dateTo, String fakeBsUrl) {
        runScript(Scripts.PPC_CAMP_GET_GOALS,
                new ScriptParams()
                        .force()
                        .skipExists()
                        .withDateFrom(dateFrom)
                        .withDateTo(dateTo)
                        .withFakeBSUrl(fakeBsUrl)
        );
    }

    @Step("Запуск скрипта ppcCampGetGoals.pl, date_from = {0}, date_to = {1}, bs-url = {2}, lines-chunk-size = {3}, "
            + "с заданным размером чанка")
    @HazelcastAnnotations.Lock("direct.ppcCampGetGoals.pl")
    public void runPpcCampGetGoals(String dateFrom, String dateTo, String fakeBsUrl, Long chunkSize) {
        runScript(Scripts.PPC_CAMP_GET_GOALS,
                new ScriptParams()
                        .force()
                        .skipExists()
                        .withDateFrom(dateFrom)
                        .withDateTo(dateTo)
                        .withFakeBSUrl(fakeBsUrl)
                        .withChunkSize(chunkSize)
        );
    }

    @Step("Запуск скрипта ppcCampGetGoals.pl, date_from = {0}, date_to = {1}, bs-url = {2}, ожидаем "
            + "ошибку с кодом \"{3}\" и начальным текстом stderr \"{4}\"")
    @HazelcastAnnotations.Lock("direct.ppcCampGetGoals.pl")
    public void runPpcCampGetGoalsExpectError(String dateFrom, String dateTo, String fakeBsUrl, int errorCode,
            String stderr)
    {
        TestScriptRunResponse response = runScriptRaw(Scripts.PPC_CAMP_GET_GOALS,
                new ScriptParams()
                        .force()
                        .skipExists()
                        .withDateFrom(dateFrom)
                        .withDateTo(dateTo)
                        .withFakeBSUrl(fakeBsUrl)
        );
        TestScriptRunResponse expected = new TestScriptRunResponse(errorCode, null, stderr);
        assertThat("получили ожидаемую ошибку при запуске скрипта", response,
                beanDiffer(expected).useCompareStrategy(DefaultCompareStrategies.onlyExpectedFields().forFields(
                        BeanFieldPath.newPath("stderr")).useMatcher(startsWith(stderr))));
    }

    @Step("Запуск скрипта ppcSendMailMaster.pl, cid = {0}, shard = {1}")
    @HazelcastAnnotations.Lock("direct.ppcSendMailMaster.pl")
    public void runPpcSendMailMaster(Long cid, Integer shard) {
        runScript("ppcSendMailMaster.pl", "--cid " + cid.toString()
                + " --shard " + shard.toString() + " --once");
    }

    @Step("Запуск скрипта runPpcSendMailNew.pl, cid = {0}, shard = {1}")
    @HazelcastAnnotations.Lock("direct.ppcSendMailNew.pl")
    public void runPpcSendMailNew(Long cid, Integer shard) {
        runScript("ppcSendMailNew.pl", "--cid " + cid.toString()
                + " --shard " + shard.toString() + " --once");
    }

    @Override
    protected String getServiceName() {
        return ScriptRunService.SERVICE_NAME;
    }

    @Override
    protected Class<ScriptRunService> getClazz() {
        return ScriptRunService.class;
    }
}
