package ru.yandex.autotests.directapi.darkside.steps;

import java.util.Objects;
import java.util.function.Supplier;

import ru.yandex.autotests.direct.utils.config.DirectTestRunProperties;
import ru.yandex.autotests.directapi.darkside.clients.IntapiJsonRpcClient;
import ru.yandex.autotests.directapi.darkside.connection.DarksideConfig;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.sandbox.StatusResponse;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.sandbox.SandboxService;
import ru.yandex.autotests.directapi.darkside.exceptions.DarkSideException;
import ru.yandex.autotests.direct.utils.clients.blackbox.PassportUserInfo;
import ru.yandex.autotests.directapi.darkside.model.SandboxSettings;
import ru.yandex.autotests.directapi.darkside.steps.base.BaseJsonRpcSteps;
import ru.yandex.qatools.allure.annotations.Step;

/**
 * Степы для взаимодействия с Sandbox-ом
 *
 * @author egorovmv
 */
public final class SandboxServiceSteps extends BaseJsonRpcSteps<SandboxService> {

    private SandboxService service;
    private static final Supplier<Long> directSandboxIntapiTvmClientIdProvider =
            () -> DirectTestRunProperties.getInstance().getDirectSandboxIntapiTvmClientId();

    /**
     * Создать sandbox для пользователя
     *
     * @return Master-token для песочницы
     */
    @Override
    public void init(DarksideConfig context) {
        super.init(context);
        setClient(new IntapiJsonRpcClient(context, directSandboxIntapiTvmClientIdProvider));
        service = initService();
    }

    @Override
    protected SandboxService service() {
        return service;
    }

    @Step("Создание sandbox-а: user = {0}, settings = {1}")
    public String create(PassportUserInfo user, SandboxSettings settings) {
        Objects.requireNonNull(user, "user");
        Objects.requireNonNull(settings, "settings");

        StatusResponse createStatus = service().create_user(
                user.getLogin(),
                user.getUid(),
                user.getFullName(),
                user.getEmail(),
                settings.getClientType(),
                settings.getCurrency(),
                settings.isInitTestData() ? 1 : 0,
                settings.isEnableSharedAccount() ? 1 : 0);
        if (createStatus.isFail()) {
            throw new DarkSideException(createStatus.getDescription());
        }

        return service().get_current_state(user.getUid()).getMasterToken();
    }

    /**
     * Удалить sandbox для пользователя
     *
     * @param uid Yandex uid пользователя
     */
    @Step("Удаление sandbox-а: uid = {0}")
    public void drop(String uid) {
        Objects.requireNonNull(uid, "uid");
        service().drop_user(uid);
    }

    @Override
    protected String getServiceName() {
        return SandboxService.SERVICE_NAME;
    }

    @Override
    protected Class<SandboxService> getClazz() {
        return SandboxService.class;
    }
}
